/* 
   GCArray.m

   Copyright (C) 1995, 1996, 1997 Ovidiu Predescu and Mircea Oancea.
   All rights reserved.

   Author: Ovidiu Predescu <ovidiu@bx.logicnet.ro>

   This file is part of the Foundation Extensions Library.

   Permission to use, copy, modify, and distribute this software and its
   documentation for any purpose and without fee is hereby granted, provided
   that the above copyright notice appear in all copies and that both that
   copyright notice and this permission notice appear in supporting
   documentation.

   We disclaim all warranties with regard to this software, including all
   implied warranties of merchantability and fitness, in no event shall
   we be liable for any special, indirect or consequential damages or any
   damages whatsoever resulting from loss of use, data or profits, whether in
   an action of contract, negligence or other tortious action, arising out of
   or in connection with the use or performance of this software.
*/

#import <Foundation/NSString.h>

#include <extensions/config.h>

#include <extensions/common.h>
#include <extensions/objc-runtime.h>
#include <extensions/GCArray.h>
#include <extensions/GCObject.h>
#include <extensions/GarbageCollector.h>
#include <extensions/NSException.h>
#include <extensions/exceptions/GeneralExceptions.h>

@implementation GCArray

+ (void)initialize
{
    static BOOL initialized = NO;

    if(!initialized) {
	initialized = YES;
	class_add_behavior(self, [GCObject class]);
    }
}

- (id)initWithObjects:(id*)objects count:(unsigned int)count
{
    int i;

    items = Calloc(count, sizeof(id));
    isGarbageCollectable = Calloc(count, sizeof(BOOL));
    itemsCount = count;
    for (i=0; i < count; i++)
	if (!(items[i] = [objects[i] retain]))
	    THROW([[InvalidArgumentException alloc] 
		    initWithReason:@"Nil object to be added in array"]);
	else isGarbageCollectable[i] = [objects[i] isGarbageCollectable];
    return self;
}

- (id)initWithArray:(NSArray*)anotherArray
{
    int i, count = [anotherArray count];

    items = Calloc(count, sizeof(id));
    isGarbageCollectable = Calloc(count, sizeof(BOOL));
    itemsCount = count;
    for (i=0; i < itemsCount; i++) {
	items[i] = [[anotherArray objectAtIndex:i] retain];
	isGarbageCollectable[i] = [items[i] isGarbageCollectable];
    }
    return self;
}

- (void)dealloc
{
    unsigned int index;

    if([GarbageCollector isGarbageCollecting]) {
	for (index = 0; index < itemsCount; index++)
	    if(!isGarbageCollectable[index])
		[items[index] release];
    }
    else {
	for (index = 0; index < itemsCount; index++)
	    [items[index] release];
    }

    Free(items);
    Free(isGarbageCollectable);
    [super dealloc];
}

- (id)copyWithZone:(NSZone*)zone
{
    if (NSShouldRetainWithZone(self, zone))
	return [self retain];
    return [[GCArray allocWithZone:zone] initWithArray:self copyItems:YES];
}

- (id)mutableCopyWithZone:(NSZone*)zone
{
    return [[GCMutableArray allocWithZone:zone]
		initWithArray:self copyItems:YES];
}

- (id)objectAtIndex:(unsigned int)index
{
    if (index >= itemsCount)
	THROW(([[RangeException alloc] 
		initWithReason:@"objectAtIndex: in NSArray" 
		size:itemsCount index:index]));
    return items[index];
}

- (unsigned int)count
{
    return itemsCount;
}

- (unsigned int)indexOfObjectIdenticalTo:(id)anObject
{
    int i;
    for (i = 0; i < itemsCount; i++)
	if (items[i] == anObject)
		return i;
    return NSNotFound;
}

- (void)gcDecrementRefCountOfContainedObjects
{
    int i, count;

    for (i = 0, count = [self count]; i < count; i++)
	if (isGarbageCollectable[i])
	    [[self objectAtIndex:i] gcDecrementRefCount];
}

- (BOOL)gcIncrementRefCountOfContainedObjects
{
    int i, count;

    if ([(id)self gcAlreadyVisited])
	return NO;
    [(id)self gcSetVisited:YES];

    for (i = 0, count = [self count]; i < count; i++)
	if(isGarbageCollectable[i]) {
	    id object = [self objectAtIndex:i];
	    [object gcIncrementRefCount];
	    [object gcIncrementRefCountOfContainedObjects];
	}
    return YES;
}

- (Class)classForCoder
{
    return [GCArray class];
}

@end /* GCArray */


@implementation GCMutableArray

+ (void)initialize
{
    static BOOL initialized = NO;
    if(!initialized) {
	initialized = YES;
	class_add_behavior(self, [GCArray class]);
    }
}

- (id)init
{
    return [self initWithCapacity:1];
}

- (id)initWithCapacity:(unsigned int)aNumItems
{
    items = Calloc(aNumItems, sizeof(id));
    isGarbageCollectable = Calloc(aNumItems, sizeof(BOOL));
    maxItems = aNumItems;
    itemsCount = 0;
    return self;
}

- (id)initWithObjects:(id *)objects count:(unsigned int)count
{
    int i;

    items = Calloc(count, sizeof(id));
    isGarbageCollectable = Calloc(count, sizeof(BOOL));
    maxItems = itemsCount = count;

    for (i=0; i < count; i++)
	if (!(items[i] = [objects[i] retain]))
	    THROW([[InvalidArgumentException alloc] 
		    initWithReason:@"Nil object to be added in array"]);
	else isGarbageCollectable[i] = [objects[i] isGarbageCollectable];
    return self;
}

- (id)initWithArray:(NSArray*)anotherArray
{
    int i, count = [anotherArray count];

    items = Calloc(count, sizeof(id));
    isGarbageCollectable = Calloc(count, sizeof(BOOL));
    maxItems = itemsCount = count;
    for (i=0; i < itemsCount; i++) {
	items[i] = [[anotherArray objectAtIndex:i] retain];
	isGarbageCollectable[i] = [items[i] isGarbageCollectable];
    }
    return self;
}

- (id)copyWithZone:(NSZone*)zone
{
    return [[GCArray allocWithZone:zone] 
		initWithArray:self copyItems:YES];
}

- (id)mutableCopyWithZone:(NSZone*)zone
{
    return [[GCMutableArray allocWithZone:zone] 
		initWithArray:self copyItems:YES];
}

- (void)insertObject:(id)anObject atIndex:(unsigned int)index
{
    unsigned int i;
    if (!anObject)
	THROW([[InvalidArgumentException alloc] 
		initWithReason:@"Nil object to be added in array"]);
    if (index > itemsCount)
	THROW([[RangeException alloc] 
		initWithReason:@"insertObject:atIndex: in GCMutableArray" 
		size:itemsCount index:index]);
    if (itemsCount == maxItems) {
	if (maxItems) {
	    maxItems += (maxItems >> 1) ? (maxItems >>1) : 1;
	}
	else {
	    maxItems = 1;
	}
	items = (id*)Realloc(items, sizeof(id) * maxItems);
	isGarbageCollectable = (BOOL*)Realloc(isGarbageCollectable,
					      sizeof(BOOL) * maxItems);
    }
    for(i = itemsCount; i > index; i--) {
	items[i] = items[i - 1];
	isGarbageCollectable[i] = isGarbageCollectable[i - 1];
    }
    items[index] = [anObject retain];
    isGarbageCollectable[index] = [anObject isGarbageCollectable];
    itemsCount++;
}

- (void)addObject:(id)anObject
{
    [self insertObject:anObject atIndex:itemsCount];
}

- (void)replaceObjectAtIndex:(unsigned int)index  withObject:(id)anObject
{
    if (!anObject)
	THROW([[InvalidArgumentException alloc] 
		initWithReason:@"Nil object to be added in array"]);
    if (index >= itemsCount)
	THROW([[RangeException alloc] 
		initWithReason:@"GCMutableArray replaceObjectAtIndex" 
		size:itemsCount index:index]);
    [anObject retain];
    [items[index] release];
    items[index] = anObject;
    isGarbageCollectable[index] = [anObject isGarbageCollectable];
}

- (void)removeObjectsFrom:(unsigned int)index
	count:(unsigned int)count
{
    int i;
    if (index + count > itemsCount)
	THROW([[RangeException alloc]
		initWithReason:@"removeObjectsFrom:count: in GCMutableArray"
		size:itemsCount index:index]);
    if (!count)
	return;
    for (i = index; i < index + count; i++)
	[items[i] release];

    for (i = index + count; i < itemsCount; i++, index++) {
	items[index] = items[i];
	isGarbageCollectable[index] = isGarbageCollectable[i];
    }
    for (; index < itemsCount; index++)
	items[index] = (id)0x3;

    itemsCount -= count;
}

- (void)removeAllObjects
{
    [self removeObjectsFrom:0 count:itemsCount];
}

- (void)removeLastObject
{
    if (itemsCount)
	[self removeObjectsFrom:(itemsCount - 1) count:1];
}

- (void)removeObjectAtIndex:(unsigned int)index
{
    [self removeObjectsFrom:index count:1];
}

- (Class)classForCoder
{
    return [GCMutableArray class];
}

@end /* GCMutableArray */
