/********************************************************************************
*                                                                               *
*                            V i s u a l   C l a s s                            *
*                                                                               *
*********************************************************************************
* Copyright (C) 1999 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXGLVisual.cpp,v 1.3 1999/10/01 21:23:44 jeroen Exp $                   *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "fxkeys.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXObjectList.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXVisual.h"
#include "FXGLVisual.h"


/*
  Notes:
  
  - FXGLVisual builds a visual/pixelformat suitable for GL drawing.
  
  - Selection of visual/pixelformat is based on a best match to a 
    given set of hints, according to some heuristics:
      
    1) Prefer color depth close to asked ones; really like to
       get a bit MORE color, rather than LESS, however.
      
    2) If we wanted Z-buffer, it is STRONGLY preferred; If there's 
       a choice, we prefer more color depth over more Z-depth; if 
       we already have more colors than requested, we prefer to meet 
       requested Z depth.
      
    3) If we wanted double buffer, we strongly prefer it over color and
       Z depth, but HAVING a Z-buffer is still more important.
       
    4) If we wanted alpha buffer, it is preferred, but Z-buffering
       and double buffering are considered more important.
       If there's a choice, we prefer to receive a few MORE bits of
       alpha buffer than we asked for, rather than LESS.
       
    5) If we wanted stereo, we prefer it, but almost everything except
       the color-, alpha-, and Z-depths are more important.
       
  - Some further tuning may be desired, but I think this should satisfy
    most cases....
    
  - Note that as long as OpenGL is in any way supported, you should ALWAYS
    be able to get at least some visual/pixelformat you can draw on.
    
  - As far as hardware acceleration goes, H/W acceleration should be
    enabled, possibly at the expense of color-, alpha-, and Z-depth;
    but NEVER at the expense of presence or absence of a requested feature.
    We only drop FEATURES which are requested if there is neither hardware
    nor software support.

    For example, we may trade in some Z-depth, but not the entire Z-buffer,
    to get a hardware accelerated visual/pixelformat.
       
  - SGI Infinite Reality may have up to 12 bits for red, green, blue each!
*/


   
/*******************************************************************************/


// Object implementation
FXIMPLEMENT(FXGLVisual,FXVisual,NULL,0)



// Deserialization
FXGLVisual::FXGLVisual(){
  }


// Construct 
FXGLVisual::FXGLVisual(FXApp* a,FXuint flgs):FXVisual(a,flgs){
  FXTRACE((100,"FXGLVisual::FXGLVisual %08x\n",this));
  redSize=8;
  greenSize=8;
  blueSize=8;
  alphaSize=0;
  depthSize=16;
  stencilSize=0;
  accumRedSize=0;
  accumGreenSize=0;
  accumBlueSize=0;
  accumAlphaSize=0;
  }


/*******************************************************************************/

#ifdef FX_NATIVE_WIN32

static FXuchar defSysClr[20][3] = {                         // System colors to match against
    { 0,   0,   0 },
    { 0x80,0,   0 },
    { 0,   0x80,0 },
    { 0x80,0x80,0 },
    { 0,   0,   0x80 },

    { 0x80,0,   0x80 },
    { 0,   0x80,0x80 },
    { 0xC0,0xC0,0xC0 },

    { 192, 220, 192 },
    { 166, 202, 240 },
    { 255, 251, 240 },
    { 160, 160, 164 },

    { 0x80,0x80,0x80 },
    { 0xFF,0,   0 },
    { 0,   0xFF,0 },
    { 0xFF,0xFF,0 },
    { 0,   0,   0xFF },
    { 0xFF,0,   0xFF },
    { 0,   0xFF,0xFF },
    { 0xFF,0xFF,0xFF }
    };
    

// Make palette
void FXGLVisual::makeOpenGLPalette(){
  unsigned char index[256],closest;
  int mindif,dif,dr,dg,db,n,i,j;
  int rmax,gmax,bmax;
  LOGPALETTE *pal;

  n = 1 << info->cColorBits;
  FXMALLOC(&pal,char,sizeof(LOGPALETTE)+sizeof(PALETTEENTRY)*n);
  pal->palVersion = 0x300;
  pal->palNumEntries = n;

  rmax=(1<<info->cRedBits)-1;
  gmax=(1<<info->cGreenBits)-1;
  bmax=(1<<info->cBlueBits)-1;
  
  // Build palette
  for(i=0; i<n; i++) {
    pal->palPalEntry[i].peRed = ((rmax*i+127)/255) << info->cRedShift;
    pal->palPalEntry[i].peGreen = ((gmax*i+127)/255) << info->cGreenShift;
    pal->palPalEntry[i].peBlue = ((bmax*i+127)/255) << info->cBlueShift;  
    pal->palPalEntry[i].peFlags = PC_NOCOLLAPSE;
    }

  // Find best mapping of default colors to new map
  if(n==256){
    for(i=0; i<256; i++) index[i]=255;
    for(j=19; j>=0; j--){
      mindif = 3*256*256;
      for(i=0; i<256; i++){
        if(index[i]==255){
          dr = (int)pal->palPalEntry[i].peRed - defSysClr[j][0];
          dg = (int)pal->palPalEntry[i].peGreen - defSysClr[j][1];
          db = (int)pal->palPalEntry[i].peBlue - defSysClr[j][2];
          dif = dr*dr+dg*dg+db*db;
          if(dif<mindif){
            closest = i;
            mindif = dif;
            }
          }
        }
      FXTRACE((100,"system color %d mapped to %d: dif = %f\n",j,closest,sqrt((double)mindif)));
      FXASSERT(index[closest]==255);
      index[closest] = j;
      }

    // Change proposed map to include already existing colors
    for(i=0; i<256; i++){
      j = index[i];
      if(j<20){
        pal->palPalEntry[i].peRed = defSysClr[j][0];
        pal->palPalEntry[i].peGreen = defSysClr[j][1];
        pal->palPalEntry[i].peBlue = defSysClr[j][2];
        pal->palPalEntry[i].peFlags = 0;
        }
      }
    }
  hPalette=CreatePalette(pal);
  FXFREE(&pal);
  }  


#endif



/*******************************************************************************/


// Initialize
void FXGLVisual::init(){
  if(!xid){
    FXTRACE((100,"FXGLVisual::init\n"));
    
#ifdef HAVE_OPENGL
#ifndef FX_NATIVE_WIN32
    
    XVisualInfo vitemplate;
    XVisualInfo *vi;
    int gdblbuf,glstereo,gldepth;      
    int glred,glgreen,glblue,glalpha,glstencil;
    int glaccred,glaccgreen,glaccblue,glaccalpha;
    int glrgba,glsupport,gllevel;
    int bestvis,dmatch,bestmatch;
    int dred,dgreen,dblue,ddepth,dalpha,dstencil;
    int daccred,daccgreen,daccblue,daccalpha;
    int nvi,i;
    
    // OpenGL is available if we're talking to an OpenGL-capable X-Server
    if(!glXQueryExtension(getApp()->display,NULL,NULL)){ 
      fxerror("%s::init: requested OpenGL extension not available.\n",getClassName()); 
      }
    
    // Scan for all visuals of given screen
    vitemplate.screen=DefaultScreen(getApp()->display);
    vi=XGetVisualInfo(getApp()->display,VisualScreenMask,&vitemplate,&nvi);
    if(!vi){ fxerror("%s::init: unable to obtain any visuals.\n",getClassName()); }
  
    // Try to find the best
    bestvis=-1;
    bestmatch=1000000000;
    
    for(i=0; i<nvi; i++){
      
      // GL RGBA Support is requested
      glXGetConfig(getApp()->display,&vi[i],GLX_USE_GL,&glsupport);
      glXGetConfig(getApp()->display,&vi[i],GLX_RGBA,&glrgba);
      glXGetConfig(getApp()->display,&vi[i],GLX_LEVEL,&gllevel);
      
      // Without GL support, we're toast
      if(glsupport && glrgba && gllevel==0){
        
        // Must have Single/Double buffer as requested
        glXGetConfig(getApp()->display,&vi[i],GLX_DOUBLEBUFFER,&gdblbuf);
          
        // Stereo support as requested
        glXGetConfig(getApp()->display,&vi[i],GLX_STEREO,&glstereo);
            
        // Get planes
        glXGetConfig(getApp()->display,&vi[i],GLX_RED_SIZE,&glred);
        glXGetConfig(getApp()->display,&vi[i],GLX_GREEN_SIZE,&glgreen);
        glXGetConfig(getApp()->display,&vi[i],GLX_BLUE_SIZE,&glblue);
        glXGetConfig(getApp()->display,&vi[i],GLX_ALPHA_SIZE,&glalpha);
        glXGetConfig(getApp()->display,&vi[i],GLX_DEPTH_SIZE,&gldepth);
        glXGetConfig(getApp()->display,&vi[i],GLX_STENCIL_SIZE,&glstencil);
        glXGetConfig(getApp()->display,&vi[i],GLX_ACCUM_RED_SIZE,&glaccred);
        glXGetConfig(getApp()->display,&vi[i],GLX_ACCUM_GREEN_SIZE,&glaccgreen);
        glXGetConfig(getApp()->display,&vi[i],GLX_ACCUM_BLUE_SIZE,&glaccblue);
        glXGetConfig(getApp()->display,&vi[i],GLX_ACCUM_ALPHA_SIZE,&glaccalpha);
 
        // We prefer to get a few MORE bits in RGBA than we asked for
        dred   = glred-redSize;     if(dred<0)   dred   *= -100;
        dgreen = glgreen-greenSize; if(dgreen<0) dgreen *= -100;
        dblue  = glblue-blueSize;   if(dblue<0)  dblue  *= -100;
        dalpha = glalpha-alphaSize; if(dalpha<0) dalpha *= -100;
        
        // Prefer better Z than asked, but colors more important
        ddepth = gldepth-depthSize; if(ddepth<0) ddepth *= -10;
 
        // We care about colors and Z depth more than stencil depth
        dstencil = glstencil-stencilSize; if(dstencil<0) dstencil *= -1;
        
        // Accumulation buffers
        daccred=glaccred-accumRedSize;       if(daccred<0)   daccred   *= -1;
        daccgreen=glaccgreen-accumGreenSize; if(daccgreen<0) daccgreen *= -1;
        daccblue=glaccblue-accumBlueSize;    if(daccblue<0)  daccblue  *= -1;
        daccalpha=glaccalpha-accumAlphaSize; if(daccalpha<0) daccalpha *= -1;

        // Want the best colors, of course
        dmatch=dred+dgreen+dblue+dalpha;

        // Accumulation buffers
        dmatch+=daccred+daccgreen+daccblue+daccalpha;

        // Want alpha
        if(alphaSize>0){
          if(glalpha<1) dmatch+=100000;
          else dmatch+=dalpha;
          }
        else{
          if(glalpha>0) dmatch+=100000;
          }
 
        // Wanted Z-buffer
        if(depthSize>0){
          if(gldepth<1) dmatch+=10000000;
          else dmatch+=ddepth;
          }
        else{
          if(gldepth>0) dmatch+=10000000;
          }
        
        // Double buffering also quite strongly preferred
        if(flags&VISUAL_DOUBLEBUFFER){
          if(!gdblbuf) dmatch+=1000000;
          }
        else{
          if(gdblbuf) dmatch+=1000000;
          }
        
        // Stencil buffers desired
        if(stencilSize>0){
          if(glstencil<1) dmatch+=10000;
          else dmatch+=dstencil;
          }
        else{
          if(glstencil>0) dmatch+=10000;
          }
        
        // Stereo not so important
        if(flags&VISUAL_STEREO){
          if(!glstereo) dmatch+=10000;
          }
        else{
          if(glstereo) dmatch+=10000;
          }
        
        // Trace
        FXTRACE((100,"Visual 0x%02x (%d) match value = %d\n",vi[i].visualid,i,dmatch));
        FXTRACE((100,"  red size   = %d\n",glred));
        FXTRACE((100,"  green size = %d\n",glgreen));
        FXTRACE((100,"  blue size  = %d\n",glblue));
        FXTRACE((100,"  alpha size = %d\n",glalpha));
        FXTRACE((100,"  depth size = %d\n",gldepth));
        FXTRACE((100,"  double buf = %d\n",gdblbuf));
        FXTRACE((100,"  stencil    = %d\n",glstencil));
        FXTRACE((100,"  acc red    = %d\n",glaccred));
        FXTRACE((100,"  acc green  = %d\n",glaccgreen));
        FXTRACE((100,"  acc blue   = %d\n",glaccblue));
        FXTRACE((100,"  acc alpha  = %d\n",glaccalpha));
        FXTRACE((100,"  stereo     = %d\n",glstereo));
            
        // May the best visual win
        if(dmatch<=bestmatch){
          
          // All other things being equal, we prefer default visual!
          if(dmatch<bestmatch || vi[i].visual==DefaultVisual(getApp()->display,DefaultScreen(getApp()->display))){
            bestmatch=dmatch;
            bestvis=i;
            }
          }
        }
      }
    
    // Shit out of luck
    if(bestvis<0){ fxerror("%s::init: requested OpenGL visual unavailable.\n",getClassName()); }
  
    // Report best visual
    FXTRACE((100,"Best Visual 0x%02x (%d) match value = %d\n",vi[bestvis].visualid,bestvis,bestmatch));
  
    // Get visual, depth
    visual=vi[bestvis].visual;
    depth=vi[bestvis].depth;

    // Keep into for later
    FXMALLOC(&info,XVisualInfo,1);
    memcpy(info,&vi[bestvis],sizeof(XVisualInfo));
    
    // Free stuff
    XFree((char*)vi);
    
    // We absolutely should have a visual now
    FXASSERT(visual);
    
    // Initialize colormap
    setupcolormap();
    
    // Make GC for this visual
    gc=makegc();    
    
    // What we matched
    FXTRACE((100,"OPENGL Visual = 0x%02x\n",visual->visualid));
    
#else
    
    
    PIXELFORMATDESCRIPTOR pfd;
    HDC hdc;
    int gdblbuf,glstereo,gldepth,glaccel;      
    int glred,glgreen,glblue,glalpha,glstencil;
    int glaccred,glaccgreen,glaccblue,glaccalpha;
    int bestvis,dmatch,bestmatch;
    int dred,dgreen,dblue,ddepth,dalpha,dstencil;
    int daccred,daccgreen,daccblue,daccalpha;
    int nvi,i;
    
    // Get some window handle
    hdc=GetDC(GetDesktopWindow());
    
    // Get number of supported pixel formats
    pfd.nSize=sizeof(PIXELFORMATDESCRIPTOR);   
    pfd.nVersion=1;
    nvi=DescribePixelFormat(hdc,1,sizeof(PIXELFORMATDESCRIPTOR),&pfd);
    if(nvi==0){ fxerror("%s::init: no OpenGL visual available.\n",getClassName()); }

    // Try to find the best
    bestvis=-1;
    bestmatch=1000000000;

    for(i=1; i<=nvi; i++) {

      // Get info about this visual
      DescribePixelFormat(hdc,i,sizeof(PIXELFORMATDESCRIPTOR),&pfd);

      // GL RGBA Support is requested.
      // If VISUAL_NOACCEL is specified, we skip hardware accelerated formats;
      // this is useful if you want to avoid using broken graphics hardware drivers.
      if((pfd.dwFlags&PFD_SUPPORT_OPENGL) && (pfd.iPixelType==PFD_TYPE_RGBA) && ((pfd.dwFlags&PFD_GENERIC_FORMAT) || !(flags&VISUAL_NOACCEL))){

        // Is this a hardware-accelerated visual?
        glaccel=(pfd.dwFlags&PFD_GENERIC_FORMAT)==0;

        // Must have single/double buffer as requested
        gdblbuf=(pfd.dwFlags&PFD_DOUBLEBUFFER)!=0;

        // Stereo support as requested
        glstereo=(pfd.dwFlags&PFD_STEREO)!=0;

        // Get planes
        glred=pfd.cRedBits;
        glgreen=pfd.cGreenBits;
        glblue=pfd.cBlueBits;
        glalpha=pfd.cAlphaBits;
        gldepth=pfd.cDepthBits;
        glstencil=pfd.cStencilBits;
        glaccred=pfd.cAccumRedBits;
        glaccgreen=pfd.cAccumGreenBits;
        glaccblue=pfd.cAccumBlueBits;
        glaccalpha=pfd.cAccumAlphaBits;

        // We prefer to get a few MORE bits in RGBA than we asked for
        dred   = glred-redSize;     if(dred<0)   dred   *= -100;
        dgreen = glgreen-greenSize; if(dgreen<0) dgreen *= -100;
        dblue  = glblue-blueSize;   if(dblue<0)  dblue  *= -100;
        dalpha = glalpha-alphaSize; if(dalpha<0) dalpha *= -100;

        // Prefer better Z than asked, but colors more important
        ddepth = gldepth-depthSize; if(ddepth<0) ddepth *= -10;

        // Want the best colors, of course
        dmatch=dred+dgreen+dblue;

        // We care about colors and Z depth more than stencil depth
        dstencil = glstencil-stencilSize; if(dstencil<0) dstencil *= -1;
        
        // Accumulation buffers
        daccred=glaccred-accumRedSize;       if(daccred<0)   daccred   *= -1;
        daccgreen=glaccgreen-accumGreenSize; if(daccgreen<0) daccgreen *= -1;
        daccblue=glaccblue-accumBlueSize;    if(daccblue<0)  daccblue  *= -1;
        daccalpha=glaccalpha-accumAlphaSize; if(daccalpha<0) daccalpha *= -1;

        // Want the best colors, of course
        dmatch=dred+dgreen+dblue+dalpha;

        // Accumulation buffers
        dmatch+=daccred+daccgreen+daccblue+daccalpha;

        // Hardware accelerated a plus
        if(!glaccel){
	  dmatch+=10000;
	  }

        // Want alpha
        if(alphaSize>0){
	  if(glalpha<1) dmatch+=1000000;
	  else dmatch+=dalpha;
	  }
        else{
	  if(glalpha>0) dmatch+=1000000;
	  }

        // Wanted Z-buffer
        if(depthSize>0){
	  if(gldepth<1) dmatch+=100000000;
	  else dmatch+=ddepth;
	  }
        else{
	  if(gldepth>0) dmatch+=100000000;
	  }

        // Double buffering also quite strongly preferred
        if(flags&VISUAL_DOUBLEBUFFER){
	  if(!gdblbuf) dmatch+=10000000;
	  }
        else{
	  if(gdblbuf) dmatch+=10000000;
	  }

        // Stencil buffers desired
        if(stencilSize>0){
          if(glstencil<1) dmatch+=10000;
          else dmatch+=dstencil;
          }
        else{
          if(glstencil>0) dmatch+=10000;
          }
        
        // Stereo not so important
        if(flags&VISUAL_STEREO){
	  if(!glstereo) dmatch+=100000;
	  }
        else{
	  if(glstereo) dmatch+=100000;
	  }

        // Trace
        FXTRACE((100,"Pixel Format (%d) match value = %d\n",i,dmatch));
        FXTRACE((100,"  red size    = %d\n",glred));
        FXTRACE((100,"  green size  = %d\n",glgreen));
        FXTRACE((100,"  blue size   = %d\n",glblue));
        FXTRACE((100,"  alpha size  = %d\n",glalpha));
        FXTRACE((100,"  depth size  = %d\n",gldepth));
        FXTRACE((100,"  double buf  = %d\n",gdblbuf));
        FXTRACE((100,"  stencil     = %d\n",glstencil));
        FXTRACE((100,"  acc red     = %d\n",glaccred));
        FXTRACE((100,"  acc green   = %d\n",glaccgreen));
        FXTRACE((100,"  acc blue    = %d\n",glaccblue));
        FXTRACE((100,"  acc alpha   = %d\n",glaccalpha));
        FXTRACE((100,"  stereo      = %d\n",glstereo));
        FXTRACE((100,"  accelerated = %d\n",glaccel));

        // May the best visual win
        if(dmatch<bestmatch){
	  bestmatch=dmatch;
	  bestvis=i;
	  }
        }
      }

    // Still no luck
    if(bestvis<0){ fxerror("%s::init: no OpenGL visual available.\n",getClassName()); }

    // Report best visual
    FXTRACE((100,"Best Pixel Format (%d) match value = %d\n",bestvis,bestmatch));
    
    // Get the true h/w capabilities
    pixelformat=bestvis;
    FXMALLOC(&info,PIXELFORMATDESCRIPTOR,1);
    info->nSize=sizeof(PIXELFORMATDESCRIPTOR);
    info->nVersion=1;
    DescribePixelFormat(hdc,pixelformat,sizeof(PIXELFORMATDESCRIPTOR),info);
    
    // Make a palette for it if needed
    if(info->dwFlags&PFD_NEED_PALETTE){
      makeOpenGLPalette();
      }
    
    // Done with that window
    ReleaseDC(GetDesktopWindow(),hdc);
    
//////////////////// STILL MISSING: SETUP FOR PALETTE, NUMCOLOR ETC ////////////////
    
#endif
    
    // Report what we got if tracing enabled
    FXTRACE((100,"  redSize:        %d\n",getActualRedSize()));
    FXTRACE((100,"  greenSize:      %d\n",getActualGreenSize()));
    FXTRACE((100,"  blueSize:       %d\n",getActualBlueSize()));
    FXTRACE((100,"  alphaSize:      %d\n",getActualAlphaSize()));
    FXTRACE((100,"  depthSize:      %d\n",getActualDepthSize()));
    FXTRACE((100,"  stencilSize:    %d\n",getActualStencilSize()));
    FXTRACE((100,"  accumRedSize:   %d\n",getActualAccumRedSize()));
    FXTRACE((100,"  accumGreenSize: %d\n",getActualAccumGreenSize()));
    FXTRACE((100,"  accumBlueSize:  %d\n",getActualAccumBlueSize()));
    FXTRACE((100,"  accumAlphaSize: %d\n",getActualAccumAlphaSize()));
    FXTRACE((100,"  doubleBuffer:   %d\n",isDoubleBuffer()));
    FXTRACE((100,"  stereo:         %d\n",isStereo()));
    FXTRACE((100,"  accelerated:    %d\n",isAccelerated()));
    
#endif
    
    // Prevent subsequent initializations
#ifndef FX_NATIVE_WIN32
    xid=1;
#else
    xid=(void*)1;
#endif
    }
  }


// Get actual red size
FXint FXGLVisual::getActualRedSize() const {
  if(!info){ fxerror("%s::getActualRedSize: visual not yet initialized.\n",getClassName()); }
#ifdef HAVE_OPENGL
  FXint s;
#ifndef FX_NATIVE_WIN32
  glXGetConfig(getApp()->display,info,GLX_RED_SIZE,&s);
#else
  s=info->cRedBits;
#endif
  return s;
#else
  return 0;
#endif
  }


// Get actual green size
FXint FXGLVisual::getActualGreenSize() const {
  if(!info){ fxerror("%s::getActualGreenSize: visual not yet initialized.\n",getClassName()); }
#ifdef HAVE_OPENGL
  FXint s;
#ifndef FX_NATIVE_WIN32
  glXGetConfig(getApp()->display,info,GLX_GREEN_SIZE,&s);
#else
  s=info->cGreenBits;
#endif
  return s;
#else
  return 0;
#endif
  }


// Get actual blue size
FXint FXGLVisual::getActualBlueSize() const {
  if(!info){ fxerror("%s::getActualBlueSize: visual not yet initialized.\n",getClassName()); }
#ifdef HAVE_OPENGL
  FXint s;
#ifndef FX_NATIVE_WIN32
  glXGetConfig(getApp()->display,info,GLX_BLUE_SIZE,&s);
#else
  s=info->cBlueBits;
#endif
  return s;
#else
  return 0;
#endif
  }


// Get actual alpha size
FXint FXGLVisual::getActualAlphaSize() const {
  if(!info){ fxerror("%s::getActualAlphaSize: visual not yet initialized.\n",getClassName()); }
#ifdef HAVE_OPENGL
  FXint s;
#ifndef FX_NATIVE_WIN32
  glXGetConfig(getApp()->display,info,GLX_ALPHA_SIZE,&s);
#else
  s=info->cAlphaBits;
#endif
  return s;
#else
  return 0;
#endif
  }



// Get actual depth size
FXint FXGLVisual::getActualDepthSize() const {
  if(!info){ fxerror("%s::getActualDepthSize: visual not yet initialized.\n",getClassName()); }
#ifdef HAVE_OPENGL
  FXint s;
#ifndef FX_NATIVE_WIN32
  glXGetConfig(getApp()->display,info,GLX_DEPTH_SIZE,&s);
#else
  s=info->cDepthBits;
#endif
  return s;
#else
  return 0;
#endif
  }
  
// Get actual stencil size
FXint FXGLVisual::getActualStencilSize() const {
  if(!info){ fxerror("%s::getActualStencilSize: visual not yet initialized.\n",getClassName()); }
#ifdef HAVE_OPENGL
  FXint s;
#ifndef FX_NATIVE_WIN32
  glXGetConfig(getApp()->display,info,GLX_STENCIL_SIZE,&s);
#else
  s=info->cStencilBits;
#endif
  return s;
#else
  return 0;
#endif
  }
  
  
// Get actual accum red size
FXint FXGLVisual::getActualAccumRedSize() const {
  if(!info){ fxerror("%s::getActualAccumRedSize: visual not yet initialized.\n",getClassName()); }
#ifdef HAVE_OPENGL
  FXint s;
#ifndef FX_NATIVE_WIN32
  glXGetConfig(getApp()->display,info,GLX_ACCUM_RED_SIZE,&s);
#else
  s=info->cAccumRedBits;
#endif
  return s;
#else
  return 0;
#endif
  }
  
  
// Get actual accum green size
FXint FXGLVisual::getActualAccumGreenSize() const {
  if(!info){ fxerror("%s::getActualAccumGreenSize: visual not yet initialized.\n",getClassName()); }
#ifdef HAVE_OPENGL
  FXint s;
#ifndef FX_NATIVE_WIN32
  glXGetConfig(getApp()->display,info,GLX_ACCUM_GREEN_SIZE,&s);
#else
  s=info->cAccumGreenBits;
#endif
  return s;
#else
  return 0;
#endif
  }
  
  
// Get actual accum blue size
FXint FXGLVisual::getActualAccumBlueSize() const {
  if(!info){ fxerror("%s::getActualAccumBlueSize: visual not yet initialized.\n",getClassName()); }
#ifdef HAVE_OPENGL
  FXint s;
#ifndef FX_NATIVE_WIN32
  glXGetConfig(getApp()->display,info,GLX_ACCUM_BLUE_SIZE,&s);
#else
  s=info->cAccumBlueBits;
#endif
  return s;
#else
  return 0;
#endif
  }
  
  
// Get actual accum alpha size
FXint FXGLVisual::getActualAccumAlphaSize() const {
  if(!info){ fxerror("%s::getActualAccumAlphaSize: visual not yet initialized.\n",getClassName()); }
#ifdef HAVE_OPENGL
  FXint s;
#ifndef FX_NATIVE_WIN32
  glXGetConfig(getApp()->display,info,GLX_ACCUM_ALPHA_SIZE,&s);
#else
  s=info->cAccumAlphaBits;
#endif
  return s;
#else
  return 0;
#endif
  }
  
  
// Is it double buffer
FXbool FXGLVisual::isDoubleBuffer() const {
  if(!info){ fxerror("%s::isDoubleBuffer: visual not yet initialized.\n",getClassName()); }
#ifdef HAVE_OPENGL
  FXint s;
#ifndef FX_NATIVE_WIN32
  glXGetConfig(getApp()->display,info,GLX_DOUBLEBUFFER,&s);
#else
  s=(info->dwFlags&PFD_DOUBLEBUFFER)!=0;
#endif
  return s;
#else
  return FALSE;
#endif
  }


// Is it stereo
FXbool FXGLVisual::isStereo() const {
  if(!info){ fxerror("%s::isStereo: visual not yet initialized.\n",getClassName()); }
#ifdef HAVE_OPENGL
  FXint s;
#ifndef FX_NATIVE_WIN32
  glXGetConfig(getApp()->display,info,GLX_STEREO,&s);
#else
  s=(info->dwFlags&PFD_STEREO)!=0;
#endif
  return s;
#else
  return FALSE;
#endif
  }


// Is it hardware-accelerated?
FXbool FXGLVisual::isAccelerated() const {
  if(!info){ fxerror("%s::isHardwareAccelerated: visual not yet initialized.\n",getClassName()); }
#ifdef HAVE_OPENGL
#ifndef FX_NATIVE_WIN32
#ifdef HAVE_MESA
  return FALSE;
#else
  return TRUE;
#endif
#else
  return (info->dwFlags&PFD_GENERIC_FORMAT)==0;
#endif
#else
  return FALSE;
#endif
  }


// Save to stream
void FXGLVisual::save(FXStream& store) const {
  FXVisual::save(store);
  store << redSize;
  store << greenSize;
  store << blueSize;
  store << alphaSize;
  store << depthSize;
  store << stencilSize;
  store << accumRedSize;
  store << accumGreenSize;
  store << accumBlueSize;
  store << accumAlphaSize;
  }


// Load from stream
void FXGLVisual::load(FXStream& store){ 
  FXVisual::load(store);
  store >> redSize;
  store >> greenSize;
  store >> blueSize;
  store >> alphaSize;
  store >> depthSize;
  store >> stencilSize;
  store >> accumRedSize;
  store >> accumGreenSize;
  store >> accumBlueSize;
  store >> accumAlphaSize;
  }


// Destroy
FXGLVisual::~FXGLVisual(){
  FXTRACE((100,"FXGLVisual::~FXGLVisual %08x\n",this));
  if(info) FXFREE(&info);
  }


