// ---------------------------------------------------------------------------
// - cstring.hxx                                                             -
// - standard system library - c string native function definition           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

/// This package contains c-string related procedures. A c string is a null
/// terminated array of characters. Most of the procedures given below are a
/// direct wrapping to standard library procedures.
/// @author amaury darsch

#ifndef ALEPH_CSTRING_HXX
#define ALEPH_CSTRING_HXX

namespace aleph {

  /// Get the size of the c string. The function is safe with null pointer.
  /// @param s the string to evaluate
  /// @return the string length or 0 if null pointer.
  long c_strlen (const char* s);

  /// Compare two strings and returns true if they are equals. This function 
  /// is safe with null pointer.
  /// @param s1 the first string
  /// @param s2 the second string
  /// @return true if the string are equals or both null
  bool c_strcmp (const char* s1, const char* s2);
 
  /// Compare two strings and returns true if they are equals for a number of
  /// characters. This function is safe with null pointer.
  /// @param s1 the first string
  /// @param s2 the second string
  /// @param size the number of characters to compare
  /// @return true if the string are equals or both null
  bool c_strncmp (const char* s1, const char* s2, const long size);
  
  /// Duplicate a string. If the string is null or has a 0 length, the 
  /// function returns the null pointer. The delete operator cleans the 
  /// strings
  /// @param s the string to duplicate
  /// @return a copy of the string
  char* c_strdup (const char* s);
  
  /// Create a string from a character.
  /// @param c the character to map to a string
  /// @return a copy of the string
  char* c_strmak (const char c);
  
  /// Copy a string from a source to destination
  /// @param dst the destination string
  /// @param src the source string
  void c_strcpy (char* dst, const char* src);
  
  /// Concatenate a string with another. The string must allocated enough.
  /// @param dst the destination string
  /// @param src the source string
  void c_strcat (char* dst, const char* src);
  
  /// Remove the leading blank and tab and return a new string
  /// @param the original string
  /// @return a new clean string
  char* c_rmlead (const char* s);
  
  /// Remove the trailing blank and tab and return a new string
  /// @param the original string
  /// @return a new clean string
  char* c_rmtrail (const char* s);
  
  /// Convert the string to upper case
  /// @param the original string
  /// @return a new upper case string
  char* c_toupper (const char* s);
  
  /// Convert the string to lower case
  /// @param the original string
  /// @return a new lower case string
  char* c_tolower (const char* s);
  
  /// Convert an integer to a c-string representation. Please delete the
  /// result when no longer needed.
  /// @param value the integer value
  /// @return the c-string representation
  char* c_ltoa (const long value);
  
  /// convert a long long integer to a c-string representation
  /// @param value the integer to convert
  /// @return the character buffer
  char* c_lltoa (const long long int value);
  
  /// convert a char string to a long long integer
  /// @param buffer the buffer to convert
  /// @param base the base to convert
  /// @param status a boolean flag set to true if the conversion succeds
  /// @return a long long integer value
  long long int c_atoll (const char* buffer, const int base, bool& status);
  
  /// convert a double float to a c-string representation
  /// @param value the float to convert
  /// @return the character buffer
  char* c_dtoa (const double value);
  
  /// convert a double float to a c-string representation with a format
  /// @param value the float to convert
  /// @param precision the conversion precision
  /// @return the character buffer
  char* c_dtoap (const double value, const long precision);
  
  /// convert a char string to a double float
  /// @param buffer the buffer to convert
  /// @param status a boolean flag set to true if the conversion succeds
  /// @return a double float value
  double c_atod (const char* buffer, bool& status);      
}

#endif
