// ---------------------------------------------------------------------------
// - String.cpp                                                              -
// - standard object library - string class implementation                   -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "cstring.hxx"
#include "String.hpp"
#include "Method.hpp"
#include "Integer.hpp"
#include "Boolean.hpp"
#include "Character.hpp"
#include "Exception.hpp"

// compute the length of a string - so much used we do it here
static inline long lenstr (const char* s) {
  if (s == aleph::nilp) return 0;
  long result = 0;
  while (*s++ != '\0') result++;
  return result;
}

// compare two strings - again local inline for performance
static inline bool cmpstr (const char* s1, const char* s2) {
  // nilp case first
  const char* ss1 = (s1 == aleph::nilp) ? "\0" : s1;
  const char* ss2 = (s2 == aleph::nilp) ? "\0" : s2;

  // check first character for fast compare
  if (*ss1 != *ss2) return false;

  // normal compare now
  while (*ss1 != '\0') {
    if (*ss2 == '\0') break;
    if (*ss1++ != *ss2++) return false;
  }
  return (*ss1 == *ss2) ? true : false;
}

namespace aleph {

  // the string data structure is private to this unit
  struct s_string {
    // string holder
    char* p_buffer;
    // reference count
    long  d_rcount;
    
    // simple constructor
    s_string (void) {
      p_buffer = aleph::nilp;
      d_rcount = 1;
    }
  };

  // Create an empty string

  String::String (void) {
    p_string = new s_string;
  }
  
  // Create a string from a c-string
  
  String::String (const char* s) {
    // create a new string holder
    p_string = new s_string;
    // copy the string
    p_string->p_buffer = c_strdup (s);
  }
  
  // Create a string from a character
  
  String::String (const char c) {
    // create a new string holder
    p_string = new s_string;
    // copy the string
    p_string->p_buffer = c_strmak (c);
  }
  
  // copy constructor
  
  String::String (const String& that) {
    that.p_string->d_rcount++;
    p_string = that.p_string;
  }
  
  // Destroy this string
  
  String::~String (void) {
    if (--p_string->d_rcount == 0) {
      delete [] p_string->p_buffer;
      delete p_string;
    }
  }

  // return the class name

  String String::repr (void) const {
    return "String";
  }

  // return a literal representation of this string
  
  String String::toLiteral (void) const {
    String result = "\"";
    result = result + p_string->p_buffer + "\"";
    return result;
  }

  // return a printable representation of this string
  
  String String::toString (void) const {
    return *this;
  }

  // assign a c-string to this string
  
  String& String::operator = (const char* s) {
    // disconnect the string or free
    if (p_string->d_rcount > 1) {
      p_string->d_rcount--;
      p_string = new s_string;
    } else {
      delete [] p_string->p_buffer;
    }
    
    // copy localy the string
    p_string->p_buffer = c_strdup (s);
    return *this;
  }
  
  // assign a character to this string
  
  String& String::operator = (const char c) {
    // disconnect the string or free
    if (p_string->d_rcount > 1) {
      p_string->d_rcount--;
      p_string = new s_string;
    } else {
      delete [] p_string->p_buffer;
    }
    
    // copy localy the string
    p_string->p_buffer = c_strmak (c);
    return *this;
  }
  
  // assign a string to this string
  
  String& String::operator = (const String& s) {
    // protect the case s = s
    s.p_string->d_rcount++;
    
    // disconnect the string
    if (--p_string->d_rcount == 0) {
      delete [] p_string->p_buffer;
      delete p_string;
    }
    
    // copy localy the string
    p_string = s.p_string;
    return *this;
  }

  // get a character at a certain position

  char String::operator [] (const long index) const {
    // check for good position first
    if ((index < 0) || (index >= this->length ()))
      throw Exception ("bound-error","in string operator []");
    // everything is fine
    return p_string->p_buffer[index];
  }

  // add a string to the current one. return a newly created string
  
  String String::operator + (const String& s) const {
    // create a temporary buffer to hold the string
    long size = this->length () + s.length () + 1;
    char* buffer = new char [size];
    
    // copy and concat the string
    c_strcpy (buffer,p_string->p_buffer);
    c_strcat (buffer,s.p_string->p_buffer);
    
    // create a new string and clean buffer
    String result = buffer;
    delete [] buffer;
    return result;
  }
  
  // add a character to the current string. Return a newly created string
  
  String String::operator + (const char c) const {
    // create a temporary buffer to hold the string
    long  size = this->length () + 2;
    char* buffer = new char [size];
    
    // copy the string and set character
    c_strcpy (buffer,p_string->p_buffer);
    buffer[size - 2] = c;
    buffer[size - 1] = '\0';
    
    // create a new string and clean buffer
    String result = buffer;
    delete [] buffer;
    return result;
  }

  // add an integer to the current string. Return a newly created string
  
  String String::operator + (const long value) const {
    // create a temporary buffer to hold the string
    char* cvalue = c_ltoa (value);
    
    // create a temporary buffer to hold the string
    long  size     = this->length () + lenstr (cvalue) + 1;
    char* buffer = new char [size];
    
    // copy and concat the string
    c_strcpy (buffer,p_string->p_buffer);
    c_strcat (buffer,cvalue);
    
    // create a new string and clean buffer
    String result = buffer;
    delete [] buffer;
    delete [] cvalue;
    return result;
  }
  
  // compare a string with another one
  
  bool String::operator == (const String& s) const {
    return cmpstr (p_string->p_buffer,s.p_string->p_buffer);
  }

  // compare a string with a c-string
  
  bool String::operator == (const char* s) const {
    return cmpstr (p_string->p_buffer,s);
  }
  
  // compare a string with another one
  
  bool String::operator != (const String& s) const {
    bool result = cmpstr (p_string->p_buffer,s.p_string->p_buffer);
    return result ? false : true;
  }
  
  // compare a string with a c-string
  
  bool String::operator != (const char* s) const {
    bool result = cmpstr (p_string->p_buffer,s);
    return result ? false : true;
  }

  // return the length of this string
  
  long String::length (void) const {
    return lenstr (p_string->p_buffer);
  }
  
  // return the length of a c-string
  
  long String::length (const char* s) {
    return lenstr (s);
  }

  // return a c-string representation or nilp for this string

  char* String::toChar (void) const {
    return c_strdup (p_string->p_buffer);
  }
  
  // remove leading blank from this string
  
  String String::stripl (void) const {
    char* buffer  = c_rmlead (p_string->p_buffer);
    String result = buffer;
    delete [] buffer;
    return result;
  }
  
  // remove trailing blank from this string
  
  String String::stripr (void) const {
    char*  buffer  = c_rmtrail (p_string->p_buffer);
    String result  = buffer;
    delete [] buffer;
    return result;
  }
  
  // remove leading and trailing blank from this string
  
  String String::strip (void) const {
    char*  lbuffer = c_rmlead  (p_string->p_buffer);
    char*  rbuffer = c_rmtrail (lbuffer);
    String result  = rbuffer;
    delete [] lbuffer;
    delete [] rbuffer;
    return result;
  }

  // convert the string to upper case
  
  String String::toUpper (void) const {
    char* buffer  = c_toupper (p_string->p_buffer);
    String result = buffer;
    delete [] buffer;
    return result;
  }

  // convert the string to lower case
  
  String String::toLower (void) const {
    char* buffer  = c_tolower (p_string->p_buffer);
    String result = buffer;
    delete [] buffer;
    return result;
  }

  // return the hashid for this string
  
  long String::hashid (void) const {
    if (p_string->p_buffer == nilp) return 0;
    
    // each letter is shifted from 0 to 23 bits, like 17, 10, 3, etc ...
    char  c;
    long  result = 0;
    long  cshift = 17;
    char* cptr   = p_string->p_buffer;
    
    while ((c = *cptr++) != '\0') {
      result = result ^ (((long) c) << cshift);
      if ((cshift = cshift - 7) < 0) cshift += 24;
    }
    return (result > 0) ? result : -result;
  }

  // return the right substring starting at a certin index

  String String::rsubstr (const long index) const {
    String result;
    long len = length ();
    if ((len == 0) || (index >= len-1)) return result;
    char* sptr = p_string->p_buffer;
    return String (sptr + index);
  }

  // return the left substring of a string

  String String::lsubstr (const long index) const {
    String result;
    long len = length ();
    if ((len == 0) || (index > len)) return result;
    char* sptr = c_strdup (p_string->p_buffer);
    sptr[index] = nilc;
    result = sptr;
    delete [] sptr;
    return result;
  }

  // return a substring of a string

  String String::substr (const long lidx, const long ridx) const {
    long len = length ();
    if ((lidx >= ridx) || (lidx < 0) || (lidx >= len) || (ridx <0) ||
	(ridx >  len)  || (len == 0)) 
      throw Exception ("index-error", "invalid index for sub string");
    // create th result string
    String result;
    char* buf = c_strdup (p_string->p_buffer);
    char* sub = buf + lidx;
    buf[ridx] = nilc;
    result = sub;
    delete [] buf;
    return result;
  }

  // fill this string with a character until a given size is reached
  
  String String::lfill (const char c, const long size) const {
    // do nothing if size exceed
    long len = size - length ();
    if (len <= 0) return *this;
    // fill the string first
    String result;
    for (long i = 0; i < len; i++) result = result + c;
    return result + *this;
  }

  // fill this string with a character until a given size is reached
  
  String String::rfill (const char c, const long size) const {
    // do nothing if size exceed
    long len = size - length ();
    if (len <= 0) return *this;
    // fill the string first
    String result = *this;
    for (long i = 0; i < len; i++) result = result + c;
    return result;
  }

  // operate this string with another object

  Object* String::oper (t_oper type, Object* object) {
    Literal* lobj = dynamic_cast <Literal*> (object);
    String*  sobj = dynamic_cast <String*>    (object);

    switch (type) {
    case Object::ADD:
      if (lobj != nilp) return new String (*this + lobj->toString ());
      break;  
    case Object::EQL:
      if (sobj != nilp) return new Boolean (*this == *sobj);
      break;
    case Object::NEQ:
      if (sobj != nilp) return new Boolean (*this != *sobj);
      break;
    default:
      throw Exception ("operator-error", "unsupported string operator");
    }
    throw Exception ("type-error", "invalid operand with string",
		     Object::repr (object));
  }

  // create a new string in a generic way

  Object* String::mknew (Vector* argv) {
    if ((argv == nilp) || (argv->length () == 0)) return new String;
    if (argv->length () != 1) 
      throw Exception ("argument-error", 
		       "too many argument with string constructor");
    // try to map the string argument
    Object* obj = argv->get (0);
    if (obj == nilp) return new String;

    // try a literal object
    Literal* lval = dynamic_cast <Literal*> (obj);
    if (lval != nilp) return new String (lval->toString ());

    // illegal object
    throw Exception ("type-error", "illegal object with string constructor",
		     obj->repr ());
  }

  // set an object to this string

  Object* String::vdef (Interp* interp, Nameset* nset, Object* object) {
    Literal* lobj = dynamic_cast <Literal*> (object);
    if (lobj != nilp) {
      *this = lobj->toString ();
      return this;
    }
    throw Exception ("type-error", "invalid object with string vdef",
		     Object::repr (object));
  }

  // evaluate this string with a member name

  Object* String::eval (Interp* interp, Nameset* nset, const String& name) {
    return new Method (name, this);
  }

  // apply this string with a method name

  Object* String::apply (Interp* interp, Nameset* nset, const String& name,
			 Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) delete argv;
    if ((name == "length") && (argc == 0)) 
      return new Integer (length ());
    if ((name == "strip-left") && (argc == 0)) 
      return new String (stripl ());
    if ((name == "strip-right") && (argc == 0)) 
      return new String (stripr ());
    if ((name == "strip") && (argc == 0)) 
      return new String (strip  ());
    if ((name == "to-upper") && (argc == 0)) 
      return new String (toUpper ());
    if ((name == "to-lower") && (argc == 0)) 
      return new String (toLower ());
    if ((name == "hashid") && (argc == 0)) 
      return new Integer(hashid ());
    // dispatch one argument
    if ((name == "+") && (argc == 1)) {
      Object* result = oper (Object::ADD, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "==") && (argc == 1)) {
      Object* result = oper (Object::EQL, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "!=") && (argc == 1)) {
      Object* result = oper (Object::NEQ, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "+=") && (argc == 1)) {
      String val = argv->getstring (0);
      *this = *this + val;
      delete argv;
      return this;
    }
    if ((name == "get") && (argc == 1)) {
      t_long val = argv->getint (0);
      char c = (*this)[val];
      delete argv;
      return new Character (c);
    }
    if ((name == "sub-right") && (argc == 1)) {
      t_long val = argv->getint (0);
      String result = rsubstr (val);
      delete argv;
      return new String (result);
    }
    if ((name == "sub-left") && (argc == 1)) {
      t_long val = argv->getint (0);
      String result = lsubstr (val);
      delete argv;
      return new String (result);
    }

    // dispatch two arguments
    if ((name == "fill-left") && (argc == 2)) {
      char   c    = argv->getchar (0);
      t_long size = argv->getint  (1);
      String result = lfill (c, size);
      delete argv;
      return new String (result);
    }
    if ((name == "fill-right") && (argc == 2)) {
      char   c    = argv->getchar (0);
      t_long size = argv->getint  (1);
      String result = rfill (c, size);
      delete argv;
      return new String (result);
    }
    if ((name == "substr") && (argc == 2)) {
      t_long lidx = argv->getint (0);
      t_long ridx = argv->getint (1);
      String result = substr (lidx, ridx);
      delete argv;
      return new String (result);
    }

    // call the object method
    Object* result = nilp;
    try {
      result =  Object::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    return result;
  }
}
