// ---------------------------------------------------------------------------
// - Predicate.cpp                                                           -
// - aleph engine - predicate builtin functions implementation               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Real.hpp"
#include "Cons.hpp"
#include "List.hpp"
#include "Vector.hpp"
#include "Interp.hpp"
#include "Builtin.hpp"
#include "Boolean.hpp"
#include "Instance.hpp"
#include "Exception.hpp"
#include "Character.hpp"

namespace aleph {

  // nilp: nilp predicate

  Object* builtin_nilp (Interp* interp, Nameset* nset, Cons* args) {
    if (args == nilp) return new Boolean (true);
    if (args->length () != 1) 
      throw Exception ("argument-error","too many arguments with nil-p");
    Object* car = args->getcar ();
    Object* obj = (car == nilp) ? nilp : car->eval (interp,nset);
    if (obj == nilp) return new Boolean (true);
    return new Boolean (false);
  }

  // this procedure checks that we have one argument only and returns
  // the evaluated object
  static inline Object* get_obj (Interp* interp, Nameset* nset, Cons* args,
				 const String& pname) {
    if ((args == nilp) || (args->length () != 1))
      throw Exception ("argument-error", "illegal arguments with predicate",
		       pname);
    Object* car = args->getcar ();
    Object* obj = (car == nilp) ? nilp : car->eval (interp,nset);
    return obj;
  }

  // intp: integer predicate

  Object* builtin_intp (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "integer-p");
    if (dynamic_cast <Integer*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // realp: real predicate

  Object* builtin_realp (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "real-p");
    if (dynamic_cast <Real*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // nump: number predicate

  Object* builtin_nump (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "number-p");
    if (dynamic_cast <Integer*> (obj) != nilp) return new Boolean (true);
    if (dynamic_cast <Real*>    (obj) != nilp) return new Boolean (true);
    return new Boolean (false);
  }
  
  // boolp: boolean predicate

  Object* builtin_boolp (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "boolean-p");
    if (dynamic_cast <Boolean*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // charp: character predicate

  Object* builtin_charp (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "character-p");
    if (dynamic_cast <Character*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }
  
  // strp: string predicate

  Object* builtin_strp (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "string-p");
    if (dynamic_cast <String*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }
  
  // litp: literal predicate

  Object* builtin_litp (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "literal-p");
    if (dynamic_cast <Literal*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }
  
  // vecp: vector predicate

  Object* builtin_vecp (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "vector-p");
    if (dynamic_cast <Vector*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // consp: cons predicate

  Object* builtin_consp (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "cons-p");
    if (dynamic_cast <Cons*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // listp: cons predicate

  Object* builtin_listp (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "list-p");
    if (dynamic_cast <List*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }
  
  // nstp: nameset predicate

  Object* builtin_nstp (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "nameset-p");
    if (dynamic_cast <Nameset*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // clsp: class predicate

  Object* builtin_clsp (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "class-p");
    if (dynamic_cast <Class*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // instp: instance predicate

  Object* builtin_instp (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "instance-p");
    if (dynamic_cast <Instance*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }

  // ashp: hash table predicate

  Object* builtin_ashp (Interp* interp, Nameset* nset, Cons* args) {
    Object* obj = get_obj (interp, nset, args, "hashtable-p");
    if (dynamic_cast <HashTable*> (obj) == nilp) return new Boolean (false);
    return new Boolean (true);
  }
}
