/******************************************************************************
    Xplanet 0.43 - render an image of the earth into an X window
    Copyright (C) 1999 Hari Nair <hari@alumni.caltech.edu>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

******************************************************************************/

/*
  planet.cc - parameters for orbital and rotational motion.  Orbital 
  parameters are from Chapter 23 of Astronomical Formulae for Calculators 
  by Meeus, rotational parameters for the planets are from Practical Ephemeris 
  Calculations by Montenbruck.  Rotational parameters for the moon are 
  from Davies et al. (1996), Celestial Mechanics 63, 127--148.
*/

#include <iostream.h>
#include <math.h>

#include "planet.h"
#include "util.h"
#include "xplanet.h"

char *body[5] = {"earth","mercury","venus","moon","mars"};
char *current_body = body[0];
int flipped = 0;  // 0 if planet's longitude increases to the east (like earth)

planet::planet (double T, int ibody)
{
  double T2000 = T - 1;                      // Centuries from 2000.0
  double d = T2000 * 36525;                  // Days from 2000.0
      
  switch (ibody)
    {
    case 1:        // Mercury
      mlong = (poly (1.78179078E2, 1.4947407078E5, 3.011E-4, 0, T)
	       * deg_to_rad);
      axis = 0.3870986;
      ecc = poly (2.0561421E-1, 2.046E-5, -3E-8, 0, T);
      incl = (poly (7.002881, 1.8608E-3, -1.83E-5, 0, T)
	      * deg_to_rad);
      perih = (poly (2.8753753E1, 3.702806E-1, 1.208E-4, 0, T)
	       * deg_to_rad);
      ascnode = (poly (4.7145944E1, 1.1852083, 1.739E-4, 0, T)
		 * deg_to_rad);
      flipped = 1; 

      alpha0 = 281.02 - 0.033 * T2000;               // Equinox J2000.0
      delta0 = 61.45 - 0.005 * T2000;
      alpha0 += 0.276 * T2000;                       // Current equinox
      delta0 += 0.107 * T2000;

      null_meridian = 329.71;                        // Equinox J2000.0
      null_meridian += 1.145 * T2000;                // Current equinox
      wdot = 6.1385025;
      
      break;
    case 2:        // Venus
      mlong = (poly (3.42767053E2, 5.851921191E4, 3.097E-4, 0, T)
	       * deg_to_rad);
      axis = 0.7233316;
      ecc = poly (6.82069E-3, -4.774E-5, 9.1E-8, 0, T);
      incl = (poly (3.393631, 1.0058E-3, -1.0E-6, 0, T)
	      * deg_to_rad);
      perih = (poly (5.4384186E1, 5.081861E-1, -1.3864E-3, 0, T)
	       * deg_to_rad);
      ascnode = (poly (7.5779647E1, 8.9985E-1, 4.1E-4, 0, T)
		 * deg_to_rad);
      flipped = 0; 

      alpha0 = 272.78;               // Equinox J2000.0
      delta0 = 67.21;
      alpha0 -= 0.043 * T2000;                       // Current equinox
      delta0 += 0.027 * T2000;

      null_meridian = 159.91;                        // Equinox J2000.0
      null_meridian += 1.436 * T2000;                // Current equinox
      wdot = -1.4814205;
      
      break;
    case 3:        // Moon, values for J2000.0
      double E[14];
      E[ 1] = 125.045 -  0.0529921 * d;
      E[ 2] = 250.089 -  0.1059842 * d;
      E[ 3] = 260.008 + 13.0120009 * d;
      E[ 4] = 176.625 + 13.3407154 * d;
      E[ 5] = 357.529 +  0.9856003 * d;
      E[ 6] = 311.589 + 26.4057084 * d;
      E[ 7] = 134.963 + 13.0649930 * d;
      E[ 8] = 276.617 +  0.3287146 * d;
      E[ 9] =  34.226 +  1.7484877 * d;
      E[10] =  15.134 -  0.1589763 * d;
      E[11] = 119.743 +  0.0036096 * d;
      E[12] = 239.961 +  0.1643573 * d;
      E[13] =  25.053 + 12.9590088 * d;
      for (int i = 1; i < 14; i++) E[i] *= deg_to_rad;

      alpha0 = 269.9949 + (0.0031 * T2000         - 3.8787 * sin (E[ 1])
			   - 0.1204 * sin (E[ 2]) + 0.0700 * sin (E[ 3])
			   - 0.0172 * sin (E[ 4]) + 0.0072 * sin (E[ 6])
			   - 0.0052 * sin (E[10]) + 0.0043 * sin (E[13]));
      delta0 =  66.5392 + (0.0130 * T2000         + 1.5419 * cos (E[ 1])
			   + 0.0239 * cos (E[ 2]) - 0.0278 * cos (E[ 3])
			   + 0.0068 * cos (E[ 4]) - 0.0029 * cos (E[ 6])
			   + 0.0009 * cos (E[ 7]) + 0.0008 * cos (E[10])
			   - 0.0009 * cos (E[13]));
      null_meridian = 38.3213 + (d * (13.17635815 - 1.4E-12 * d) 
				 + 3.5610 * sin (E[ 1]) + 0.1208 * sin (E[ 2])
				 - 0.0642 * sin (E[ 3]) + 0.0158 * sin (E[ 4])
				 + 0.0252 * sin (E[ 5]) - 0.0066 * sin (E[ 6])
				 - 0.0047 * sin (E[ 7]) - 0.0046 * sin (E[ 8])
				 + 0.0028 * sin (E[ 9]) + 0.0052 * sin (E[10])
				 + 0.0040 * sin (E[11]) + 0.0019 * sin (E[12])
				 - 0.0044 * sin (E[13]));
      wdot = 0;
      flipped = 0;
      break;
    case 4:        // Mars
      mlong = (poly (2.93737334E2, 1.914169551E4, 3.107E-4, 0, T) 
	       * deg_to_rad);
      axis = 1.5236883;
      ecc = poly (9.33129E-2, 9.2064E-5, -7.7E-8, 0, T);
      incl = (poly (1.850333, -6.75E-4, 1.26E-5, 0, T) * deg_to_rad);
      perih = (poly (2.85431761E2, 1.0697667, 1.313E-4, 4.41E-6, T) 
	       * deg_to_rad);
      ascnode = (poly (4.8786442E1, 7.709917E-1, -1.4E-6, -5.33E-6, T) 
		 * deg_to_rad);
      flipped = 1;
      
      alpha0 = 317.681 - 0.108 * T2000;             // Equinox J2000.0
      delta0 = 52.886 - 0.061 * T2000;
      alpha0 += 0.786 * T2000;                      // Current equinox
      delta0 += 0.413 * T2000;

      null_meridian = 176.655;                       // Equinox J2000.0
      null_meridian += 0.620 * T2000;                // Current equinox
      wdot = 350.891983;
      
      break;
    }

  alpha0 *= deg_to_rad;
  delta0 *= deg_to_rad;
  null_meridian += wdot * d;
  null_meridian = fmod (null_meridian, 360.0);
  null_meridian *= deg_to_rad;  

  current_body = body[ibody];
}

void 
planet::calc_helio ()
{
  manom = mlong - perih - ascnode;
  double eccanom = kepler (ecc, manom);
  double nu = 2 * atan (sqrt ((1 + ecc) / (1 - ecc)) 
			* tan (eccanom/2));
  double lat = perih + nu;
  helio.lon = atan2 (cos (incl) * sin (lat), cos (lat)) + ascnode;
  helio.lat = asin (sin (lat) * sin (incl));
  helio.dist = axis * (1 - ecc * cos (eccanom));
}

void 
planet::calc_geo ()
{
  double N = helio.dist * cos (helio.lat) * sin (helio.lon - sunlon);
  double D = (helio.dist * cos (helio.lat) * cos (helio.lon - sunlon)
	      + sundist);
  geo.dist = N*N + D*D + 
    (helio.dist * sin (helio.lat)) * (helio.dist * sin (helio.lat));
  geo.dist = sqrt (geo.dist);
  geo.lon = atan2 (N, D) + sunlon;
  geo.lat = asin (helio.dist * sin (helio.lat) / geo.dist);
}

coordinates
planet::calc_coordinates (coordinates body, double eps)
{
  compute_ra_dec (body.lon, body.lat, body.alpha, body.delta, eps);
  body.alpha = TWO_PI * body.alpha / 24.;

  axis_position = atan2 (cos (delta0) * sin (alpha0 - body.alpha),
			 sin (delta0) * cos (body.delta) 
			 - (cos (delta0) * sin (body.delta) 
			    * cos (alpha0 - body.alpha)));
  double lat = -asin (sin (delta0) * sin (body.delta) 
		      + (cos (delta0) * cos (body.delta) 
			 * cos (alpha0 - body.alpha)));
  double lon = atan2 (sin (delta0) * cos (body.delta) 
		      * cos (alpha0 - body.alpha)
		      - cos (delta0) * sin (body.delta),
		      cos (body.delta) * sin (alpha0 - body.alpha));

  double light_time = 5.7756E-3 * geo.dist * wdot * deg_to_rad;
  lon -= null_meridian;
  lon = fmod (lon, TWO_PI);
  if (lon < 0) lon += TWO_PI;
  coordinates returnval = {lon, lat, axis_position, light_time, 0};
  return (returnval);	
}
