/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "RenderOptions.h"
#include "FileIO.h"
#include "GlobalTrace.h"
#include "GlobalSanityCheck.h"
#include "TFOptions.h"			// for sealevel
#include "terraform.h"


#if 0
// dummy table to trick xgettext into extracting the translatable strings
char *foobar[] = {
			N_("Plastic Model"),
			N_("Real Sky"),
			N_("Real Sky with Sunbeams (Fast)"),
			N_("Real Sky with Sunbeams (Slow)"),
 };
#endif



/* 
 * constructor: initialize all the class members
 */
RenderOptions::RenderOptions (char *HFfilename)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ RenderOptions\n");

	SanityCheck::bailout ((!HFfilename), "HFfilename==NULL", "RenderOptions::writePOVFile");

	char 	buf[256];

	b_smooth = TRUE;
	b_halfYScale = TRUE;
	b_keepFiles = FALSE;
	b_haveClouds = TRUE;
	b_haveFog = FALSE;

	d_width = 400;
	d_height = 300;
	d_seaLevel = TFOptions::s_sealevel;
	d_waterClarity = 0.95;
	d_lightingLevel = 0;
	d_POVquality = 0;

	d_seaLevel = 0.33;
	d_timeOfDay = 12.0;
	d_northDir = 0;
	d_cameraX = 0;
	d_cameraY = 0.75;
	d_cameraZ = -1.1;
	d_lookAtX = d_lookAtY = d_lookAtZ = 0;

	sprintf (buf, "%s/%s", TF_DATADIR, TF_DEFAULT_POV_FILE);
	p_POVfile = strdup (buf);
	p_HFfile = strdup (HFfilename);
}


/* 
 * destructor: delete any dynamically allocated data memebers
 */
RenderOptions::~RenderOptions ()
{
	if (p_POVfile)
		free (p_POVfile);
	if (p_HFfile)
		free (p_HFfile);

	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- RenderOptions\n");
}


/* 
 * writePOVFile: write out a POVRay-style file with the render settings
 */
int RenderOptions::writePOVFile (char *fname)
{
	SanityCheck::bailout ((!p_HFfile), "p_HFfile==NULL", "RenderOptions::writePOVFile");

	FileIO	*fPOV = new FileIO (fname);
	FILE	*fp;
	float	ys = (b_halfYScale ? TFOptions::s_yscale/2 : TFOptions::s_yscale);

	fp = fPOV->open ("w");
	if (!fp)
		return -1;

	if (b_smooth)
		fprintf (fp, "#declare TF_HEIGHT_FIELD = height_field { tga \"%s\" smooth }\n", p_HFfile);
	else
		fprintf (fp, "#declare TF_HEIGHT_FIELD = height_field { tga \"%s\" }\n", p_HFfile);
	fprintf (fp, "#declare TF_HEIGHT_FIELD = height_field { tga \"%s\" smooth }\n", p_HFfile);
	fprintf (fp, "#declare TF_CAMERA_LOCATION = <%f, %f, %f>;\n", d_cameraX-0.5, d_cameraY*ys, (1-d_cameraZ)-0.5);
	fprintf (fp, "#declare TF_CAMERA_LOOK_AT = <%f, %f, %f>;\n", d_lookAtX-0.5, d_lookAtY*ys, (1-d_lookAtZ)-0.5);
	fprintf (fp, "#declare TF_WATER_LEVEL = %f;\n", d_seaLevel);
	fprintf (fp, "#declare TF_Y_SCALE = %f;\n", ys);
	fprintf (fp, "#declare TF_HAVE_WATER = %s;\n", (TFOptions::s_fillSea ? "true" : "false"));
	fprintf (fp, "#declare TF_WATER_CLARITY = %f;\n", d_waterClarity);
	fprintf (fp, "#declare TF_TIME_OF_DAY = %f;\n", d_timeOfDay);
	fprintf (fp, "#declare TF_HAVE_CLOUDS = %s;\n", (b_haveClouds ? "true" : "false"));
	fprintf (fp, "#declare TF_HAVE_FOG = %s;\n", (b_haveFog ? "true" : "false"));
	fprintf (fp, "#declare TF_NORTH_DIR = %f;\n", d_northDir);

	if (d_lightingLevel == 0)
		fprintf (fp, "#declare TF_REAL_LIGHTING = false;\n");
	else
		fprintf (fp, "#declare TF_REAL_LIGHTING = true;\n");

	if (d_lightingLevel > 1)
		fprintf (fp, "#declare TF_HAVE_SUNBEAMS = true;\n");
	else
		fprintf (fp, "#declare TF_HAVE_SUNBEAMS = false;\n");

	if (d_lightingLevel > 2)
		fprintf (fp, "#declare TF_FAST_SUNBEAMS = false;\n");
		
	fprintf (fp, "\n\n");
	fprintf (fp, "#include \"%s\"", p_POVfile);
	fprintf (fp, "\n\n");

	fPOV->close ();
	return 0;
}



/* ******************************************************************** */
/* ** mutators to set parameters and do some limited sanity checking ** */
/* ******************************************************************** */
void RenderOptions::setSmooth (bool smooth)
{
	b_smooth = smooth;
}


void RenderOptions::setHalfYScale (bool halfYScale)
{
	b_halfYScale = halfYScale;
}


void RenderOptions::setKeepFiles (bool keepFiles)
{
	b_keepFiles = keepFiles;
}


void RenderOptions::setHaveClouds (bool haveClouds)
{
	b_haveClouds = haveClouds;
}


void RenderOptions::setHaveFog (bool haveFog)
{
	b_haveFog = haveFog;
}


void RenderOptions::setSize (UI width, UI height)
{
	SanityCheck::bailout ((width < 1), "width<1", "RenderOptions::setSize");
	SanityCheck::bailout ((height < 0), "height<0", "RenderOptions::setSize");

	d_width = width;

	if (height)
		d_height = height;
	else
		d_height = (UI)(d_width*0.75);
}


void RenderOptions::setLightingLevel (UI lightLevel)
{
	d_lightingLevel = lightLevel;
}


//void RenderOptions::setPOVquality (UI quality)
//{
//	d_POVquality = quality;
//}


void RenderOptions::setSealevel (float sealevel)
{
	SanityCheck::bailout ((sealevel < 0), "sealevel<0", "RenderOptions::setSealevel");
	SanityCheck::bailout ((sealevel > 1), "sealevel>1", "RenderOptions::setSealevel");

	d_seaLevel = sealevel;
}


void RenderOptions::setWaterClarity (float clarity)
{
	SanityCheck::bailout ((clarity < 0), "clarity<0", "RenderOptions::setWaterClarity");
	SanityCheck::bailout ((clarity > 1), "clarity>1", "RenderOptions::setWaterClarity");

	d_waterClarity = clarity;
}


void RenderOptions::setTimeOfDay (float timeOfDay)
{
	SanityCheck::bailout ((timeOfDay < 0), "timeOfDay<0", "RenderOptions::setTimeOfDay");
	SanityCheck::bailout ((timeOfDay > 24), "timeOfDay>24", "RenderOptions::setTimeOfDay");

	d_timeOfDay = timeOfDay;
}


void RenderOptions::setNorthDir (float northDir)
{
	SanityCheck::bailout ((northDir < 0), "northDir<0", "RenderOptions::setTimeOfDay");
	SanityCheck::bailout ((northDir > 360), "northDir>24", "RenderOptions::setTimeOfDay");

	d_northDir = northDir;
}


void RenderOptions::setCameraLocation (float x, float y, float z)
{
	d_cameraX = x;
	d_cameraY = y;
	d_cameraZ = z;
}


void RenderOptions::setLookAtLocation (float x, float y, float z)
{
	d_lookAtX = x;
	d_lookAtY = y;
	d_lookAtZ = z;
}


void RenderOptions::setPOVFile (char *filename)
{
	SanityCheck::bailout ((!filename), "filename==NULL", "RenderOptions::setPOVFile");

	if (p_POVfile)
		free (p_POVfile);
	p_POVfile = strdup (filename);
}


void RenderOptions::setHFFile (char *filename)
{
	SanityCheck::bailout ((!filename), "filename==NULL", "RenderOptions::setPOVFile");

	if (p_HFfile)
		free (p_HFfile);
	p_HFfile = strdup (filename);
}


/* ******************************************************************** */
/* **************** accessors to return parameters ******************** */
/* ******************************************************************** */
bool RenderOptions::getSmooth ()
{
	return b_smooth;
}


bool RenderOptions::getHalfYScale ()
{
	return b_halfYScale;
}


bool RenderOptions::getKeepFiles ()
{
	return b_keepFiles;
}


bool RenderOptions::getHaveClouds ()
{
	return b_haveClouds;
}


bool RenderOptions::getHaveFog ()
{
	return b_haveFog;
}


void RenderOptions::getSize (UI *width, UI *height)
{
	if (width)
		*width = d_width;
	if (height)
		*height = d_height;
}


UI RenderOptions::getLightingLevel ()
{
	return d_lightingLevel;
}


//UI RenderOptions::getPOVquality ()
//{
//	return d_POVquality;
//}


float RenderOptions::getSealevel ()
{
	return d_seaLevel;
}


float RenderOptions::getWaterClarity ()
{
	return d_waterClarity;
}


float RenderOptions::getTimeOfDay ()
{
	return d_timeOfDay;
}


float RenderOptions::getNorthDir ()
{
	return d_northDir;
}


void RenderOptions::getCameraLocation (float *x, float *y, float *z)
{
	if (x)
		*x = d_cameraX;
	if (y)
		*y = d_cameraY;
	if (z)
		*z = d_cameraZ;
}


void RenderOptions::getLookAtLocation (float *x, float *y, float *z)
{
	if (x)
		*x = d_lookAtX;
	if (y)
		*y = d_lookAtY;
	if (z)
		*z = d_lookAtZ;
}


char *RenderOptions::getPOVFile ()
{
	return p_POVfile;
}


char *RenderOptions::getHFFile ()
{
	return p_HFfile;
}


