/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "GuiBufferedDrawingArea.h"
#include "GlobalSanityCheck.h"
#include "GlobalTrace.h"


/*
 *  constructor: initialize everything, connect signals
 */
GuiBufferedDrawingArea::GuiBufferedDrawingArea () 
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ GuiBufferedDrawingArea\n");

	set_events (GDK_EXPOSURE_MASK
		| GDK_LEAVE_NOTIFY_MASK
		| GDK_BUTTON_PRESS_MASK
		| GDK_BUTTON_RELEASE_MASK
		| GDK_POINTER_MOTION_MASK
		| GDK_POINTER_MOTION_HINT_MASK);

	p_pixmap = NULL;
	p_GC=GTK_WIDGET (gtkobj())->style->black_gc;
	d_syncInterval = 1;
	d_noSyncCount = 0;
}


/*
 *  destructor: free any resources we're still holding
 */
GuiBufferedDrawingArea::~GuiBufferedDrawingArea ()
{
	gdk_pixmap_unref (p_pixmap);

	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- GuiBufferedDrawingArea\n");
}


/*
 *  clear: clear the entire drawing area 
 */
void GuiBufferedDrawingArea::clear (GdkColor *col)
{
	GdkRectangle 	r;

	r.x = 0;
	r.y = 0;
	r.width = width();
	r.height = height();

	if (!col)
    		gdk_draw_rectangle (p_pixmap, 
			GTK_WIDGET(gtkobj())->style->white_gc,
			TRUE, 0, 0, this->width(), this->height());
	else
		{
		setColors (col);
    		drawRectangle (0, 0, this->width(), this->height(), TRUE);
		}

	if (d_syncInterval > 0)
		sync (&r);
}


/*
 *  sync: copy the specified area from the offscreen pixmap to the window. 
 *  	  The area specified by rectangle r is added to the area which 
 * 	  has not been displayed since the last sync();
 */
void GuiBufferedDrawingArea::sync (GdkRectangle *update_rect)
{
#define SYNC_MAX(a, b)		(a >= b ? a : b)
#define SYNC_MIN(a, b)		(a <= b ? a : b)

	bool		didDraw = FALSE;
	unsigned int	minx, maxx, miny, maxy;
	GdkRectangle	r;

	if (!update_rect)
		{
		r.x = r.y = 0;
		r.width = this->width();
		r.height = this->height();
		draw (&r);

		d_noSyncCount = 0;
		d_sMaxx = d_sMaxy = 0;
		d_sMinx = r.width;
		d_sMiny = r.height;
		return;
		}
	else
		{
		r.x = SYNC_MAX (update_rect->x, 0);
		r.y = SYNC_MAX (update_rect->y, 0);
		r.width = SYNC_MAX (gint (update_rect->x+update_rect->width-r.x), 0);
		r.height = SYNC_MAX (gint (update_rect->y+update_rect->height-r.y), 0);
		}

	if (d_syncInterval == 1)
		{
		draw (&r);
		didDraw = TRUE;
		}
	else
	if (d_syncInterval > 1)
		{
		d_noSyncCount++;
		minx = SYNC_MIN (r.x, gint(r.x+r.width-1));
		maxx = SYNC_MAX (r.x, gint(r.x+r.width-1));
		miny = SYNC_MIN (r.y, gint(r.y+r.height-1));
		maxy = SYNC_MAX (r.y, gint(r.y+r.height-1));

		// figure out if new points expand repaint rect
		if (minx < d_sMinx)
			d_sMinx = minx;
		if (maxx > d_sMaxx)
			d_sMaxx = maxx;
		if (miny < d_sMiny)
			d_sMiny = miny;
		if (maxy > d_sMaxy)
			d_sMaxy = maxy;

		// time to sync
		if (d_noSyncCount == d_syncInterval)
			{
			r.x = d_sMinx;
			r.y = d_sMiny;
			r.width = (d_sMaxx-d_sMinx)+1;
			r.height = (d_sMaxy-d_sMiny)+1;
			draw (&r);
			didDraw = TRUE;

			d_noSyncCount = 0;
			d_sMaxx = d_sMaxy = 0;
			d_sMinx = this->width();
			d_sMiny = this->height();
			}
		}

	if (didDraw)
		{
		// This seemed a good idea but causes excessive 
		// flickering when redrawing in interactive mode. 
		// 
		//Gtk_Main *gtkMain = Gtk_Main::instance();
		//while (gtkMain->events_pending())
		//	gtkMain->iteration ();
		}
}


/*
 *  setSync: set the sync interval then sync the entire window 
 *	0 means that we don't sync
 *	>0 means that we sync after every n draw operations
 */
void GuiBufferedDrawingArea::setSync (int syncInterval)
{
	GdkRectangle	r;

	d_syncInterval = syncInterval;

	if (d_syncInterval < 0)			// force valid range
		d_syncInterval = 0;

	// sync entire window
	r.x = 0;
	r.y = 0;
	r.width = width();
	r.height = height();

	// reset sync window trackers
	d_sMaxx = d_sMaxy = 0;
	d_sMinx = r.width;
	d_sMiny = r.height;
	d_noSyncCount = 0;

	sync (&r);
}


/*
 *  setUsize: set the window's size and create a new offscreen pixmap
 */
void GuiBufferedDrawingArea::setUsize (int x, int y)
{
	if (x > width() || y > height())
		{
		if (p_pixmap)
			gdk_pixmap_unref(p_pixmap);
		p_pixmap = gdk_pixmap_new(get_window(), x, y, -1);
    		gdk_draw_rectangle (p_pixmap, 
			GTK_WIDGET(gtkobj())->style->white_gc,
			TRUE, 0, 0, x, y);
		this->set_usize (x, y);
		}

	this->show ();
}


/*
 *  setColors: set the fore- and background colors
 */
void GuiBufferedDrawingArea::setColors (GdkColor *fg, GdkColor *bg)
{
	if (fg)
		gdk_gc_set_foreground (GTK_WIDGET (gtkobj())->style->black_gc, fg);
	if (bg)
		gdk_gc_set_foreground (GTK_WIDGET (gtkobj())->style->black_gc, bg);
}


/*
 *  configure_event_impl: create a new backing pixmap of the appropriate size 
 */
int GuiBufferedDrawingArea::configure_event_impl (GdkEventConfigure *event)
{
    	//GtkWidget 	*widget = GTK_WIDGET(gtkobj());
	if (event);			// silence compiler warning 

	if (p_pixmap)
		gdk_pixmap_unref(p_pixmap);

	p_pixmap = gdk_pixmap_new(get_window(),  width(), height(), -1);
    	gdk_draw_rectangle (p_pixmap, 
		GTK_WIDGET(gtkobj())->style->white_gc,
		TRUE, 0, 0, width(), height());

	return TRUE;
}


/*
 *  expose_event_impl: redraw the exposed area from the backing pixmap
 */
int GuiBufferedDrawingArea::expose_event_impl (GdkEventExpose *event)
{
	gdk_draw_pixmap(get_window(),
	    GTK_WIDGET (gtkobj())->style->fg_gc[GTK_WIDGET_STATE (GTK_WIDGET(gtkobj()))],
	    p_pixmap,
	    event->area.x, event->area.y,
	    event->area.x, event->area.y,
	    event->area.width, event->area.height);
	return FALSE;
}


/*
 *  drawPoint: draw a Point on the screen
 */
void GuiBufferedDrawingArea::drawPoint (int x, int y)
{
	GdkRectangle 	r;

	r.x = x;
	r.y = y;
	r.width = 1;
	r.height = 1;
	gdk_draw_point (p_pixmap, 
		GTK_WIDGET (gtkobj())->style->black_gc, r.x, r.y);

	if (d_syncInterval > 0)
		sync (&r);
}


/*
 *  drawLine: draw a line on the screen
 */
void GuiBufferedDrawingArea::drawLine (int x1, int y1, int x2, int y2)
{
	GdkRectangle 	r;

	r.x = (x1<x2 ? x1 : x2);
	r.y = (y1<y2 ? y1 : y2);
	r.width = (x1>x2 ? x1 : x2)-r.x+1;
	r.height = (y1>y2 ? y1 : y2)-r.y+1;
	// gdk_draw_line (GTK_WIDGET (this), 
	gdk_draw_line (p_pixmap, 
		GTK_WIDGET (gtkobj())->style->black_gc, x1, y1, x2, y2);

	if (d_syncInterval > 0)
		sync (&r);
}


/*
 *  drawRectangle: draw a rectangle on the screen
 */
void GuiBufferedDrawingArea::drawRectangle (int x1, int y1, 
					    int width, int height, 
					    bool filled)
{
	GdkRectangle 	r;

	r.x = x1;
	r.y = y1;
	r.width = width;
	r.height = height;
	gdk_draw_rectangle (p_pixmap,
		GTK_WIDGET (gtkobj())->style->black_gc,
		filled, r.x, r.y, r.width, r.height);

	if (d_syncInterval > 0)
		sync (&r);
}


/*
 *  drawPolygon: draw a polygon on the screen
 */
void GuiBufferedDrawingArea::drawPolygon (GdkPoint *points, gint npoints, 
					  bool filled)
{
	GdkRectangle 	r;
	GdkPoint	*p;
	int		sx, sy, bx, by;		// smallest & biggest x/y values

	sx = bx = points[0].x;
	sy = by = points[0].y;

	for (int i=1; i<npoints; i++)
		{
		p = &(points[i]);

		// find max and min x points
		if (p->x < sx)
			sx = p->x;
		else
		if (p->x > bx)
			bx = p->x;

		// find max and min y points
		if (p->y < sy)
			sy = p->y;
		else
		if (p->y > by)
			by = p->y;
		}

	r.x = sx;
	r.y = sy;
	r.width = bx-sx+1;
	r.height = by-sy+1;
	gdk_draw_polygon (p_pixmap,
		GTK_WIDGET (gtkobj())->style->black_gc,
		filled, points, npoints);

	if (d_syncInterval > 0)
		sync (&r);
}


/*
 *  drawPolygon: draw a filled and then a non-filled polygon
 */
void GuiBufferedDrawingArea::drawBorderedPolygon (GdkPoint *points, int npoints, 
				GdkColor *borderColor, GdkColor *polygonColor)
{
	setColors (polygonColor, borderColor);
	drawPolygon (points, npoints, TRUE);
	setColors (borderColor, polygonColor);
	drawPolygon (points, npoints, FALSE);
}


/*
 *  drawText: draw the specified text to the screen
 */
void GuiBufferedDrawingArea::drawText (int x1, int y1, 
					char *textData, GdkFont *textFont)
{
	GdkRectangle 	r;

	r.x = x1;
	r.y = y1;
	r.width = x1 + gdk_string_width (textFont, textData);
	r.height = y1+12;
	gdk_draw_string (p_pixmap, textFont, 
		GTK_WIDGET (gtkobj())->style->black_gc, r.x, r.y, textData);

	if (d_syncInterval > 0)
		sync (&r);
}


/*
 *  drawPixmap: draw the specified pixmap to the screen 
 */
void GuiBufferedDrawingArea::drawPixmap (GdkPixmap *pixmap, 
					 int xsrc, int ysrc, 
					 int xdest, int ydest, 
					 int width, int height)
{
	GdkRectangle 	r;

	r.x = xdest;
	r.y = ydest;
	r.width = width;
	r.height = height;
	gdk_draw_pixmap (p_pixmap, GTK_WIDGET (gtkobj())->style->black_gc, 
			pixmap, xsrc, ysrc, xdest, ydest, width, height);
}


/*
 *  drawRGB: draw a byte stream using GdkRGB
 * 	Important: if you wish to use the RGB interface, you must 
 * 	call gdk_rgb_init() after gtk_init(). If you don't do this, 
 * 	the attempt to draw (below) will dump core. 
 */
void GuiBufferedDrawingArea::drawRGB (int x, int y, int width, int height, 
				guchar *buf, gint rowstride)
{
	GdkRectangle 	r;

	SanityCheck::bailout ((!buf), "buf==NULL", "GuiBufferedDrawingArea::drawRGB");

	r.x = x;
	r.y = y;
	r.width = width;
	r.height = height;
	gdk_draw_rgb_image (p_pixmap, GTK_WIDGET (gtkobj())->style->black_gc,
		x, y, width, height, GDK_RGB_DITHER_NONE, buf, rowstride);

	if (d_syncInterval > 0)
		sync (&r);
}



/*
 *  button_press_event_impl: process a button press
 */
//int GuiBufferedDrawingArea::button_press_event_impl (GdkEventButton *event)
//{
//	return TRUE;
//}


/*
 *  motion_notify_event_impl: process a moving mouse 
 */
//int GuiBufferedDrawingArea::motion_notify_event_impl (GdkEventMotion *event)
//{
//	return TRUE;
//}


