/***************************************************************************
                          rfileparser.cpp  -  description
                             -------------------
    begin                : Sun Sep 12 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "rfileparser.h"

#include "rprgdef.h"

#include <stdio.h>

#include <qfile.h>
#include <qfileinfo.h>
#include <qtextstream.h>


/**
 * \class RFileParser
 * Class RFileParser offers some useful methode for parsing MAS-files.
 * A section is defined as a Header '[section]' and a body '{ ... }'.
 * Section Headers are not case sensitive.
 */


/**
 * Constructor.
 */
RFileParser::RFileParser()
{
}


/**
 * Destructor.
 */
RFileParser::~RFileParser()
{
}


/**
 * Gets the contents of a file.
 *
 * \param fileName File name.
 * \param comments Include comments?
 * \param sections Include sections in form [section] { } ?
 * \param whiteSpace Include whitespace (false means that white space gets simplified)
 * \param htmlComments Include HTML comments (<!-- -->)?
 */
QString
RFileParser::getContents(QString fileName, bool comments, bool sections, bool whiteSpace, bool htmlComments)
{
  QFileInfo fi(fileName);
  QFile f(fi.absFilePath());
  QString s="";                   // Buffer for the whole file

  if(f.open(IO_ReadOnly)) {       // file opened successfully
    QTextStream t(&f);            // use a text stream

    s = t.read();                 // The whole file in a string
    f.close();

    if(!comments)     s = RFileParser::removeComments(s);
    if(!sections)     s = RFileParser::removeSections(s);
    if(!whiteSpace)   s = s.simplifyWhiteSpace();
    if(!htmlComments) s = RFileParser::removeHtmlComments(s);
  }
  else {
    // Can't open file
  }

  return s;
}


/**
 * Gets the body of a section from a string.
 *   Comment lines starting with '//' get removed.
 *
 * \param s The string.
 * \param sectionName Name of the section (e.g. '[styles]'). This methode ignores case for section names.
 */
QString
RFileParser::getSection(QString s, QString sectionName)
{
  QString result="";

  QChar ch;                     // A single byte of the file
  int bracketCounter=1;         // Bracket counter (increase on '{' and decrese on '}'.
  int i=0;                      // Current index
  int l=0;                      // Length of current part we must have

  if((i=s.find(sectionName, 0, false)) >= 0 &&       // Jump to section [styles]
     (i=s.find('{', i))                >= 0    ) {

    ++i;

    while(i+l<(int)s.length()) {
      ch = s.at(i+l);           // Single byte

      ++l;

      if(ch=='{') ++bracketCounter;
      if(ch=='}') --bracketCounter;

      if(bracketCounter==0) break;
    }

    result = s.mid(i, l-1);
  }

  return result;
}



/**
 * Gets the HTML comments out of a file.
 *
 * \param
 */
QString
RFileParser::getHtmlComment(QString s)
{
  QString result="";
  int length=0;            // length of the sub string
  int index=0;

  if((index =s.find("<!--", index))>=0 &&
     (index+=4) &&
     (length=s.find("-->",  index)-index)!=0) {

    result=s.mid(index, length-3);
  }

  return result;
}



/**
 * Gets the next String between two given characters. The index-parameter gets moved to the character after
 *   the stopper or to -1 if the starter / stopper were not found.
 *
 * \param s  The original string to search through.
 * \param index The index where the search starts.
 * \param starter The starter character.
 * \param stopper The stopper character.
 * \return The next string between the two characters excluding the characters themselves or an empty string.
 */
QString
RFileParser::getNextStringBetween(QString s, int& index, QChar starter, QChar stopper)
{
  QString result="";
  int length=0;            // length of the sub string

  if((index =s.find(starter, index))>=0 &&
     (length=s.find(stopper, index+1)-index)!=0) {

    result=s.mid(index+1, length-1);
    index+=2;
  }

  return result;
}


/**
 * Gets the next String between the given index and a given stopper character. The index-parameter gets moved to
 *   the character after the stopper or to -1 if the stopper was not found.
 *
 * \param s  The original string to search through.
 * \param index The index where the search starts.
 * \param stopper The stopper character.
 * \return The next string between the given index and the stopper character excluding the stopper character or an empty string.
 */
QString
RFileParser::getNextStringUntil(QString s, int& index, QChar stopper)
{
  QString result="";
  int length=0;            // length of the sub string

  if((length=s.find(stopper, index)-index)!=0) {
    result=s.mid(index, length);
    index++;
  }

  return result;
}


/**
 * Removes all comments (between '/ *' and '* /').
 */
QString
RFileParser::removeComments(QString s)
{
  QString result="";
  int i1=0, i2=0;

  while((i2=s.find("/*", i1))>=0) {
    result += s.mid(i1, i2-i1);

    i1+=2;
    i1=s.find("*/", i1);
    i1+=2;
  }

  result += s.mid(i1, s.length()-i1);

  return result;
}



/**
 * Removes all HTML comments (between '<!--' and '-->').
 */
QString
RFileParser::removeHtmlComments(QString s)
{
  QString result="";
  int i1=0, i2=0;

  while((i2=s.find("<!--", i1))>=0) {
    result += s.mid(i1, i2-i1);

    i1+=4;
    i1=s.find("-->", i1);
    i1+=3;
  }

  result += s.mid(i1, s.length()-i1);

  return result;
}



/**
 * Removes all sections ('[section] { }').
 */
QString
RFileParser::removeSections(QString s)
{
  QString result="";

  QChar ch;                     // A single byte of the file
  int bracketCounter;           // Bracket counter (increase on '{' and decrese on '}'.
  int i=0;                      // Current index

  while(i<(int)s.length()) {
    ch = s.at(i);           // Single byte

    if(ch=='[') {
      bracketCounter=1;
      while(i<(int)s.length() && ch!=']') { ch = s.at(i); ++i; }
      ++i;
      while(i<(int)s.length() && ch!='{') { ch = s.at(i); ++i; }
      ++i;

      while(i<(int)s.length() && bracketCounter!=0) {
        ch = s.at(i);
        if(ch=='{') ++bracketCounter;
        if(ch=='}') --bracketCounter;
        ++i;
      }
      ++i;
    }
    else {
      result+=ch;
    }

    ++i;
  }

  return result;
}


/**
 * Format plain text into HTML-code with a given maximal width.
 *   Spaces get replaced with non breaking spaces. Tabulators get filled up
 *   with non breaking spaces.
 *
 * \param s The original plain text string
 */
QString
RFileParser::plainTextToHtml(QString s, int autoBreak)
{
  QString result="\n";

  if(!s.isEmpty()) {
    int col=1, i;

    for(i=0; i<(int)s.length(); ++i) {
      // Line feed:
      //
      if(s[i]=='\n') {
        result+="<BR>\n";
        col=1;
      }

      // Auto break:
      //
      else if(col==autoBreak && autoBreak!=0) {
        result+="<BR>\n";
        result+=s[i];
        col=1;
      }

      // Tab:
      //
      else if(s[i]=='\t') {
        while(col%8!=0) { result+="&nbsp;"; ++col; }
        result+="&nbsp;";
        ++col;
      }

      // Space:
      //
      else if(s[i]==' ') {
        result+="&nbsp;";
        ++col;
      }

      // Normal char / special code:
      //
      else {
        if(s[i].isLetter() || s[i].isNumber()) {
          result+=s[i];
        }
        else {
          result+=charToHtml(s[i]);
        }
        ++col;
      }
    }

    result+="\n";
  }

  return result;
}


/**
 * Converts a special character to html code (e.g.: '' to "&#187;")
 */
QString
RFileParser::charToHtml(QChar c)
{
  QString s;
  QString uc;
  uc.setNum(c.unicode());
  s = "&#" + uc + ";";
  return s;
}


// EOF
