/*
 * main.c - v1.7 - 28 Mar 1999
 * (c) Stevie Strickland, 1999
 *
 * This program has been placed under the GPL.  Any bugfixes or enhancements
 * will be greatly appreciated :)
 *
 * Stevie Strickland - sstrickl@resnet.gatech.edu
 */

#include "rolldice.h"

// The version number :)
static const int MAJOR_VERSION = 1;
static const int MINOR_VERSION = 7;

// Stores the random number file to use
static int rand_file;

// Should we print out the separate results of the dice rolls or not?
static int print_separate;

/*
 * print_usage() - Prints the usage for rolldice
 *
 * Parameters: none
 * Returns: none
 */
static void print_usage() {
    fprintf(stderr, "Usage: rolldice [options] <dice string>\n");
    fprintf(stderr, "For dice string format and options, ");
    fprintf(stderr, "see rolldice(6).\n");
    exit(EXIT_FAILURE);
}

/*
 * print_version() - Prints the version number
 *
 * Parameters: none
 * Returns: none
 */
static void print_version() {
    printf("rolldice, v%d.%d\n", MAJOR_VERSION, MINOR_VERSION);
    printf("Written by Stevie Strickland (sstrickl@resnet.gatech.edu)\n");
    exit(EXIT_SUCCESS);
}

/*
 * print_opt_error() - Prints out the error for an unknown option
 *
 * Parameters: option - unknown option to print out
 * Returns: none
 */
static void print_opt_error(char *option) {
    fprintf(stderr, "%s: Unknown option\n", option);
    print_usage();
}

/*
 * parse_options() - Parse the command-line options
 *
 * Parameters: String to parse for options
 * Returns: none
 */
static void parse_options(char *option) {

    option++;

    if(*option == '-') {
	option++;
	if((strcmp(option, "version")) == 0) print_version();
	else if((strcmp(option, "help")) == 0) print_usage();
	else if((strcmp(option, "urandom")) == 0) {
	    rand_file = URANDOM;
	}
	else if((strcmp(option, "random")) == 0) {
	    rand_file = RANDOM;
	}
	else if((strcmp(option, "separate")) == 0) print_separate = TRUE_VAL;
	else print_opt_error(option);
    }
    else {
	while((strcmp(option, "")) != 0) {
	    switch(*option) {	
	    case 'h': print_usage();
	    case 'v': print_version();	
	    case 'u': rand_file = URANDOM; break;
	    case 'r': rand_file = RANDOM; break;
	    case 's': print_separate = TRUE_VAL; break;
	    default: print_opt_error(option);
	    }
	    option++;
	}
    }
}

/* print_rolls() - Prints the rolls, either just the totals or the
 *                 separate rolls, also.
 * 
 * Parameters: Dice string with which to calculate dice rolls
 * Returns: None
 */
void print_rolls(int *dice_nums) {
    int i, j, k, temp_int, temp_index, temp_total;
    int temp_roll[dice_nums[NUM_DICE]];

    for(i = 0; i < dice_nums[NUM_ROLLS]; i++) {
	temp_total = 0;
	if(print_separate) printf("Roll #%d: (", i+1);
	for(j = 0; j < dice_nums[NUM_DICE]; j++) {
	    temp_roll[j] = rolldie(dice_nums[NUM_SIDES]);
	    if(print_separate) printf("%d ", temp_roll[j]);
	    temp_total += temp_roll[j];
	}
	for(j = 0; j < dice_nums[NUM_DROP]; j++) {
	    temp_int = MAXSHORT;
	    for(k = 0; k < dice_nums[NUM_DICE]; k++)
		if(temp_int > temp_roll[k]) {
		    temp_int = temp_roll[k];
		    temp_index = k;
		}
	    if(print_separate) printf("- %d ", temp_int);
	    temp_total -= temp_int;
	    temp_roll[temp_index] = MAXSHORT;
	}
	if(print_separate) printf(") ");
	if(dice_nums[MULTIPLIER] != 1) {
	    if(print_separate) printf("* %d ", dice_nums[MULTIPLIER]);
	    temp_total *= dice_nums[MULTIPLIER];
	}
	if(dice_nums[MODIFIER]) {
	    if(print_separate) printf("+ %d ", dice_nums[MODIFIER]);
	    temp_total += dice_nums[MODIFIER];
	}
	if(print_separate) printf("= ");
	printf("%d ", temp_total);
	if(print_separate) printf("\n");
    }
    if(!print_separate) printf("\n");
}

int main(int argc, char **argv) {
  
    int index, *totals, *dice_nums;
    
    dice_nums = NULL;
    
    init_random(rand_file);
    
    if ( argc < 2 ) {
	print_usage();
    } else {
	for ( index = 1; index < argc; index++ ) {
	    if ( *argv[index] == '-' ) parse_options(argv[index]);
	    else {
		dice_nums = parse_string( argv[index] );
		print_rolls(dice_nums);
		free(dice_nums);
	    }
	}
    }
	
    if ( dice_nums == NULL ) {
	fprintf(stderr, "%s: Failure in getting dice attributes\n", argv[0]);
	return 1;
    }

    
    return EXIT_SUCCESS;
}
