/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 by the Free Software Foundation.               *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*  Parts of this code were given to us by Tommy Thorn */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <assert.h>
#include "ttsgf.h"
#include "sgf_properties.h"
#include "liberty.h"


extern int board_size;
SGFNodeP lastnode=0; /*last node where move was placed*/

/* Utility: a checking, initializing malloc */

void *xalloc(unsigned int size)
{
  void *pt = malloc(size);

  if (!pt) {
    fprintf(stderr, "%s: Out of memory!\n",__FUNCTION__);
    exit(2);
  }
  /*MemSet(p, (unsigned long) size, 0); */
  memset(pt, 0, (unsigned long) size);
  return pt;
}

/*allocate memory for a new node*/
SGFNodeP sgfNewNode()
{
  SGFNodeP res = xalloc(sizeof(*res));
  return res;
}

void
sgfSetLastNode(SGFNodeP last_node)
{
    lastnode=last_node;
}

/*add a text property to the gametree*/
void sgfAddProperty(SGFNodeP n, const char *name, const char *value)
{
    SGFPropertyP pr = n->prop;
  if (pr)
    while (pr->next)
      pr=pr->next;
  sgfMkProperty(name, value, n, pr);
}

/*add a integer property to the gametree*/
void sgfAddPropertyInt(SGFNodeP n, const char *name, long v)
{
  char buffer[10];

/*  StrIToA(buffer, v);
    #define StrIToA(buf, i) sprintf(buf, "%d", i)*/

  snprintf(buffer,10, "%ld", v);
  sgfAddProperty(n, name, buffer);
}

/*add a float property to the gametree*/
void sgfAddPropertyFloat(SGFNodeP n, const char *name, float v)
{
  char buffer[10];

/*  StrIToA(buffer, v);
    #define StrIToA(buf, i) sprintf(buf, "%d", i)*/

  snprintf(buffer,10, "%3.1f", v);
  sgfAddProperty(n, name, buffer);
}

/*read a property as int from the gametree*/
int sgfGetIntProperty(SGFNodeP n, const unsigned char *name, int *value)
{
  SGFPropertyP pr;
  short nam = name[0] | name[1] << 8;

  for (pr = n->prop; pr; pr=pr->next)
    if (pr->name == nam) {
      *value = atoi(pr->value);
      return 1;
    }
  return 0;
}

/*read a property as float from the gametree*/
int sgfGetFloatProperty(SGFNodeP n, const unsigned char *name, float *value)
{
  SGFPropertyP pr;
  short nam = name[0] | name[1] << 8;

  for (pr = n->prop; pr; pr=pr->next)
    if (pr->name == nam) {
      *value = atof(pr->value);
      return 1;
    }
  return 0;
}

/*read a property as text from the gametree*/
int sgfGetCharProperty(SGFNodeP n, const unsigned char *name, char **value)
{
  SGFPropertyP pr;
  short nam = name[0] | name[1] << 8;

  for (pr = n->prop; pr; pr=pr->next)
    if (pr->name == nam) {
      *value=pr->value;
      return 1;
    }
  return 0;
}

/* overwrite a property in the gametree with text
   or create a new one if it does not exist */
void sgfOverwriteProperty(SGFNodeP n, const unsigned char *name, const char *text)
{
  SGFPropertyP pr;
  short nam = name[0] | name[1] << 8;

  for (pr = n->prop; pr; pr=pr->next)
   if (pr->name == nam)
   {
      strncpy(pr->value,text,strlen(pr->value)); /*FIXME realloc pr with strlen text*/
      return;
   }
  sgfAddProperty(n,name,text);
}

/* overwrite a float property in the gametree with v
   or create a new one if it does not exist */
void sgfOverwritePropertyFloat(SGFNodeP n, const unsigned char *name, float v)
{
  SGFPropertyP pr;
  short nam = name[0] | name[1] << 8;

  for (pr = n->prop; pr; pr=pr->next)
   if (pr->name == nam)
   {
      snprintf(pr->value,strlen(pr->value)+1,"%3.1f",v);    /*FIXME realloc pr*/
      return;
   }
  sgfAddPropertyFloat(n,name,v);
}

/* overwrite a int property in the gametree with v
   or create a new one if it does not exist */
void sgfOverwritePropertyInt(SGFNodeP n, const unsigned char *name, int v)
{
  SGFPropertyP pr;
  short nam = name[0] | name[1] << 8;

  for (pr = n->prop; pr; pr=pr->next)
   if (pr->name == nam)
   {
      snprintf(pr->value,strlen(pr->value)+1,"%d",v);    /*FIXME realloc pr*/
      return;
   }
  sgfAddPropertyFloat(n,name,v);
}

/* make a property */
/*FIXME to speed up, we should have MkMoveProperty*/
SGFPropertyP sgfMkProperty(const unsigned char *name,const  unsigned char *value,
			   SGFNodeP n,
			   SGFPropertyP last)
{
  if (strlen(name) == 1) 
  {
    SGFPropertyP pr = xalloc(sizeof(*pr)+strlen(value));
    pr->name = name[0] | ' ' << 8;
    strcpy(pr->value, value);

    if (last == NULL)
      n->prop = pr;
    else
      last->next = pr;
    return pr;
  }
  else 
  {
    SGFPropertyP pr = xalloc(sizeof(*pr)+strlen(value));
    pr->name = name[0] | name[1] << 8;
    strcpy(pr->value, value);

    if (last == NULL)
      n->prop = pr;
    else
      last->next = pr;
    return pr;
  }
}


/*goto previous node*/
SGFNodeP
sgfPrev(SGFNodeP pr)
{
  if (pr->parent) {
    SGFNodeP q = pr->parent->child, prev = NULL;
    while (q && q != pr) {
      prev = q;
      q = q->next;
    }
    return prev;
  } else return NULL;
}

/*goto root node*/
SGFNodeP
sgfRoot(SGFNodeP pr)
{
  while (pr->parent)
    pr = pr->parent;
  return pr;
}


/*add a stone to the current or the given node
 return the node where the stone was added */
SGFNodeP
sgfAddStone(SGFNodeP pr, int color, int movex,int movey)
{
  SGFNodeP node=sgfNodeCheck(pr);
  char move[3];
  sprintf(move,"%c%c",movey+'a',movex+'a');
  sgfAddProperty(node,(color==BLACK) ? "AB":"AW",move);
  return node;
}

/*add a move to the gametree*/
SGFNodeP
sgfAddPlay(SGFNodeP pr, int who, int movex,int movey)
{
  char move[3];
    
  SGFNodeP new;
  SGFNodeP last= sgfNodeCheck(pr);
  /* if it is a pass move */
  if ((movex==movey) && (movey==board_size))
     move[0]=0;
  else
    sprintf(move,"%c%c",movey+'a',movex+'a');
  if(last->child)
  {
    lastnode=sgfStartVariantFirst(last->child);
    sgfAddProperty(lastnode,(who==BLACK) ? "B":"W",move);
    return lastnode;
  }else
  {
    new = sgfNewNode();
    last->child=new;
    new->parent=last;
    lastnode=new;
    sgfAddProperty(new,(who==BLACK) ? "B":"W",move);
    return new;
  }
}

SGFNodeP sgfCreateHeaderNode(float komi)
{
    SGFNodeP root=sgfNewNode();
    sgfAddPropertyInt(root,"SZ",board_size);
    sgfAddPropertyFloat(root,"KM",komi);
    sgf_root=root;
    return root;
}

/*add a comment to the gametree*/
SGFNodeP
sgfAddComment(SGFNodeP pr, const char *comment)
{
  SGFNodeP ret;
  assert(sgf_root);
  ret=sgfNodeCheck(pr);
  sgfAddProperty(ret,"C ",comment);
  return ret;
}

/*returns the node to modify
 when p==0 then lastnode is used, except lastnode is NULL, then the
 current end of game is used*/
SGFNodeP
sgfNodeCheck(SGFNodeP pr)
{
   SGFNodeP last=pr;
    assert(sgf_root);
   if(!last)
   {
       if(lastnode)
           last=lastnode;
       else
       {
           last=sgf_root;
           if(last->child)
           {
               for (; last->child; last = last->child);
           }
       }
   }
   return last;
}


/*place text on the board at position i,j*/
SGFNodeP
sgfBoardText(SGFNodeP pr,int i,int j, const char *text)
{
  void *str = malloc(strlen(text)+3);
  SGFNodeP ret;
  assert(sgf_root);
  ret=sgfNodeCheck(pr);
  if (!str)
    perror("sgfBoardText: Out of memory!\n");
  else
  {
      sprintf(str,"%c%c:%s",j+'a',i+'a',text);
      sgfAddProperty(ret,"LB",str);
      free(str);
  }
  return ret;
}


/*place a charcter on the board at position i,j*/
SGFNodeP
sgfBoardChar(SGFNodeP pr,int i,int j, unsigned char c)
{
  char text[2]="";
  text[0]=c;
  text[1]=0;
  return sgfBoardText(pr,i,j,text);
}

/*place a number on the board at position i,j*/
SGFNodeP 
sgfBoardNumber(SGFNodeP pr,int i,int j, int number)
{
    char text[10];
    SGFNodeP ret=sgfNodeCheck(pr);
    snprintf(text,10,"%c%c:%i",j+'a',i+'a',number);
    sgfAddProperty(ret,"LB",text);
    return ret;
}

/*place a circle mark on the board at position i,j*/
SGFNodeP 
sgfTriangle(SGFNodeP pr,int i,int j)
{
    SGFNodeP ret=sgfNodeCheck(pr);
    char text[3];
    snprintf(text,3,"%c%c",j+'a',i+'a');
    sgfAddProperty(ret,"TR",text);
    return ret;
}

/*place a circle mark on the board at position i,j*/
SGFNodeP 
sgfCircle(SGFNodeP pr,int i,int j)
{
    SGFNodeP ret=sgfNodeCheck(pr);
    char text[3];
    snprintf(text,3,"%c%c",j+'a',i+'a');
    sgfAddProperty(ret,"CR",text);
    return ret;
}

/*place a square mark on the board at position i,j*/
SGFNodeP
sgfSquare(SGFNodeP pr,int i,int j)
{
    return sgfMark(pr,i,j);   /*cgoban 1.9.5 does not understand SQ*/
}

/*place a (square) mark on the board at position i,j*/
SGFNodeP
sgfMark(SGFNodeP pr,int i,int j)
{
    char text[3];
    SGFNodeP ret=sgfNodeCheck(pr);
    snprintf(text,3,"%c%c",j+'a',i+'a');
    sgfAddProperty(ret,"MA",text);
    return ret;
}


/*start a new variant, returns a pointer to the new node*/
SGFNodeP
sgfStartVariant(SGFNodeP pr)
{
    SGFNodeP last=sgfNodeCheck(pr);
    for (; last->next; last = last->next);
    last->next=sgfNewNode();
    last->next->parent=last;
    return last->next;
}

/*start a new variant as first child, returns a pointer to the new node*/
SGFNodeP
sgfStartVariantFirst(SGFNodeP pr)
{
    SGFNodeP old_first_child=sgfNodeCheck(pr);
    SGFNodeP new_first_child=sgfNewNode();
    new_first_child->next=old_first_child;
    if(old_first_child->parent)
      new_first_child->parent=old_first_child->parent;
      old_first_child->parent=new_first_child;
    if(new_first_child->parent)
      new_first_child->parent->child=new_first_child;
    return new_first_child;
}

/*
void
sgfFreeTree(SGFNodeP p)
{
}
*/


/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */

