/************************************************************************/
/*  Description of a bitmap.						*/
/************************************************************************/

#   ifndef	BITMAP_H
#   define	BITMAP_H

#   include	<stdio.h>
#   include	<sioGeneral.h>

/************************************************************************/
/*  Color encodings.							*/
/************************************************************************/
#   define	BMcoBLACKWHITE		0
#   define	BMcoWHITEBLACK		1
#   define	BMcoRGB			2
#   define	BMcoRGB8PALETTE		3

#   define	BMcoILLEGALVALUE	4

extern const char *	bmcoIntToString( int colorEncodingInt );
extern int		bmcoStringToInt( const char * colorEncodingString );
extern const char *	bmcoStrings[];

/************************************************************************/
/*  Units of measurement.						*/
/************************************************************************/
#   define	BMunCM			0
#   define	BMunINCH		1
#   define	BMunPOINT		2

#   define	BMunILLEGALVALUE	3

#   define	POINTS_PER_CM		28.3465

extern const char *	bmunIntToString( int unitInt );
extern int		bmunStringToInt( const char * unitString );
extern const char *	bmunStrings[];

/************************************************************************/
/*  An RGB8 Color.							*/
/*  The most usual way to display images on a computer screen.		*/
/************************************************************************/
typedef struct	RGB8Color
    {
    unsigned char	rgb8Red;
    unsigned char	rgb8Green;
    unsigned char	rgb8Blue;
    } RGB8Color;

/************************************************************************/
/*  Description of a bitmap.						*/
/************************************************************************/
typedef struct	BitmapDescription
    {
					/********************************/
					/*  Number of bytes in an image	*/
					/*  buffer; Idem for one row.	*/
					/*  Rows are never divided over	*/
					/*  more than one byte.		*/
					/********************************/
    unsigned int	bdBufferLength;
    unsigned int	bdBytesPerRow;
					/********************************/
					/*  Height/width of an image	*/
					/********************************/
    unsigned int	bdPixelsWide;
    unsigned int	bdPixelsHigh;
					/********************************/
					/*  Description of how pixels	*/
					/*  are encoded.		*/
					/*  For direct images, all three*/
					/*  refer to the buffer.	*/
					/*  For palette type images,	*/
					/*  BitsPerSample,		*/
					/*  SamplesPerPixel refer to the*/
					/*  image; BitsPerPixel refers	*/
					/*  to the buffer.		*/
					/********************************/
    int			bdBitsPerSample;
    int			bdSamplesPerPixel;
    int			bdBitsPerPixel;
					/********************************/
					/*  Pixels per metric unit;	*/
					/*  The unit.			*/
					/********************************/
    int			bdXResolution;
    int			bdYResolution;
    unsigned char	bdUnit;
					/********************************/
					/*  How are colors encoded.	*/
					/********************************/
    unsigned char	bdColorEncoding;
					/********************************/
					/*  Does it have a transparency	*/
					/*  mask?			*/
					/********************************/
    unsigned char	bdHasAlpha;
					/********************************/
					/*  Palette.			*/
					/*  ColorCount is only relevant	*/
					/*  with a palette.		*/
					/********************************/
    unsigned int	bdColorCount;
    union
	{
	RGB8Color *	bdpRGB8Palette;
	} bdPaletteUninterpreted;

#   define	bdRGB8Palette	bdPaletteUninterpreted.bdpRGB8Palette

    } BitmapDescription;

/************************************************************************/
/*  Description of a bitmap file format.				*/
/************************************************************************/
typedef struct	BitmapFileType
    {
    int (*bftWrite)(	const char *			filename,
			const unsigned char *		buffer,
			const BitmapDescription *	bd,
			int				privateFormat,
			double				compressionFactor );

    int (*bftCanWrite)( const BitmapDescription *	bd,
			int				privateFormat,
			double				compressionFactor );

    int (*bftRead)(	const char *		filename,
			unsigned char **	pBuffer,
			BitmapDescription *	bd,
			int *			pPrivateFormat,
			double *		pCompressionFactor	);

    char *	bftFileExtension;
    char *	bftFileFilter;
    char *	bftTypeId;
    char *	bftTypeDescription;
    } BitmapFileType;

typedef struct	BitmapFileFormat
    {
    char *		bffDescription;
    char *		bffId;
    int			bffPrivate;
    BitmapFileType *	bffFileType;
    } BitmapFileFormat;

/************************************************************************/
/*  Description of selection from a bitmap.				*/
/************************************************************************/
typedef struct	BitmapSelection
    {
    int		bsX0;
    int		bsY0;
    int		bsPixelsWide;
    int		bsPixelsHigh;
    } BitmapSelection;

/************************************************************************/
/*  Catalog of available of a bitmap file formats.			*/
/************************************************************************/
extern BitmapFileFormat	bmFileFormats[];
extern int	bmNumberOfFileFormats;

extern BitmapFileType *	bmFileTypes[];
extern int	bmNumberOfFileTypes;

/************************************************************************/
/*  For Reading/Writing from/to streams.				*/
/************************************************************************/
extern int bmWriteGifFile(	const char *			filename,
				const unsigned char *		buffer,
				const BitmapDescription *	bd,
				int				privateFormat,
				double				factor );

extern int bmCanWriteGifFile(	const BitmapDescription *	bd,
				int				privateFormat,
				double				factor );

extern int bmWriteJpegFile(	const char *			filename,
				const unsigned char *		buffer,
				const BitmapDescription *	bd,
				int				privateFormat,
				double				factor );

extern int bmCanWriteJpegFile(	const BitmapDescription *	bd,
				int				privateFormat,
				double				factor );

/************************************************************************/
/*  Write part of a bitmap to postscript.				*/
/************************************************************************/

extern int bmPsPrintBitmap(	FILE *				f,
				int				level,
				double				xscale,
				double				yscale,
				int				ox,
				int				oy,
				int				x0,
				int				y0,
				int				wide,
				int				high,
				const BitmapDescription *	bd,
				const unsigned char *		buffer	);

extern int bmPsPrintBitmapData(	FILE *				f,
				int				level,
				double				xscale,
				double				yscale,
				int				ox,
				int				oy,
				int				x0,
				int				y0,
				int				wide,
				int				high,
				const BitmapDescription *	bd,
				const unsigned char *		buffer	);

extern int bmPngWritePng(	const BitmapDescription *	bd,
				const unsigned char *		buffer,
				SimpleOutputStream *		sos );

extern int bmXvWritePaste(	const BitmapDescription *	bd,
				const unsigned char *		buffer,
				SimpleOutputStream *		sos );

extern int bmBmpReadDib(	BitmapDescription *	bd,
				unsigned char **	pBuffer,
				SimpleInputStream *	sis );

extern int bmJpegReadJfif(	BitmapDescription *	bd,
				unsigned char **	pBuffer,
				SimpleInputStream *	sis );

extern int bmPngReadPng(	BitmapDescription *	bd,
				unsigned char **	pBuffer,
				SimpleInputStream *	sis );

extern int bmXvReadPaste(	BitmapDescription *	bd,
				unsigned char **	pBuffer,
				SimpleInputStream *	sis );

extern int bmBmpSaveDib(	const BitmapDescription *	bd,
				const unsigned char *		buffer,
				int				bytesWritten,
				void *				voidsos );

/************************************************************************/
/*  Routines.								*/
/************************************************************************/

extern void bmInitDescription(	BitmapDescription *	bd	);

extern void bmCleanDescription(	BitmapDescription *	bd	);

extern int bmCopyDescription(	BitmapDescription *		bdOut,
				const BitmapDescription *	bdIn	);

extern int bmWrite(	const char *		filename,
			const unsigned char *	buffer,
			BitmapDescription *	bd,
			int			fileFormat,
			double			compressionFactor	);

extern int bmCanWrite(	BitmapDescription *	bd,
			int			fileFormat,
			double			compressionFactor	);

extern int bmRead(	const char *		filename,
			unsigned char **	pBuffer,
			BitmapDescription *	bd,
			int *			pFileFormat,
			double *		pCompressionFactor	);

extern int bmSelect(	unsigned char **		pBuffer,
			BitmapDescription *		bdOut,
			const unsigned char *		inputBuffer,
			const BitmapDescription *	bdIn,
			int				x0,
			int				y0,
			int				xs,
			int				ys		);

extern int bmComponents( void ***		pComponents,
			int *			pCount,
			const unsigned char *	buffer,
			const BitmapDescription * bd			);

extern int bmComponentBitmap( unsigned char **		buffer,
			BitmapDescription *		bdout,
			BitmapSelection *		bs,
			const BitmapDescription *	bdin,
			const void *			component	);

extern int bmGroupBitmap( unsigned char **		buffer,
			BitmapDescription *		bdout,
			BitmapSelection *		bs,
			const BitmapDescription *	bdin,
			const void *			vbc	);

int bmgGroupBitmap(	unsigned char **		pBuffer,
			BitmapDescription *		bdout,
			BitmapSelection *		bs,
			const BitmapDescription *	bdin,
			const void *			vcg );

extern int bmGroups(	void ***			pGroups,
			void **				comps,
			int				ncomp,
			const BitmapDescription *	bd		);

extern void bmFreeGroups( void **			groups,
			int				numberOfGroups	);

/************************************************************************/
/*  Palette expansion for drawing.					*/
/************************************************************************/

void bmExpandRGB8Palette(	unsigned char *		to,
				const unsigned char *	from,
				int			pixelsWide,
				int			bitsPerColor,
				RGB8Color *		palette,
				int			hasAlpha	);

/************************************************************************/
/*  Bitmap transformations.						*/
/************************************************************************/

typedef int (*BitmapTransformation)(BitmapDescription *		bdOut,
				const BitmapDescription *	bdIn,
				unsigned char **		pBufOut,
				const unsigned char *		bufIn,
				int				option );

extern int bmWhiteToTransparent(BitmapDescription *		bdOut,
				const BitmapDescription *	bdIn,
				unsigned char **		pBufOut,
				const unsigned char *		bufIn,
				int				ignoredInt );

extern int bmVerticalFlip(	BitmapDescription *		bdOut,
				const BitmapDescription *	bdIn,
				unsigned char **		pBufOut,
				const unsigned char *		bufIn,
				int				ignoredInt );

extern int bmHorizontalFlip(	BitmapDescription *		bdOut,
				const BitmapDescription *	bdIn,
				unsigned char **		pBufOut,
				const unsigned char *		bufIn,
				int				ignoredInt );

extern int bmUpsideDown(	BitmapDescription *		bdOut,
				const BitmapDescription *	bdIn,
				unsigned char **		pBufOut,
				const unsigned char *		bufIn,
				int				ignoredInt );

extern int bmRotate90(		BitmapDescription *		bdOut,
				const BitmapDescription *	bdIn,
				unsigned char **		pBufOut,
				const unsigned char *		bufIn,
				int				ignoredInt );

extern int bmColorReduce(	BitmapDescription *		bdOut,
				const BitmapDescription *	bdIn,
				unsigned char **		pBufOut,
				const unsigned char *		bufIn,
				int				colorCount );

extern int bmBlur(	BitmapDescription *		bdOut,
			const BitmapDescription *	bdIn,
			unsigned char **		pBufOut,
			const unsigned char *		bufIn,
			int				colorCount );

int bmRotate(	BitmapDescription *		bdOut,
		const BitmapDescription *	bdIn,
		unsigned char **		pBufOut,
		const unsigned char *		bufIn,
		double				angle	);

int bmFilterSobel(	BitmapDescription *		bdOut,
			const BitmapDescription *	bdIn,
			unsigned char **		pBufOut,
			const unsigned char *		bufIn,
			int				ignoredInt	);

int bmFilterLaplace(	BitmapDescription *		bdOut,
			const BitmapDescription *	bdIn,
			unsigned char **		pBufOut,
			const unsigned char *		bufIn,
			int				ignoredInt	);

int bmFilterSmoothe(	BitmapDescription *		bdOut,
			const BitmapDescription *	bdIn,
			unsigned char **		pBufOut,
			const unsigned char *		bufIn,
			int				ignoredInt	);

#   endif
