/*
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Brian W. Barrett, Arun F. Rodrigues, Jeffrey M. Squyres,
 * 	 and Andrew Lumsdaine
 *
 * This file is part of XMPI
 *
 * You should have received a copy of the License Agreement for XMPI 
 * along with the software; see the file LICENSE.  If not, contact 
 * Office of Research, University of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 *
 * Additional copyrights may follow.

 *
 *	$Id: xmpi_dbase.c,v 1.2 1999/11/08 06:20:23 bbarrett Exp $
 * 
 *	Function:	- process/trace information database
 */

#include <stdlib.h>
#include <stdio.h>

#include "all_list.h"
#include "lam.h"
#include "xmpi.h"
#include "xmpi_dbase.h"

/*
 * global functions
 */
int			xmpi_db_getarrows();
void			xmpi_db_getcomm();
int			xmpi_db_getgpeer();
int			xmpi_db_getlpeer();
int			xmpi_db_getload();
int			xmpi_db_getnprocs();
int			xmpi_db_gettraces();
void			*xmpi_db_getdtype();
double			xmpi_db_getmaxtime();
double			xmpi_db_getmintime();
double			xmpi_db_getminlapse();
int			xmpi_db_getmaxtraces();
int			xmpi_db_getstat();
void			xmpi_db_settime();

/*
 * local functions
 */
static int		group_index();
static void		fillmsgstat();
static int		addmesg();
static struct xmdbtr *	db_findtrace();

/*
 * global variables
 */
struct xmdb		dbase = {0, 0, 0, 0.0, 0.0, 0.0}; /* the database */

/*
 *	xmpi_db_getnprocs
 *
 *	Function:	- get the # processes
 *	Returns:	- # processes
 */
int
xmpi_db_getnprocs()

{
	return(dbase.xdb_nprocs);
}

/*
 *	xmpi_db_getmintime
 *
 *	Function:	- get the min (start) time
 *	Returns:	- start time
 */
double
xmpi_db_getmintime()

{
	return(dbase.xdb_mintime);
}

/*
 *	xmpi_db_getmaxtime
 *
 *	Function:	- get the max (end) time
 *	Returns:	- end time
 */
double
xmpi_db_getmaxtime()

{
	return(dbase.xdb_maxtime);
}

/*
 *	xmpi_db_getminlapse
 *
 *	Function:	- get the min lapse time
 *	Returns:	- lapse time
 */
double
xmpi_db_getminlapse()

{
	return(dbase.xdb_minlapse);
}

/*
 *	xmpi_db_getmaxtraces
 *
 *	Function:	- get the maximum number of traces for any process
 *	Returns:	- number of traces
 */
int
xmpi_db_getmaxtraces()

{
	struct xmdbproc	*p;			/* favourite pointer */
	int		i;			/* favourite index */
	int		trmax = 0;		/* max traces */

	for (i = 0, p = dbase.xdb_procs; i < dbase.xdb_nprocs; ++i, ++p) {
		if (al_count(p->xdbp_traces) > trmax) {
			trmax = al_count(p->xdbp_traces);
		}
	}

	return(trmax);
}

/*
 *	xmpi_db_getload
 *
 *	Function:	- get blocking & system load (percentages)
 *	Accepts:	- size of arrays (# entries)
 *			- array of percent block load (out)
 *			- array of percent system load (out)
 *	Returns:	- # entries filled
 */
int
xmpi_db_getload(size, block, system)

int			size;
double			*block;
double			*system;

{
	int		n;			/* # entries to fill */
	int		i;			/* favourite index */
	double		tottime;		/* total time */
	double		delta;			/* time difference */
	struct xmdbproc	*p;			/* favourite pointer */
	struct xmdbtr	*ptrace;		/* ptr trace */

	n = dbase.xdb_nprocs;
	if (size < n) n = size;

	tottime = dbase.xdb_curtime - dbase.xdb_mintime;
/*
 * Fill the arrays.
 */
	p = dbase.xdb_procs;

	for (i = 0; i < n; ++i, ++p) {
		ptrace = p->xdbp_curtrace;

		if (ptrace == 0) {
			block[i] = system[i] = -1.0;
			continue;
		}

		if (tottime <= 0.0) {
			block[i] = system[i] = 0.0;
			continue;
		}

		block[i] = ptrace->xdbt_blktotal;
		system[i] = ptrace->xdbt_systotal;

		delta = dbase.xdb_curtime - ptrace->xdbt_time;

		if (ptrace->xdbt_state == XMPI_SBLOCK) {
			block[i] += (delta > ptrace->xdbt_lapse) ?
						ptrace->xdbt_lapse : delta;
		}
		else if (ptrace->xdbt_state == XMPI_SSYSTEM) {
			system[i] += (delta > ptrace->xdbt_lapse) ?
						ptrace->xdbt_lapse : delta;
		}

		block[i] /= tottime - ((delta > ptrace->xdbt_lapse) ?
					(delta - ptrace->xdbt_lapse) : 0.0);
		system[i] /= tottime - ((delta > ptrace->xdbt_lapse) ?
					(delta - ptrace->xdbt_lapse) : 0.0);
	}

	return(n);
}

/*
 *	xmpi_db_gettraces
 *
 *	Function:	- get traces for a time period
 *			- array of traces malloc'ed by the function
 *			- it is up to the caller to free the array
 *	Accepts:	- time period start
 *			- time period end
 *			- trace array (out)
 *	Returns:	- # array entries or LAMERROR
 */
int
xmpi_db_gettraces(t1, t2, ptraces)

double			t1;
double			t2;
struct xmtrace		**ptraces;

{
	int		n;			/* # traces filled */
	int		i;			/* favourite index */
	int		size;			/* size of array */
	unsigned	esz;			/* array expansion size */
	double		curtime;		/* current time */
	struct xmtrace	*traces;		/* trace array */
	struct xmtrace	*etraces;		/* expanded trace array */
	struct xmdbproc	*proc;			/* ptr process */
	struct xmdbtr	*p;			/* favourite pointer */
	LIST		*trlist;		/* trace list */
/*
 * Move to the start time.
 */
	curtime = dbase.xdb_curtime;
	xmpi_db_settime(t1);

	n = 0;
	size = XMPI_DBNTRACES;
	traces = (struct xmtrace *)
			malloc((unsigned) size * sizeof(struct xmtrace));
	if (traces == 0) {
		xmpi_db_settime(curtime);
		return(LAMERROR);
	}
/*
 * Loop over the processes.
 */
	proc = dbase.xdb_procs;

	for (i = 0; i < dbase.xdb_nprocs; ++i, ++proc) {

		trlist = proc->xdbp_traces;
/*
 * Loop adding process traces in the time period.
 */
		p = proc->xdbp_curtrace;

		for (; p && (p->xdbt_time <= t2);
			p = (struct xmdbtr *) al_next(trlist, (char *) p)) {
/*
 * Expand the trace array if needed.
 */
			if (n == size) {
				esz = 2 * size * sizeof(struct xmtrace);

				etraces = (struct xmtrace *)
						realloc((char *) traces, esz);
				if (etraces == 0) {
					xmpi_db_settime(curtime);
					free((char *) traces);
					return(LAMERROR);
				}

				size += size;
				traces = etraces;
			}
/*
 * Append the trace to the array.
 */
			traces[n].xmt_rank = p->xdbt_grank;
			traces[n].xmt_state = p->xdbt_state;
			traces[n].xmt_time = p->xdbt_time;
			traces[n].xmt_lapse = p->xdbt_lapse;

			++n;
		}
	}

	*ptraces = traces;

	xmpi_db_settime(curtime);

	return(n);
}

/*
 *	xmpi_db_getarrows
 *
 *	Function:	- get arrows for a time period
 *			- array of arrows malloc'ed by the function
 *			- it is up to the caller to free the array
 *	Accepts:	- time period start
 *			- time period end
 *			- arrow array (out)
 *	Returns:	- # array entries or LAMERROR
 */
int
xmpi_db_getarrows(t1, t2, parrows)

double			t1;
double			t2;
struct xmarrow		**parrows;

{
	int		n;			/* # arrows filled */
	int		i;			/* favourite index */
	int		size;			/* size of array */
	unsigned	esz;			/* array expansion size */
	double		curtime;		/* current time */
	struct xmarrow	*arrows;		/* arrow array */
	struct xmarrow	*earrows;		/* expanded arrow array */
	struct xmdbproc	*proc;			/* ptr process */
	struct xmdbtr	*p;			/* favourite pointer */
	struct xmdbtr	*snd, *rcv;		/* send/recv traces */
	LIST		*trlist;		/* trace list */
/*
 * Move to the start time.
 */
	curtime = dbase.xdb_curtime;
	xmpi_db_settime(t1);

	n = 0;
	size = XMPI_DBNARROWS;
	arrows = (struct xmarrow *)
			malloc((unsigned) size * sizeof(struct xmarrow));
	if (arrows == 0) {
		xmpi_db_settime(curtime);
		return(LAMERROR);
	}
/*
 * Loop over the processes.
 */
	proc = dbase.xdb_procs;

	for (i = 0; i < dbase.xdb_nprocs; ++i, ++proc) {

		trlist = proc->xdbp_traces;
/*
 * Loop adding arrows affecting the time period.
 */
		p = proc->xdbp_curtrace;

		for (; p && (p->xdbt_time <= t2);
			p = (struct xmdbtr *) al_next(trlist, (char *) p)) {

			if (p->xdbt_arrowdir == XMPI_DBNA) continue;
/*
 * Expand the arrow array if needed.
 */
			if (n == size) {
				esz = 2 * size * sizeof(struct xmarrow);

				earrows = (struct xmarrow *)
						realloc((char *) arrows, esz);
				if (earrows == 0) {
					xmpi_db_settime(curtime);
					free((char *) arrows);
					return(LAMERROR);
				}

				size += size;
				arrows = earrows;
			}
/*
 * Append the arrow to the array.
 */
			if (p->xdbt_arrowdir == XMPI_DBOUT) {
				snd = p;
				rcv = p->xdbt_arrow;
			} else {
				rcv = p;
				snd = p->xdbt_arrow;
			}

			if (snd) {
				arrows[n].xma_srank = snd->xdbt_grank;
				arrows[n].xma_stime = snd->xdbt_time;

				if (snd->xdbt_arrowloc == XMPI_DBEND)
					arrows[n].xma_stime += snd->xdbt_lapse;
			} else {
				arrows[n].xma_srank = -1;
			}

			if (rcv) {
				arrows[n].xma_rrank = rcv->xdbt_grank;
				arrows[n].xma_rtime = rcv->xdbt_time;

				if (rcv->xdbt_arrowloc == XMPI_DBEND)
					arrows[n].xma_rtime += rcv->xdbt_lapse;
			} else {
				arrows[n].xma_rrank = -1;
			}

			++n;
		}
	}

	*parrows = arrows;

	xmpi_db_settime(curtime);

	return(n);
}

/*
 *	xmpi_db_getstat
 *
 *	Function:	- get current status of processes
 *	Accepts:	- # of processes
 *			- ptr process status array (filled)
 *	Returns:	- true iff process view data changed
 */
int
xmpi_db_getstat(nprocs, procstat)

int			nprocs;
struct xmproc		*procstat;

{
	int		i;			/* favourite index */
	struct xmproc	*p;			/* favourite pointer */
	struct xmdbproc	*pproc;			/* ptr dbase process */
	struct xmdbtr	*ptrace;		/* ptr trace */
	double		t1, t2;			/* trace end-times */
	int		laststate;		/* last state */
	int		lastnmsg;		/* last number of messages */
	int		unchanged = 1;		/* view data unchanged? */
/*
 * Loop over the processes.
 */
	p = procstat;
	pproc = dbase.xdb_procs;

	for (i = 0; i < nprocs; ++i, ++p, ++pproc) {

		laststate = p->xmp_state;
		lastnmsg = p->xmp_nmsg;
		
		ptrace = pproc->xdbp_curtrace;

		if (ptrace == 0) {
			p->xmp_state = XMPI_SUNDEF;
			unchanged &= (laststate == p->xmp_state);
			continue;
		}

		t1 = ptrace->xdbt_time;
		t2 = t1 + ptrace->xdbt_lapse;

		if ((t1 > dbase.xdb_curtime) || (t2 < dbase.xdb_curtime)) {
			p->xmp_state = XMPI_SUNDEF;
			unchanged &= (laststate == p->xmp_state);
			continue;
		}

		p->xmp_state = (ptrace->xdbt_state == XMPI_SMISSING) ?
					XMPI_SUNDEF : ptrace->xdbt_state;

		p->xmp_func = ptrace->xdbt_envelop.xdbe_func;
		p->xmp_wfunc = ptrace->xdbt_envelop.xdbe_wfunc;
		p->xmp_cid = ptrace->xdbt_envelop.xdbe_cid;
		p->xmp_tag = ptrace->xdbt_envelop.xdbe_tag;
		p->xmp_lpeer = ptrace->xdbt_envelop.xdbe_lpeer;
		p->xmp_gpeer = (p->xmp_lpeer >= 0)
			? xmpi_db_getgpeer(p->xmp_cid, i, p->xmp_lpeer) : -1;
		p->xmp_lrank = xmpi_db_getlpeer(p->xmp_cid, i, i);
		p->xmp_cnt = ptrace->xdbt_envelop.xdbe_count;
		p->xmp_dtype = ptrace->xdbt_envelop.xdbe_dtype;
		strcpy(p->xmp_prog, pproc->xdbp_prog);
		p->xmp_lroot = p->xmp_lpeer;
		p->xmp_groot = p->xmp_gpeer;

		fillmsgstat(p, pproc);

		unchanged &= (laststate == p->xmp_state)
					&& (lastnmsg == p->xmp_nmsg);
	}

	return(!unchanged);
}

/*
 *	fillmsgstat
 *
 *	Function:	- fill the process buffered messages list
 *	Accepts:	- ptr process status entry
 *			- ptr process database entry
 */
static void
fillmsgstat(pstat, pproc)

struct xmproc		*pstat;
struct xmdbproc		*pproc;

{
	LIST		*list;
	struct xmdbtr	*ptrace;
	struct xmdbtr	**p;
	struct xmdbtr	*snd, *rcv;
	struct xmmsg	msgelem;
	struct xmdbenv	*env;
	double		sndtime, rcvtime;

	if (pstat->xmp_msgs) al_free(pstat->xmp_msgs);
	pstat->xmp_msgs = 0;
	pstat->xmp_nmsg = 0;
	pstat->xmp_more = 0;
	pstat->xmp_curmsg = 0;

	ptrace = pproc->xdbp_curtrace;
	if (ptrace == 0) return;

	p = ptrace->xdbt_senders;
	if (p == 0) return;

	list = pproc->xdbp_msgsnd;
	pstat->xmp_msgs = al_init(sizeof(struct xmmsg), 0);

	for (; p; p = (struct xmdbtr **) al_next(list, (char *) p)) {
		snd = *p;
		if (! (rcv = snd->xdbt_arrow)) continue;

		sndtime = snd->xdbt_time;
		if (snd->xdbt_arrowloc == XMPI_DBEND)
				sndtime += snd->xdbt_lapse;

		rcvtime = rcv->xdbt_time;
		if (rcv->xdbt_arrowloc == XMPI_DBEND)
				rcvtime += rcv->xdbt_lapse;

		if ((sndtime < dbase.xdb_curtime) &&
				(rcvtime > dbase.xdb_curtime)) {
			env = &snd->xdbt_envelop;
			msgelem.xmm_tag = env->xdbe_tag;
			msgelem.xmm_lsrc = xmpi_db_getlpeer(env->xdbe_cid,
							rcv->xdbt_grank,
							snd->xdbt_grank);
			msgelem.xmm_gsrc = snd->xdbt_grank;
			msgelem.xmm_cid = env->xdbe_cid;
			msgelem.xmm_cnt = env->xdbe_count;
			msgelem.xmm_dtype = env->xdbe_dtype;
			msgelem.xmm_nmsg = 1;

			if (addmesg(pstat, &msgelem)) {
				xmpi_fail("xmpi (addmesg)");
			}
		}
	}

	pstat->xmp_curmsg = al_top(pstat->xmp_msgs);
}

/*
 *	addmesg
 *
 *	Function:	- add message to process entry
 *	Accepts:	- process entry
 *			- message
 *	Returns:	- 0 or LAMERROR
 */
static int
addmesg(proc, msg)

struct xmproc		*proc;
struct xmmsg		*msg;

{
	struct xmmsg	*p;

	proc->xmp_nmsg++;
/*
 * Try to locate a message aggregate of the same type.
 * If found, increment its count.
 */
	p = al_top(proc->xmp_msgs);
	while (p) {
		if ((p->xmm_cid == msg->xmm_cid)
			&& (p->xmm_tag == msg->xmm_tag)
			&& (p->xmm_gsrc == msg->xmm_gsrc)
			&& (p->xmm_cnt == msg->xmm_cnt) 
			&& (p->xmm_dtype == msg->xmm_dtype)) {

			p->xmm_nmsg++;
			return(0);
		}

		p = al_next(proc->xmp_msgs, p);
	}
/*
 * No matching message aggregate found.  Add a new one.
 */
	return((al_append(proc->xmp_msgs, msg) == 0) ? LAMERROR : 0);
}

/*
 *	xmpi_db_settime
 *
 *	Function:	- set current time to given time
 *			- find first trace at that time
 */
void
xmpi_db_settime(time)

double			time;

{
	int		i;			/* favourite index */
	struct xmdbproc	*p;			/* favourite pointer */
/*
 * If current time setting is ok, do nothing.
 */
	if (time == dbase.xdb_curtime) return;
/*
 * Adjust each process' current trace to given time.
 */
	p = dbase.xdb_procs;

	for (i = 0; i < dbase.xdb_nprocs; ++i, ++p)
		p->xdbp_curtrace = db_findtrace(time, p);

	dbase.xdb_curtime = time;
}

/*
 *	xmpi_db_getlpeer
 *
 *	Function:	- return peer local rank
 *	Accepts:	- context ID
 *			- my global rank
 *			- peer global rank (may be me)
 *	Returns:	- peer local rank or LAMERROR
 */
int
xmpi_db_getlpeer(cid, grank, gpeer)

int			cid;
int			grank;
int			gpeer;

{
	struct xmdbcid	*p;			/* favourite pointer */
	LIST		*list;			/* list of cids */
	int		i;			/* favourite index */
/*
 * Find a matching context ID entry.
 */
	list = dbase.xdb_cids;

	for (p = (struct xmdbcid *) al_top(list); p;
			p = (struct xmdbcid *) al_next(list, (char *) p)) {

		if (p->xdbc_cid != cid) continue;

		if (p->xdbc_nrprocs <= 0) {
/*
 * Match found and it is an intracommunicator.
 */
			i = group_index(grank, p->xdbc_lgrp, p->xdbc_nlprocs);

			if (i >= 0) {
				if (grank == gpeer) {
					return(i);
				} else {
					return(group_index(gpeer, p->xdbc_lgrp,
						p->xdbc_nlprocs));
				}
			}
		}
		else {
/*
 * Match found and it is an intercommunicator.
 */
			i = group_index(grank, p->xdbc_lgrp, p->xdbc_nlprocs);

			if (i >= 0) {
				if (grank == gpeer) {
					return(i);
				} else {
					return(group_index(gpeer, p->xdbc_rgrp,
						p->xdbc_nrprocs));
				}
			}
			
			i = group_index(grank, p->xdbc_rgrp, p->xdbc_nrprocs);

			if (i >= 0) {
				if (grank == gpeer) {
					return(i);
				} else {
					return(group_index(gpeer, p->xdbc_lgrp,
						p->xdbc_nlprocs));
				}
			}
		}
	}
	
	return(LAMERROR);
}


/*
 *	xmpi_db_getgpeer
 *
 *	Function:	- return peer global rank
 *	Accepts:	- context ID
 *			- my global rank
 *			- peer local rank
 *	Returns:	- peer global rank or LAMERROR
 */
int
xmpi_db_getgpeer(cid, grank, lpeer)

int			cid;
int			grank;
int			lpeer;

{
	struct xmdbcid	*p;			/* favourite pointer */
	int		i;			/* favourite index */
	LIST		*list;			/* list of cids */
/*
 * Find a matching context ID entry.
 */
	list = dbase.xdb_cids;

	for (p = (struct xmdbcid *) al_top(list); p;
			p = (struct xmdbcid *) al_next(list, (char *) p)) {

		if (p->xdbc_cid != cid) continue;

		if (p->xdbc_nrprocs == 0) {
/*
 * Intracommunicator.
 */
			if (lpeer >= p->xdbc_nlprocs) continue;
/*
 * Check if my global rank is in it.
 */
			i = group_index(grank, p->xdbc_lgrp, p->xdbc_nlprocs);

			if (i >= 0) return(p->xdbc_lgrp[lpeer].gps_grank);
		}
		else {
/*
 * Intercommunicator. First check if my global rank is in local group.
 */
			i = group_index(grank, p->xdbc_lgrp, p->xdbc_nlprocs);

			if (i >= 0) {
				if (lpeer < p->xdbc_nrprocs)
					return(p->xdbc_rgrp[lpeer].gps_grank);
				else
					return(LAMERROR);
			}
/*
 * Not in local group. Check if my global rank is in remote group.
 */
			i = group_index(grank, p->xdbc_rgrp, p->xdbc_nrprocs);

			if (i >= 0) {
				if (lpeer < p->xdbc_nlprocs)
					return(p->xdbc_lgrp[lpeer].gps_grank);
				else
					return(LAMERROR);
			}
		}
	}

	return(LAMERROR);
}

/*
 *	group_index
 *
 *	Function:	- find index of process in a group 
 *	Accepts:	- global rank of process
 *			- ptr group GPS
 *			- size of group
 *	Returns:	- index in group or if not present LAMERROR
 */
static int
group_index(rank, group, n)

int			rank;
struct _gps		*group;
int			n;

{
	int 		i;
	struct _gps	*pg;

	for (i = 0, pg = group; i < n; ++i, ++pg) {
		if (pg->gps_grank == rank) return(i);
	}

	return(LAMERROR);
}

/*
 *	xmpi_db_getcomm
 *
 *	Function:	- get communicator processes
 *	Accepts:	- context ID
 *			- member global rank
 *			- array of GPS (out)
 *			- num processes local group  (out)
 *			- num processes remote group (out)
 *			- local/remote flag (out)
 *
 *	If no matching communicator processes can be found the number
 *	of processes in the local group is returned as zero.
 *  	In case of an intra-comm the number of processes in the group
 *  	is returned in the num processes local group parameter and 0
 *  	is returned for the num processes remote group parameter.  In
 *  	case of an inter-comm flag is returned as 1 if member global
 *  	rank is in the local group and 0 if in the remote group.
 */
void
xmpi_db_getcomm(cid, grank, pgps, nlocal, nremote, islocal)

int			cid;
int			grank;
struct _gps		**pgps;
int			*nlocal;
int			*nremote;
int			*islocal;

{
	struct xmdbcid	*p;			/* favourite pointer */
	int		i;			/* favourite index */
	LIST		*list;			/* list of cids */
/*
 * Find a matching context ID entry.
 */
	list = dbase.xdb_cids;

	cid = xmpi_coll2pt(cid);

	for (p = (struct xmdbcid *) al_top(list); p;
			p = (struct xmdbcid *) al_next(list, (char *) p)) {

		if (p->xdbc_cid != cid) continue;

		if (p->xdbc_nrprocs == 0) {
/*
 * Intracommunicator. Check if my global rank is in it.
 */
			i = group_index(grank, p->xdbc_lgrp, p->xdbc_nlprocs);

			if (i >= 0) {
				*pgps = p->xdbc_lgrp;
				*nlocal = p->xdbc_nlprocs;
				*nremote = 0;
				*islocal = 1;
				return;
			}
		} else {			
/*
 * Intercommunicator. First check if member global rank is in local group.
 */
			i = group_index(grank, p->xdbc_lgrp, p->xdbc_nlprocs);
			if (i >= 0) {
				*pgps = p->xdbc_lgrp;
				*nlocal = p->xdbc_nlprocs;
				*nremote = p->xdbc_nrprocs;
				*islocal = 1;
				return;
			}
/*
 * Not in local group. Check if member global rank is in remote group.
 */
			i = group_index(grank, p->xdbc_rgrp, p->xdbc_nrprocs);
			if (i >= 0) {
				*pgps = p->xdbc_lgrp;
				*nlocal = p->xdbc_nlprocs;
				*nremote = p->xdbc_nrprocs;
				*islocal = 0;
				return;
			}
		}
	}
	
	*nlocal = 0;
}

/*
 *	xmpi_db_getdtype
 *
 *	Function:	- get datatype description for a process
 *	Accepts:	- process global rank
 *			- datatype index
 *	Returns:	- datatype trace if found, else 0 if dtype is
 *			  in range of all known basic types, else -1
 */
void *
xmpi_db_getdtype(rank, dtype)

int			rank;
int			dtype;

{
	struct xmdbdt	dtelem;			/* datatype element */
	struct xmdbdt	*p;			/* favourite pointer */
	LIST		*list;			/* list of datatypes */
	void		*trace = 0;		/* datatype trace */

	list = dbase.xdb_procs[rank].xdbp_dtypes;

	if (list) {
		dtelem.xdbd_dtype = dtype;
		if ((p = (struct xmdbdt *) al_find(list, (char *) &dtelem))) {
			trace = p->xdbd_dtbuf;
		}
	}

	if (!trace && (dtype > XMPI_MAX_BASIC_TYPE)) {
		return((void *) -1);
	} else {
		return(trace);
	}
}

/*
 *	db_findtrace
 *
 *	Function:	- locate process trace at given time
 *	Accepts:	- time
 *			- ptr process entry
 *	Returns:	- ptr trace or NULL
 */
static struct xmdbtr *
db_findtrace(time, proc)

double			time;
struct xmdbproc		*proc;

{
	struct xmdbtr	*p;			/* favourite pointer */
	LIST		*trlist;		/* list of traces */
/*
 * Set current trace.
 */
	trlist = proc->xdbp_traces;
	p = proc->xdbp_curtrace;

	if (p == 0) {

		if (dbase.xdb_curtime <=
				((dbase.xdb_mintime + dbase.xdb_maxtime) / 2)) {
			p = (struct xmdbtr *) al_top(trlist);
		} else {
			p = (struct xmdbtr *) al_bottom(trlist);
		}

		if (p == 0) return(p);
	}
/*
 * Loop (forward or backward) finding the valid trace.
 */
	while (p) {

		if ((time >= p->xdbt_time) &&
				(time <= (p->xdbt_time + p->xdbt_lapse))) {
			break;
		}
		else if (time < p->xdbt_time) {
			p = (struct xmdbtr *) al_prev(trlist, (char *) p);
		}
		else {
			p = (struct xmdbtr *) al_next(trlist, (char *) p);
		}
	}

	if (p == 0) p = (struct xmdbtr *) al_bottom(trlist);

	return(p);
}
