/* 
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: util.h,v 1.57 1999/04/29 18:02:16 tiggr Exp $  */

#ifndef _trt_util_h_include_once
#define _trt_util_h_include_once
#include <stdarg.h>
#include <tom/trt.h>

/* Return the address of the extension variables of the extension
   identified by EXTENSION_ID, in the object SELF.  */
TRT_INLINE void *
trt_ext_address (tom_object self, int extension_id)
{
  struct trt_extension_offset_table *eot = self->isa->info.eot;
  int offset = eot->offset[extension_id];

  /* Check for an attempt to access variables of an extension not present
     in the object SELF.  */
  if (extension_id >= eot->st.num)
    abort ();
  if (!offset)
    {
      /* This is ok if it is the main State extension.  */
      extern unsigned int _ei_i_tom_State, _ei_c_tom_State;

      /* XXX This test could be simpler if this were two seperate
         functions, one for class objects and one for instances.
	 Sat May  3 19:32:13 1997, tiggr@tricky.es.ele.tue.nl  */
      if (extension_id != _ei_i_tom_State && extension_id != _ei_c_tom_State)
	abort ();
    }

  return (char *) self + offset;
}

/* Return !0 iff the selector argument descriptors A and B denote the same
   argument types.  */
TRT_INLINE int
trt_selector_args_match (struct trtd_selector_args *a,
			 struct trtd_selector_args *b)
{
  int i;

  /* This test would suffice if there were no dynamic loading, since the
     arg type descriptors are uniqued by the resolver.

     XXX Maybe they should be uniqued also by the dynamic resolver, or
     maybe they should bear unique identifiers like the selectors do.
     Then a simple comparison (either of the pointer or of the identity)
     will allways suffice.  */
  if (a == b)
    return 1;

  if (a->num != b->num)
    return 0;

  for (i = 0; i < a->num; i++)
    if (a->args[i] != b->args[i])
      return 0;

  return 1;
}

/* Return !0 iff the two selectors denote the same selector.  */
TRT_INLINE int
trt_selectors_equal (selector a, selector b)
{
  return a == b || a->sel_id == b->sel_id;
}

/* Make the object O gray if it is not gray, black, or nil.  */
TRT_INLINE void
mark_if_needed (struct trt_instance *o)
{
  if (o && TGC_COLOUR (o->asi) == TGC_WHITE)
    trt_make_gray (o);
}

/* The environment, as passed to main.  */
extern char **trt_environ;

/* The `main' main.  */
int trt_main (int argc, char **argv);

/* Return the selector whose name is LEN bytes long, and pointed to by S.  */
selector trt_selector_named (char *s, int len);

/* Return the size of the arguments, to be passed to __builtin_apply, for
   the selector SEL.  */
int trt_all_arguments_size (selector sel);

/* Return a new byte-string containing a copy of the first N bytes pointed
   to by S.  */
tom_object byte_string_with_string (char *s, int n);

/* Return a new byte-string containing a copy of the bytes from the
   (zero-terminated) C-string S.  */
tom_object byte_string_with_c_string (char *s);

/* Return a malloced zero terminated string with the LEN characters at
   BUF.  */
char *c_string_with_length (char *buf, int len);

/* Return a malloced zero terminated string with the byte STRING's contents.  */
char *c_string_with_tom_string (tom_object string);

/* Likewise, but as a macro returning alloca'd memory.  */
#define C_STRING_WITH_TOM_STRING(S, LEN, TS)  \
  do									  \
    {									  \
      DECL_USEL (tom, _pi__byteStringContents);				  \
      (S) = TRT_SEND (_PI_, (TS), USEL (tom, _pi__byteStringContents), &(LEN));\
      (S) = memcpy (alloca ((LEN) + 1), (S), (LEN));			  \
      (S)[(LEN)] = 0;							  \
    } while (0)

/* Return the address of the ivar named NAME in the object SELF.  The
   variable should have the TYPE.  */
void *trt_address_of_ivar (tom_object self, tom_object name,
			   enum trt_type_encoding type);

/* Return the name of the indicated type.  */
char *trt_type_name (enum trt_type_encoding type);

/* Indicate that the object SELF does not recognize the selector CMD.  */
void trt_unrecognized (tom_object self, selector cmd);

/* Issue a fatal error (in the runtime).  */
void fatal (char *fmt, ...);
void unimplemented (char *fmt, ...);
void vfatal (char *fmt, va_list ap);

/* Sensible memory allocation.  */
void *xmalloc (unsigned int n);
void *xcalloc (unsigned int n, unsigned int m);
void *xrealloc (void *p, unsigned int n);
void xfree (void *p);

/* Delayed free.  The runtime must be locked.  */
void safe_free (void *p);

/* Safe realloc, by malloc and delayed free.  The runtime must be locked.  */
void *safe_realloc (void *p, int old_size, int new_size);

#ifdef DEBUG
#undef ASSERT
#define ASSERT(X)  \
  do if (!(X))								\
    fatal ("%s:%d: assertion failed: %s", __FILE__, __LINE__, #X);	\
  while (0)
#else
#define ASSERT(X) do ; while (0)
#endif

#define TRT_SIGNAL(OBJ, CLASS, MSG)  \
  ({									\
     tom_object c = TRT_SEND ((reference_imp), _mr_c_tom_Condition,	\
			      USEL (tom, r_for_r_class_r_message_r),	\
			      (OBJ), (CLASS), (MSG));			\
     TRT_SEND ((reference_imp), c, USEL (tom, r_signal));})

#define TRT_RAISE(OBJ, CLASS, MSG)  \
  do									\
    {									\
      tom_object c = TRT_SEND ((reference_imp), _mr_c_tom_Condition,	\
			       USEL (tom, r_for_r_class_r_message_r),	\
			       (OBJ), (CLASS), (MSG));			\
      TRT_SEND ((reference_imp), c, USEL (tom, v_raise));		\
    } while (0)

/* Raise or signal a selector condition, or, if the CMD is nil, a normal
   condition.  */
tom_object trt_raise (int signal_p, tom_object object, selector cmd,
		      void *condition_class, char *fmt, ...);


#define TRT_PANIC_MODE_P()  \
  do							\
    {							\
      if (THREADS_PACKAGE != THREADS_NO_THREADS && trt_runtime_lock)	\
	trt_rlock_lock (trt_runtime_lock);		\
      c_tom_Runtime_panic_mode++;			\
    } while (0)

#define TRT_PANIC_MODE_V()  \
  do									\
    {									\
      if (THREADS_PACKAGE != THREADS_NO_THREADS && trt_runtime_lock)	\
	trt_rlock_unlock (trt_runtime_lock);				\
      if (!c_tom_Runtime_quit_pending)					\
	c_tom_Runtime_panic_mode--;					\
      else								\
	TRT_SEND ((void_imp), _mr_c_tom_Runtime, SEL (v_panic_disable));  \
    } while (0)

/* Threads and locks are opaque types.  */
typedef tom_int trt_thread_t;
typedef struct trt_lock *trt_lock_t;
typedef struct trt_rlock *trt_rlock_t;
typedef struct trt_sem *trt_sem_t;

/* The mega lock on modifying the runtime structure, if !0  */
extern trt_rlock_t trt_runtime_lock;

/* The lock on the object allocation structures.  */
extern trt_rlock_t trt_alloc_lock;

/* The lock on the gray stack.  */
extern trt_lock_t trt_gray_lock;

/* Get or set the thread-local variable X.  */
#define GET_THREAD_LOCAL(LOCAL_BASE, TYPE, LOCAL_VAR)  \
  (*(TYPE *) ((char *) (LOCAL_BASE) + _tlo_##LOCAL_VAR))
#define SET_THREAD_LOCAL(LOCAL_BASE, TYPE, LOCAL_VAR, VALUE)  \
  (*(TYPE *) ((char *) (LOCAL_BASE) + _tlo_##LOCAL_VAR) = VALUE)

/* Create a new recursive lock.  A recursive lock is a binary lock which
   can be locked multiple times by the same thread.  */
trt_rlock_t trt_rlock_create (void);

/* Destroy the recursive lock.  */
void trt_rlock_destroy (trt_rlock_t rlock);

/* Block until a lock on the LOCK can be obtained.  */
void trt_rlock_lock (trt_rlock_t rlock);

/* Release a lock on the LOCK.  */
void trt_rlock_unlock (trt_rlock_t rlock);

/* Attempt to obtain the lock on the RLOCK.  Return 0 if failed.  */
tom_byte trt_rlock_try (trt_rlock_t rlock);

#if THREADS_PACKAGE != THREADS_NO_THREADS
#define LOCK_RUNTIME()  \
  do						\
    {						\
      if (trt_runtime_lock)			\
	trt_rlock_lock (trt_runtime_lock);	\
    } while (0)
#define UNLOCK_RUNTIME()  \
  do						\
    {						\
      if (trt_runtime_lock)			\
	trt_rlock_unlock (trt_runtime_lock);	\
    } while (0)
#if MALLOC_THREAD_SAFE
#define LOCK_ALLOC()  \
  do ; while (0)
#define UNLOCK_ALLOC()  \
  do ; while (0)
#else
#define LOCK_ALLOC()  \
  do						\
    {						\
      if (trt_alloc_lock)			\
	trt_rlock_lock (trt_alloc_lock);	\
    } while (0)
#define UNLOCK_ALLOC()  \
  do						\
    {						\
      if (trt_alloc_lock)			\
	trt_rlock_unlock (trt_alloc_lock);	\
    } while (0)
#endif
#define LOCK_GRAY()  \
  do						\
    {						\
      if (trt_gray_lock)			\
	trt_lock_lock (trt_gray_lock);	\
    } while (0)
#define UNLOCK_GRAY()  \
  do						\
    {						\
      if (trt_gray_lock)			\
	trt_lock_unlock (trt_gray_lock);	\
    } while (0)
#else
#define LOCK_RUNTIME()  \
  do ; while (0)
#define UNLOCK_RUNTIME()  \
  do ; while (0)
#define LOCK_ALLOC()  \
  do ; while (0)
#define UNLOCK_ALLOC()  \
  do ; while (0)
#define LOCK_GRAY()  \
  do ; while (0)
#define UNLOCK_GRAY()  \
  do ; while (0)
#endif

/* Create a new semaphore.  The first ALLOW locks will succeed.  */
trt_sem_t trt_sem_create (int allow);

/* Destroy the semaphore SEM.  */
void trt_sem_destroy (trt_sem_t sem);

/* Block until a lock on the semaphore SEM can be obtained.  */
void trt_sem_lock (trt_sem_t sem);

/* Release a lock on the semaphore SEM.  */
void trt_sem_unlock (trt_sem_t sem);

/* Attempt to obtain a lock on the semaphore SEM.  Return 0 if failed.  */
tom_byte trt_sem_try (trt_sem_t sem);

/* Create a new lock.  */
trt_lock_t trt_lock_create (void);

/* Destroy the LOCK.  */
void trt_lock_destroy (trt_lock_t lock);

/* Block until the LOCK can be locked.  */
void trt_lock_lock (trt_lock_t lock);

/* Release the LOCK.  */
void trt_lock_unlock (trt_lock_t lock);

/* Attempt to lock the LOCK.  Return 0 if failed.  */
tom_byte trt_lock_try (trt_lock_t lock);

/* Short-cut method invocation typing.  */
#define _TB_  tom_byte
#define _TC_  tom_char
#define _TI_  tom_int
#define _TL_  tom_long
#define _TF_  tom_float
#define _TD_  tom_double
#define _TP_  void *

#define _BI_  (byte_imp)
#define _CI_  (char_imp)
#define _II_  (int_imp)
#define _LI_  (long_imp)
#define _FI_  (float_imp)
#define _DI_  (double_imp)
#define _PI_  (pointer_imp)

#define MAT(RET, ARGS...)  \
  (RET (*) (_TP_, _TP_ , ## ARGS))

#define CREF(CLASS)  _mr_c_##CLASS

/* The CURRENT_UNIT should be set if compilation and resolution is run
   with the -1 option.  */
#ifdef CURRENT_UNIT
#define DECL_XSEL(X, Y) extern struct trtd_selector _sd_ ## X ## _ ## Y
#define DECL_SEL(Y) DECL_USEL (CURRENT_UNIT, Y)
#define DECL_USEL(X, Y)  DECL_XSEL (X, Y)
#else
#define DECL_XSEL(Y) extern struct trtd_selector _sd_ ## Y
#define DECL_SEL(Y) DECL_XSEL (Y)
#define DECL_USEL(X, Y) DECL_XSEL (Y)
#endif

#ifdef CURRENT_UNIT
/* Super references.  */
#define XSR(UNIT, SUB, SUPER) _sr_ ## UNIT ## _ ## SUB ## _ ## SUPER
#define SR(SUB, SUPER)  USR (CURRENT_UNIT, SUB, SUPER)
#define USR(UNIT, SUB, SUPER)  XSR (UNIT, SUB, SUPER)

/* Declaration of super references.  */
#define DECL_XSR(UNIT, SUB, SUPER)  \
     extern void * _sr_ ## UNIT ## _ ## SUB ## _ ## SUPER
#define DECL_SR(SUB, SUPER)  DECL_USR (CURRENT_UNIT, SUB, SUPER)
#define DECL_USR(UNIT, SUB, SUPER)  DECL_XSR (UNIT, SUB, SUPER)

#else
/* Super references.  */
#define XSR(SUB, SUPER) _sr_ ## SUB ## _ ## SUPER
#define SR(SUB, SUPER)  XSR (SUB, SUPER)
#define USR(UNIT, SUB, SUPER)  XSR (SUB, SUPER)

/* Declaration of super references.  */
#define DECL_XSR(SUB, SUPER)  \
     extern void * _sr_ ## SUB ## _ ## SUPER
#define DECL_USR(UNIT, SUB, SUPER)  DECL_XSR (SUB, SUPER)
#define DECL_SR(UNIT, SUB, SUPER)  DECL_XSR (SUB, SUPER)

#endif

#endif
