<copyright> TOO internet ports.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright (C) 1996 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: inet.t,v 1.10 1998/01/05 01:16:54 tiggr Exp $</id>
    </copyright>

/******************** InetAddress ********************/

<doc> An {InetAddress} really is an IPv4 address.  It depends on the
    underlying IPv6 implementation if this class is usable for IPv6
    addresses.  </doc>
implementation class
InetAddress: State, Address

<doc> Return a new instance with the indicated fields.  (The {address}
    will be deallocated upon the death of the newly created address.)
    </doc>
instance (id)
  with (InetHost, pointer, int) (h, a, l)
{
  = [[self alloc] init (h, a, l)];
}

end;

implementation instance
InetAddress
{
  <doc> The host on which this address resides.  </doc>
  InetHost host;

  <doc> The internet address.  </doc>
  pointer address;

  <doc> The length in bytes of the address.  </doc>
  int address_length;
}

<doc> Return the low-level bare address.  </doc>
(pointer, int)
  osAddress
{
  = (address, address_length);
}

extern void
  dealloc;

extern boolean
  equal id other;

extern int
  hash;

<doc> Return the host of this address.  The host is looked up if the
    address was not yet related to a host.  </doc>
InetHost
  host
{
  if (!host)
    host = [InetHost addressed self];

  = host;
}

<doc> Designated initializer.  </doc>
protected id
  init (InetHost, pointer, int) (h, a, l)
{
  (host, address, address_length) = (h, a, l);

  = self;
}

<doc> Output the address in dotted decimal octet notation.  </doc>
extern OutputStream
  write OutputStream s;

end;

/******************** InetHost ********************/

implementation class
InetHost: State, Host
{
  <doc> All internet hosts currently known, keyed on their name(s).  </doc>
  static MutableDictionary hosts_by_name;

  <doc> All internet hosts currently known, keyed on their address(es).  </doc>
  static MutableDictionary hosts_by_addr;

  <doc> The wildcard local host.  </doc>
  static InetHost local_host_any;
}

<doc> Return the host known with the address {addr}.  If the host can be
    found in the cache, no lookup is performed.  </doc>
instance (id)
  addressed InetAddress addr
{
  instance (id) h;

  if (!hosts_by_name)
    [self initialize];
  else
    h = hosts_by_addr[addr];

  if (!h)
    {
      h = [self hostWithAddress addr];
      if (h)
	[self cacheHost h];
    }

  = h;
}

<doc> Add the host {h} to the cache.  </doc>
protected void
  cacheHost instance (id) h
{
  int i, n;

  Array names = [h names];
  for ({i = 0; n = [names length];}; i < n; i++)
    hosts_by_name[names[i]] = h;

  Array addresses = [h addresses];
  for ({i = 0; n = [addresses length];}; i < n; i++)
    hosts_by_addr[addresses[i]] = h;
}

void
  initialize
{
  if (!hosts_by_name)
    {
      hosts_by_name = [MutableDictionary new];
      hosts_by_addr = [MutableDictionary new];
    }
}

void
  load Array arguments
{
  local_host_any = [[self alloc] init ([ObjectArray with "*"],
				       [ObjectArray new])];
}

<doc> Return the host named {name}.  If the host can be found in the
    cache, no lookup is performed.  </doc>
instance (id)
  named String name
{
  instance (id) h;

  if (!hosts_by_name)
    [self initialize];
  else
    h = hosts_by_name[name];

  if (!h)
    {
      h = [self hostWithName name];
      if (h != nil)
	[self cacheHost h];
    }

  = h;
}

<doc> Perform a lookup of the host addressed {addr}.  Return the host, or
    {nil} if it could not be found.  The cache remains unaffected.  </doc>
protected extern instance (id)
  hostWithAddress InetAddress addr;

<doc> Perform a lookup of the host named {name}.  Return the host, or
    {nil} if it could not be found.  The cache remains unaffected.  </doc>
protected extern instance (id)
  hostWithName String name;

<doc> Return a newly allocated host with the names {n} and addresses {a}.
    </doc>
protected instance (id)
  with (Array, Array) (n, a)
{
  = [[self alloc] init (n, a)];
}

end;

implementation instance
InetHost
{
  <doc> The names by which this host is known.  </doc>
  public Array names;

  <doc> The addresses by which this host is known.  </doc>
  public Array addresses;
}

<doc> Designated initializer.  </doc>
id
  init (Array, Array) (n, a)
{
  (names, addresses) = (n, a);

  = self;
}

String
  name
{
  = names[0];
}

end;

/******************** InetPort ********************/

<doc> An {InetPort} is an abstract port on an internet host.  </doc>
implementation class
InetPort: Port, State

<doc> Return a newly created {InetPort} with the {address} and the {port}.
    </doc>
instance (id)
   with int port
     at InetAddress address
{
  = [[self alloc] initWithPort port at address];
}

end;

implementation instance
InetPort
{
  <doc> The address (of the host) at which this port resides.  </doc>
  public InetAddress address;

  <doc> The port on the host.  </doc>
  public int port;
}

boolean
  equal id other
{
  = (self == other || (other != nil
		       && port == [other port]
		       && [address equal [other address]]));
}

int
  hash
{
  = [address hash] ^ port;
}

<doc> Designated initializer.  </doc>
protected id
  initWithPort int p
	    at InetAddress a
{
  (address, port) = (a, p);

  = self;
}

<doc> Output the address in dotted decimal octet notation followed by a
    colon and the port number.  </doc>
OutputStream
  write OutputStream s
{
  = [s print (address, ':', port)];
}

end;

/******************** ConnectedInetPort ********************/

<doc> A {ConnectedInetPort} is a bytestream on a connected TCP socket.
    </doc>
implementation class
ConnectedInetPort: ConnectedPort, InetPort

end;

implementation instance
ConnectedInetPort
{
  <doc> Description of the server to which we connected or from which we
    accepted.  In the latter case, this will be a {ServerInetPort}.  </doc>
  public InetPort server;

  <doc> The peer socket.  Set by invoking {InetPort [self peer]}.  </doc>
  InetPort peer;
}

<doc> Designated initializer.  Connect to the port {p} at the address {a}.
    </doc>
protected extern id
  initWithPort int p
	    at InetAddress a;

<doc> Initialization used by `ConnectedInetPort [ServerInetPort accept]'.
    </doc>
id
  initWithPort int p
	    at InetAddress a
    descriptor int d
	server ServerInetPort s
	  peer InetPort pr
{
  (server, peer) = (s, pr);

  [super initWithPort p at a];
  [super init d];

  = self;
}

<doc> Return the peer port.  </doc>
extern InetPort
  peer;

void
  registerForRead DescriptorReadDelegate d
{
  [[RunLoop current] addDescriptorForRead self delegate d];
}

void
  registerForWrite DescriptorWriteDelegate d
{
  [[RunLoop current] addDescriptorForWrite self delegate d];
}

end;

/******************** ServerInetPort ********************/

<doc> A {ServerInetPort} is a TCP port which is listening for connections
    to accept.  </doc>
implementation class
ServerInetPort: InetPort, ServerPort, Descriptor

end;

implementation instance
ServerInetPort

<doc> Accept a connection on the receiving port, returning a connected
    port.  </doc>
extern ConnectedInetPort
  accept;

<doc> Designated initializer.  Listen on the {port} at the {address}.  If
    the {address} is {nil}, any local address will do; if the port is 0,
    it is assigned by the operating system.  </doc>
protected extern id
  initWithPort int port
	    at InetAddress address;

void
  registerForRead DescriptorReadDelegate d
{
  [[RunLoop current] addDescriptorForRead self delegate d];
}

end;
