/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: toplev.m,v 1.140 1999/08/18 20:42:55 tiggr Exp $  */

#import "global.h"
#import <ctype.h>
#import "OTMAlias.h"
#import "OTMAliasAlias.h"
#import "OTMAnyRefType.h"
#import "OTMAnyType.h"
#import "OTMArgument.h"
#import "OTMAsm.h"
#import "OTMAssignment.h"
#import "OTMBasic.h"
#import "OTMBind.h"
#import "OTMBreak.h"
#import "OTMBuiltinMethod.h"
#import "OTMCast.h"
#import "OTMCatch.h"
#import "OTMClass.h"
#import "OTMCompound.h"
#import "OTMCondExpr.h"
#import "OTMConstant.h"
#import "OTMContinue.h"
#import "OTMCustomMethod.h"
#import "OTMDynamicType.h"
#import "OTMError.h"
#import "OTMExtension.h"
#import "OTMForeignExpr.h"
#import "OTMForeignType.h"
#import "OTMITE.h"
#import "OTMIdentifier.h"
#import "OTMInstance.h"
#import "OTMInvocation.h"
#import "OTMLabel.h"
#import "OTMLocalVar.h"
#import "OTMLoop.h"
#import "OTMMeta.h"
#import "OTMMetaRef.h"
#import "OTMModAssign.h"
#import "OTMNumberCST.h"
#import "OTMObjectVar.h"
#import "OTMOld.h"
#import "OTMRefVar.h"
#import "OTMReturn.h"
#import "OTMSelector.h"
#import "OTMStringCST.h"
#import "OTMTuple.h"
#import "OTMTypeTuple.h"
#import "OTMUnvocation.h"
#import "OTMUnwind.h"
#import "OTMVariable.h"

@class OTMAlias, OTMAliasAlias, OTMAnyRefType, OTMAnyType;
@class OTMArgument, OTMAssignment, OTMBasic, OTMBind;
@class OTMBreak, OTMBuiltinMethod, OTMCast, OTMCatch;
@class OTMClass, OTMCompound, OTMCondExpr, OTMConstant;
@class OTMContinue, OTMCustomMethod, OTMDynamicType;
@class OTMEntity, OTMError, OTMExpr, OTMExtension;
@class OTMForeignExpr, OTMForeignType, OTMIdentifier;
@class OTMInstance, OTMInvocation, OTMITE, OTMLoop;
@class OTMLocalVar, OTMMeta, OTMMetaRef, OTMMethod;
@class OTMModAssign, OTMNumberCST, OTMObjectVar, OTMOld;
@class OTMRefVar, OTMReturn, OTMSelector, OTMStringCST;
@class OTMTop, OTMTuple, OTMType, OTMTypeTuple;
@class OTMUnvocation, OTMUnwind, OTMVariable;

@interface Object (Bogus)

+self;

@end;

id CO_OTMAlias, CO_OTMAliasAlias, CO_OTMAnyRefType, CO_OTMAnyType;
id CO_OTMArgument, CO_OTMAsm, CO_OTMAssignment, CO_OTMBasic, CO_OTMBind;
id CO_OTMBreak, CO_OTMBuiltinMethod, CO_OTMCast, CO_OTMCatch;
id CO_OTMClass, CO_OTMCompound, CO_OTMCondExpr, CO_OTMConstant;
id CO_OTMContinue, CO_OTMCustomMethod, CO_OTMDynamicType;
id CO_OTMEntity, CO_OTMError, CO_OTMExpr, CO_OTMExtension;
id CO_OTMForeignExpr, CO_OTMForeignType, CO_OTMIdentifier;
id CO_OTMInstance, CO_OTMInvocation, CO_OTMITE, CO_OTMLoop;
id CO_OTMLocalVar, CO_OTMMeta, CO_OTMMetaRef, CO_OTMMethod;
id CO_OTMModAssign, CO_OTMNumberCST, CO_OTMObjectVar, CO_OTMOld;
id CO_OTMRefVar, CO_OTMReturn, CO_OTMSelector, CO_OTMStringCST;
id CO_OTMTop, CO_OTMTuple, CO_OTMType, CO_OTMTypeTuple;
id CO_OTMUnvocation, CO_OTMUnwind, CO_OTMVariable, CO_OTMLabel;

/* The name of the unit a file of which we are compiling.  */
id <TLString> top_unit_name;

/* The basename of the file we are to compile.  */
TLString *input_basename;

/* How much space to put in front of a line.  */
int indent_level = 0;

/* Common declaration in case tl was compiled without DEBUG_GC.  */
int debug_gc;

/* Iff !0, the resolver will do single-unit resolution.  */
int flag_1 = 0;

/* Iff !0, conditions handlers are skipped.  */
int flag_suppress_conditions = 0;

/* Iff !0, we should not run GC.  */
int flag_inhibit_gc = 1;

/* Iff !0, do not issue warnings about empty expressions.  */
int flag_inhibit_empty;

/* Iff !0, do not warn about unqualified redeclaration.  */
int flag_inhibit_unqualified_redeclare = 1;

/* Iff !0, do not warn about double-slash comments.  */
int flag_inhibit_comment = 0;

/* The number of times `-v' was specified on the command line.  */
int flag_verbose;

/* Determine how lookups are output, and how lookups to super are output.  */
int flag_lookup, flag_super;

/* Iff !0, the code is compiled for atomic GC.  Otherwise, it can handle
   incremental GC.  */
int flag_atomic_gc;

/* Iff !0, the output files are precious and are not deleted, even when
   errors are detected.  */
int flag_precious;

/* Iff !0, the C output should be readable and debuggable.  */
int flag_readable;

/* Iff 0, do not emit precondition checking.  */
int flag_pre_checks = 1;

/* Iff 0, do not emit postcondition checking.  */
int flag_post_checks = 1;

/* Iff !0, identifiers are traced to ease debugging.  */
int flag_trace_identifiers;

/* Iff !0, use trt_ext_address to extract extension addresses.  */
int flag_check_extension_address;

/* The void, nil and any_type expressions.  */
OTMExpr *void_expr, *nil_expr, *any_type_expr;

/* The dictionary of encountered string constants.  */
TLDictionary *strings;

/* The dictionary of used string constants.  */
TLDictionary *used_strings;

/* The set of uniqued identifiers.  */
TLSet *identifiers;

/* The uniqued `:'.  */
TLString *unique_identifier_colon;

/* The name of the extension fields of the state class and instance.  */
id <TLString> tom_i_state_field_name;
id <TLString> tom_c_state_field_name;

/* The main extension of the Common instance.  */
LTTExtension *common_main_ext;

/* The compound into which old expressions are evaluated for the purpose
   of postcondition checking.  */
static OTMCompound *expr_old_compound;

/* The type matching any other type, and the one matching any other
   reference type.  */
OTMType *the_any_type;
OTMType *the_any_ref_type;

/* Some classes/instances needed by the compiler.  */
OTMInstance *tom_string_instance;
OTMInstance *tom_condition_instance;
OTMInstance *tom_condition_class_instance;

/* Collected pre- and postconditions of the current method.  */
TLVector *current_preconditions, *current_postconditions;

static id <TLString>
types_description1 (id <TLEnumerator> e, int skip, BOOL is_types)
{
  TLMutableString *s = nil;
  OTMExpr *o;

  while ((o = [e nextObject]))
    if (skip)
      skip--;
    else
      {
	OTMType *t = is_types ? (id) o : (id) [o type];

	s = formac (s, s ? @", %@" : @"%@", type_name (t));
      }

  return s ? (id) s : (id) @"void";
}

id <TLString>
types_description (TLVector *v)
{
  return types_description1 ([v enumerator], 0, YES);
}

id <TLString>
elements_type_description (TLVector *v, int skip)
{
  return types_description1 ([v enumerator], skip, NO);
}

OTMExpr *
emit_assignment (OTMExpr *lhs, OTMExpr *rhs)
{
  if (lhs == an_error || !rhs);
  else if (lhs != void_expr && [lhs lhsInvalid])
    error (@"invalid lhs");
  else
    {
      OTMType *lt = [lhs type], *rt = [rhs type];

      if ([lt matchesConvertibly:
	      [rt actualSelf: [current_either semantics]]] < 0)
	error (@"lhs type %@ does not match rhs type %@",
	       type_name (lt), type_name (rt));
      else if (![rhs isTuple])
	if (lhs == void_expr)
	  emit_statement (rhs);
	else
	  emit_statement ([CO_OTMAssignment assignmentWithLhs: lhs
					    rhs: implicit_cast (rhs, lt)]);
      else
	{
	  /* Tuple assignment where the rhs is also a tuple (and not
             something else with a tuple type).  Split it into multiple
             single assignments.  */
	  TLVector *vl = [(OTMTuple *) lhs elements];
	  TLVector *vr = [(OTMTuple *) rhs elements];
	  int i, n = [vl length];

	  for (i = 0; i < n; i++)
	    emit_assignment ([vl _elementAtIndex: i], [vr _elementAtIndex: i]);
	}
    }

  return lhs;
}

OTMExpr *
emit_local_var (OTMVariable *v)
{
  if (current_compound)
    {
#if STACK_REF_STRUCT
      if ([v residesInRefStruct])
	[(OTMCompound *) [current_method body] addVariable: v];
      else
#endif
	[current_compound addVariable: v];
    }

  return v;
}

OTMExpr *
emit_old_expr (OTMExpr *expr)
{
  OTMExpr *e;

  if (!expr_old_compound)
    expr_old_compound = enter_compound ();

  e = emit_expr (expr);
  [current_compound setValue: e];
  return emit_assignment (temp_something_with_type ([e type], NO), e);
}

void
emit_conditions_pre (OTMCustomMethod *m)
{
  /* XXX This is an ugly hack.  */
  static char next;
  int i, n;

  if (!flag_pre_checks && !flag_post_checks)
    return;

  current_preconditions = [m collectPreconditions];
  current_postconditions = [m collectPostconditions];

  if (!current_preconditions && !current_postconditions)
    return;

  if (!next)
    {
      next = 1;
      formac (of, @"extern tom_byte c_tom_Runtime_preconditions_enabled;\n"
	      @"extern tom_byte c_tom_Runtime_postconditions_enabled;\n");
    }

  if (flag_pre_checks && current_preconditions)
    {
      OTMExpr *cond = nil;

      for (i = 0, n = [current_preconditions length]; i < n; i++)
	{
	  OTMExpr *c = [current_preconditions _elementAtIndex: i];

	  cond = (!cond ? c
		  : op_invocation2 (builtin_operator_name[BO_SC_OR], c, cond));
	}

      emit_expr ([CO_OTMCondExpr iteWithCondition:
		  op_invocation2
		    (builtin_operator_name[BO_SC_AND],
		     [[CO_OTMForeignExpr alloc]
		      initWithType: basic_type[BT_BOOLEAN]
		      string: @"c_tom_Runtime_preconditions_enabled"],
		     op_invocation1 (builtin_operator_name[BO_NOT], cond))
		 thenPart:
		   invocation ([current_method argumentNamed: TO_NAME_SELF],
			       unique_identifier (@"preconditionFailed"), 
			       [current_method argumentNamed: TO_NAME_CMD])
		 elsePart: nil]);
    }

  if (flag_post_checks && current_postconditions)
    {
      for (i = 0, n = [current_postconditions length]; i < n; i++)
	{
	  OTMExpr *nc, *c = [current_postconditions _elementAtIndex: i];

	  nc = [c oldsEliminated];
	  if (c != nc)
	    [current_postconditions _replaceElementAtIndex: i by: nc];
	}

      if (expr_old_compound)
	{
	  exit_compound ();

	  emit_expr ([CO_OTMCondExpr
		      iteWithCondition:
			[[CO_OTMForeignExpr alloc]
			 initWithType: basic_type[BT_BOOLEAN]
			 string: @"c_tom_Runtime_postconditions_enabled"]
		      thenPart: expr_old_compound elsePart: nil]);
	  expr_old_compound = nil;
	}
    }
}

void
emit_conditions_post (OTMCustomMethod *m)
{
  if (flag_post_checks && current_postconditions)
    {
      OTMExpr *cond = nil;
      int i, n;

      for (i = 0, n = [current_postconditions length]; i < n; i++)
	{
	  OTMExpr *c = [current_postconditions _elementAtIndex: i];

	  cond = (!cond ? c
		  : op_invocation2 (builtin_operator_name[BO_SC_AND], c, cond));
	}

      emit_expr ([CO_OTMCondExpr iteWithCondition:
		  op_invocation2
		    (builtin_operator_name[BO_SC_AND],
		     [[CO_OTMForeignExpr alloc]
		      initWithType: basic_type[BT_BOOLEAN]
		      string: @"c_tom_Runtime_postconditions_enabled"],
		     op_invocation1 (builtin_operator_name[BO_NOT], cond))
		 thenPart:
		   invocation ([current_method argumentNamed: TO_NAME_SELF],
			       unique_identifier (@"postconditionFailed"), 
			       [current_method argumentNamed: TO_NAME_CMD])
		 elsePart: nil]);
    }
}

id
invocation (OTMExpr *rcv, id name, OTMExpr *args)
{
  return build_invocation (rcv, CONS (CONS (name, args), nil), 0, nil);
}

id
build_invocation (id rcv, TLCons *part,
		  BOOL is_super, OTMMeta *super_confined)
{
  TLVector *args, *name_parts;

  if ([part consp])
    {
      TLCons *np;

      args = [CO_TLVector vector];
      name_parts = [CO_TLVector vector];
      while (part)
	{
	  id <TLString> ne;
	  OTMExpr *arg;

	  DECONS (part, np, part);
	  DECONS (np, ne, arg);

	  [name_parts addElement: ne];
	  [args addElement: arg];
	}
    }
  else
    {
      name_parts = [CO_TLVector vectorWithElements: 1, part];
      args = nil;
    }

  return [CO_OTMUnvocation unvocationWithReceiver: rcv
			sender: [current_either semantics]
			arguments: args nameParts: name_parts
			super: is_super confined: super_confined];
}

id
op_invocation1 (id name, OTMExpr *arg1)
{
  OTMExpr *e = [[CO_OTMExpr gcAlloc] initWithType:
				  [[current_either itsClass] semantics]];

  return build_invocation (e, CONS (CONS (name, arg1), nil), 0, nil);
}

id
op_invocation2 (id name, OTMExpr *arg1, OTMExpr *arg2)
{
  OTMExpr *e = [[CO_OTMExpr gcAlloc] initWithType:
				  [[current_either itsClass] semantics]];

  return build_invocation (e,
			   CONS (CONS (name, arg1),
				 CONS (CONS (unique_identifier_colon, arg2),
				       nil)), 0, nil);
}

OTMExpr *
temp_something_with_type (OTMType *t, BOOL reuse)
{
  if (!t)
    return [CO_OTMError sharedError];

  if ([t isTuple])
    return temp_tuple_with_type ((id) t, reuse);

  return temp_var_with_type ((id) t, reuse);
}

OTMTuple *
temp_tuple_with_type (OTMTypeTuple *t, BOOL reuse)
{
  TLVector *type_elts = [t elements];
  int i, n = [type_elts length];
  TLVector *v = [CO_TLVector vectorWithCapacity: n];
  OTMTuple *ret;

  for (i = 0; i < n; i++)
    [v addElement:
     temp_something_with_type ([type_elts _elementAtIndex: i], reuse)];

  ret = [CO_OTMTuple tupleWithSequence: v];
  [ret setType: t];
  [ret setIsTemporary];
  return ret;
}

OTMVariable *
temp_var_with_type (OTMType *t, BOOL reuse)
{
  OTMVariable *v;

  if (!current_compound)
    return [CO_OTMLocalVar temporaryVariableWithType: t];

  if (reuse)
    {
#if STACK_REF_STRUCT
      if ([t isObjectType])
	return [(OTMCompound *) [current_method body] temporaryWithType: t];
      else
#endif
	return [current_compound temporaryWithType: t];
    }

  v = [CO_OTMLocalVar temporaryVariableWithType: t];
  [(OTMCompound *) [current_method body] addVariable: v];
  return v;
}

OTMExpr *
nil_something_with_type (OTMType *t)
{
  if (![t isTuple])
    return [[CO_OTMForeignExpr gcAlloc] initWithType: t string: @"0"];

  {
    TLVector *type_elts = [(OTMTypeTuple *) t elements];
    int i, n = [type_elts length];
    TLVector *v = [CO_TLVector vectorWithCapacity: n];
    OTMTuple *ret;

    for (i = 0; i < n; i++)
      [v addElement:
	 nil_something_with_type ([type_elts _elementAtIndex: i])];

    ret = [CO_OTMTuple tupleWithSequence: v];
    [ret setType: t];
    return ret;
  }
}

void
emit_statement (id s)
{
  if (s && current_compound)
    [current_compound addStatement: s];
}

OTMExpr *
emit_expr (OTMExpr *e)
{
  e = [resolve_expr (e, nil, nil, [current_either semantics]) elaborate];
  emit_statement (e);
  return [e value];
}

id <TLString>
qualifier_name (otm_qualifiers q)
{
  id <TLString> rv = nil;

  if (Q_STATIC (q))
    {
      rv = @"static";
      q = Q_SET_STATIC (q, 0);
    }
  else if (Q_DEFERRED (q))
    {
      rv = @"deferred";
      q = Q_SET_DEFERRED (q, 0);
    }
  else if (Q_MUTABLE (q))
    {
      rv = @"mutable";
      q = Q_SET_MUTABLE (q, 0);
    }
  else if (Q_REDO (q) == OQ_REDEFINE)
    {
      rv = @"redefine";
      q = Q_SET_REDO (q, 0);
    }
  else if (Q_REDO (q) == OQ_REDECLARE)
    {
      rv = @"redeclare";
      q = Q_SET_REDO (q, 0);
    }
  else if (Q_PROTECTION (q) == OQ_PUBLIC)
    {
      rv = @"public";
      q = Q_SET_PROTECTION (q, 0);
    }
  else if (Q_PROTECTION (q) == OQ_PROTECTED)
    {
      rv = @"protected";
      q = Q_SET_PROTECTION (q, 0);
    }
  else if (Q_PROTECTION (q) == OQ_PRIVATE)
    {
      rv = @"private";
      q = Q_SET_PROTECTION (q, 0);
    }
  else if (!q)
    rv = @"none";

  if (!rv || q)
    ABORT ();

  return rv;
}

otm_qualifiers
mask_qualifiers (otm_qualifiers value, otm_qualifiers mask)
{
  static otm_qualifiers masks[] = {OQ_STATIC_MASK, OQ_DEFERRED_MASK,
				   OQ_REDO_MASK, OQ_PROTECTION_MASK,
				   OQ_MUTABLE_MASK};
  int i;

  for (i = 0; i < sizeof (masks) / sizeof (masks[0]); i++)
    if (MASK_SELECT (value, masks[i]) && ! MASK_SELECT (mask, masks[i]))
      {
	warning (@"inapplicable %@ qualifier ignored",
		 qualifier_name (MASK_SELECT (value, masks[i])));
	value = MASK_SET (value, masks[i], 0);
      }

  return value;
}

id <TLString>
accessor_method_name (id <TLString> name)
{
  return unique_identifier (formac (nil, @"%@", name));
}

id <TLString>
modifier_method_name (id <TLString> name)
{
  return unique_identifier (formac (nil, @"set_%@", name));
}

id
method_name (OTMMethod *m, BOOL print_meta)
{
  id s = formac (nil, @"<%@> `%@' <%@>", type_name ([m returnType]),
		 [m methodName], elements_type_description ([m arguments],
						      [m implicitArguments]));

  if (print_meta)
    formac (s, @" implemented by %@",
	    ltt_meta_name ([[[m extension] structure] meta]));

  return s;
}

id
method_name1 (id name, id args)
{
  return formac (nil, @"named `%@' with argument types <%@>",
		 name, elements_type_description (args, 0));
}

id
method_name2 (id parts, id args)
{
  id <TLEnumerator> ae = [args enumerator];
  id <TLEnumerator> pe = [parts enumerator];
  id s = nil, p, a;
  id leading = @"";

  while ((p = [pe nextObject]))
    {
      a = [ae nextObject];

      if (a)
	s = formac (s, @"%@`%@' <%@>", leading, p, type_name ([a type]));
      else
	s = formac (s, @"%@`%@'", leading, p, type_name ([a type]));

      leading = @" ";
    }

  return s;
}

id
type_name (OTMType *c)
{
  if ([c isObjectType] && ![c isKindOf: [CO_OTMBasic class]])
    if ([c isKindOf: [CO_OTMAnyRefType self]])
      return @"any reference type";
    else
      return ltt_meta_name ([(OTMMeta *) c structure]);
  else if ([c isTuple])
    {
      id e = [(OTMTypeTuple *) c elements];
      id s = formac (nil, @"(");
      int i, n = [(TLVector *) e length];

      for (i = 0; i < n; i++)
	formac (s, i ? @", %@" : @"%@", type_name ([e _elementAtIndex: i]));

      return formac (s, @")");
    }

  return formac (nil, @"%@", [c typeName]);
}

id <TLString>
type_name_list (TLCons *l)
{
  OTMType *t;
  id s = nil;
  int i = 0;

  while (l)
    {
      DECONS (l, t, l);

      s = formac (s, @"%@%@",
		  (!i ? @"" : (l ? @", " : (i > 1 ? @", and " : @" and "))),
		  type_name (t));
      i++;
    }

  return s;
}

OTMExpr *
implicit_cast (OTMExpr *expr, OTMType *desired_type)
{
  if (desired_type)
    {
      OTMType *t = [expr type];

      if (t)
	{
	  t = [t actualSelf: [current_either semantics]];
	  desired_type = [desired_type actualSelf: [current_either semantics]];

	  if (t != desired_type
	      && [t isKindOf: [CO_OTMMeta self]]
	      && [desired_type isKindOf: [CO_OTMMeta self]])
	    expr = [CO_OTMCast castWithExpr: expr type: desired_type];
	}
    }

  return expr;
}

OTMExpr *
resolve_expr (OTMExpr *expr, TLCons *expected,
	      OTMType *convertible, OTMType *cxt)
{
  TLCons *t;

  if (expr == an_error)
    return expr;

  if (!expr)
    ABORT ();

  t = [expr resolveWithExpected: expected convertible: convertible
	    context: cxt indices: NULL index: -1];

  if (t && ![expr type] && !expected && !convertible)
    {
      /* Force _some_ type if the expression doesn't care and neither does
         the caller.  Prefer the void type if present.  */
      TLCons *vt = [t memq: basic_type[BT_VOID]];

      t = [expr resolveWithExpected: CONS (vt ? [vt car] : [t car], nil)
		convertible: nil context: cxt indices: NULL index: -1];
    }

  if (!t || ![expr type])
    {
      error (@"unable to deduce type of expression");
      if (t)
	cerror_for (expr, @"with possible types %@", type_name_list (t));
      if (expected)
	cerror_for (expr, @"in context of possible types: %@",
		    type_name_list (expected));
      else if (convertible)
	cerror_for (expr, @"in context of cast to %@", type_name (convertible));
      else
	cerror_for (expr, @"in void context");
    }

  return expr;
}

TLCons *
default_resolve_expr (OTMType *type, TLCons *expected, OTMType *to,
		      OTMType *cxt, int *indices, int index)
{
  if (!type)
    return nil;

  type = [type actualSelf: cxt];

  if (!expected)
    return CONS (type, nil);

  {
    TLCons *possible = nil, *last = nil;
    OTMType *t;

    while (expected)
      {
	OTMType *s;

	DECONS (expected, t, expected);
	s = (t == the_dynamic_type) ? type : [t typeAt: index in: indices];
	if (s && (s == type || [s matchesConvertibly: type] >= 0))
	  {
	    TLCons *c = CONS (t, nil);

	    if (possible)
	      [last setCdr: c];
	    else
	      possible = c;

	    last = c;
	  }
      }

    return possible;
  }
}

TLCons *
types_intersect (TLCons *a, TLCons *b, TLCons *c)
{
  TLCons *is = nil, *last = nil;

  if (!c)
    while (a)
      {
	id aa, bb, ab;

	DECONS (a, aa, a);
	bb = b;
	while (bb)
	  {
	    OTMType *tp;

	    DECONS (bb, ab, bb);

	    tp = [aa matchesExactly: ab];
	    if (tp)
	      {
		TLCons *e = CONS (tp, nil);

		/* XXX Remove AB from B.  */

		if (last)
		  [last setCdr: e];
		else
		  is = e;

		last = e;
		break;
	      }
	  }
      }
  else
    {
      while (a && b && c)
	{
	  id aa, ab, ac;

	  DECONS (c, ac, c);

	  if ([ac equal: [a car]])
	    {
	      DECONS (a, aa, a);
	      if (ac == [b car])
		{
		  TLCons *e = CONS (ac, nil);

		  DECONS (b, ab, b);

		  if (last)
		    [last setCdr: e];
		  else
		    is = e;

		  last = e;
		}
	    }
	  else if ([ac equal: [b car]])
	    DECONS (b, ab, b);
	}

      if (!c && (a || b))
	ABORT ();
    }

  return is;
}

BOOL
types_equal (TLCons *a, TLCons *b, TLCons *c)
{
  if (!c)
    ABORT ();

  while (a && b)
    {
      id aa, ab;

      DECONS (a, aa, a);
      DECONS (b, ab, b);
      if (![aa equal: ab])
	return NO;
    }

  return !a && !b;
}

TLCons *
types_add_element (OTMType *t, TLCons *a)
{
  TLCons *b = a;

  while (b)
    {
      OTMType *s, *s2;

      DECONS (b, s, b);

      if ([s equal: t])
	return a;
    }

  return CONS (t, a);
}

TLCons *
types_add_elt_ordered (OTMType *t, TLCons *a, TLCons *c)
{
  TLCons *b = a, *bprev = nil;
  OTMType *aelt, *celt;

  if (!c)
    ABORT ();

  while (c)
    {
      DECONS (c, celt, c);

      aelt = [b car];

      if ([aelt equal: t])
	{
	  /* T has already been handled, as it already is in A.  */
	  return a;
	}

      if ([celt equal: aelt])
	{
	  /* This elt of C already is in A.  */
	  bprev = b;
	  b = [b cdr];
	}
      else if ([celt matchesConvertibly: t] >= 0)
	{
	  /* This element from C matches and must be inserted into A.  */
	  TLCons *ins = CONS (celt, b);

	  if (bprev)
	    [bprev setCdr: ins];
	  else
	    a = ins;

	  bprev = ins;
	}
    }

  return a;
}

OTMType *
types_element_of (OTMType *t, TLCons *a, OTMType *context,
		  int *indices, int index, BOOL arg_not_proto)
{
  BOOL ot = [t isObjectType];
  OTMType *match = nil;
  int level = 1000000;

  while (a)
    {
      OTMType *s, *as;
      int i;

      DECONS (a, s, a);
      as = [s typeAt: index in: indices];

      /* If it is the identical type, return it immediately; a more closer
         match is impossible.  */
      if (as == t)
	return s;

      if (!as)
	continue;

      i = (arg_not_proto ? [as matchesConvertibly: t]
	   : [t matchesConvertibly: as]);
      if (i >= 0 && (!match || i < level))
	{
	  if (ot)
	    {
	      /* Pick the most specific class.  */
	      OTMMeta *mt = (id) [t actualSelf: context];
	      OTMMeta *mas = (id) [as actualSelf: context];

	      if (mas == (id) the_any_ref_type || mas == (id) the_dynamic_type)
		{
		  if (match)
		    continue;
		}
	      else if ([[mas structure] isProperSub: [mt structure]])
		if (match)
		  {
		    OTMMeta *mm = (id) [match typeAt: index in: indices];

		    if (![[mm structure] isProperSub: [mas structure]])
		      continue;
		  }
	    }

	  match = s;
	  level = i;
	}
    }

  return match;
}

TLString *
unique_identifier (id <TLString> s)
{
  TLString *u = [identifiers member: s];

  if (u)
    s = u;
  else
    [identifiers addElement: s];

  return s;
}

void
otm_init (int argc, char **argv)
{
  /* Obtain -ObjC semantics for non-NeXT systems.  */
  extern int get_patches;
  get_patches = 0;

  CO_OTMAlias = [OTMAlias self];
  CO_OTMAliasAlias = [OTMAliasAlias self];
  CO_OTMAnyRefType = [OTMAnyRefType self];
  CO_OTMAnyType = [OTMAnyType self];
  CO_OTMArgument = [OTMArgument self];
  CO_OTMAsm = [OTMAsm self];
  CO_OTMAssignment = [OTMAssignment self];
  CO_OTMBasic = [OTMBasic self];
  CO_OTMBind = [OTMBind self];
  CO_OTMBreak = [OTMBreak self];
  CO_OTMBuiltinMethod = [OTMBuiltinMethod self];
  CO_OTMCast = [OTMCast self];
  CO_OTMCatch = [OTMCatch self];
  CO_OTMClass = [OTMClass self];
  CO_OTMCompound = [OTMCompound self];
  CO_OTMCondExpr = [OTMCondExpr self];
  CO_OTMConstant = [OTMConstant self];
  CO_OTMContinue = [OTMContinue self];
  CO_OTMCustomMethod = [OTMCustomMethod self];
  CO_OTMDynamicType = [OTMDynamicType self];
  CO_OTMEntity = [OTMEntity self];
  CO_OTMError = [OTMError self];
  CO_OTMExpr = [OTMExpr self];
  CO_OTMExtension = [OTMExtension self];
  CO_OTMForeignExpr = [OTMForeignExpr self];
  CO_OTMForeignType = [OTMForeignType self];
  CO_OTMIdentifier = [OTMIdentifier self];
  CO_OTMInstance = [OTMInstance self];
  CO_OTMInvocation = [OTMInvocation self];
  CO_OTMITE = [OTMITE self];
  CO_OTMLoop = [OTMLoop self];
  CO_OTMLocalVar = [OTMLocalVar self];
  CO_OTMMeta = [OTMMeta self];
  CO_OTMMetaRef = [OTMMetaRef self];
  CO_OTMMethod = [OTMMethod self];
  CO_OTMModAssign = [OTMModAssign self];
  CO_OTMNumberCST = [OTMNumberCST self];
  CO_OTMObjectVar = [OTMObjectVar self];
  CO_OTMOld = [OTMOld self];
  CO_OTMRefVar = [OTMRefVar self];
  CO_OTMReturn = [OTMReturn self];
  CO_OTMSelector = [OTMSelector self];
  CO_OTMStringCST = [OTMStringCST self];
  CO_OTMTop = [OTMTop self];
  CO_OTMTuple = [OTMTuple self];
  CO_OTMType = [OTMType self];
  CO_OTMTypeTuple = [OTMTypeTuple self];
  CO_OTMUnvocation = [OTMUnvocation self];
  CO_OTMUnwind = [OTMUnwind self];
  CO_OTMVariable = [OTMVariable self];
  CO_OTMLabel = [OTMLabel self];

  ltt_init (argc, argv);

  extensions_output = [TLSet set];
  [extensions_output gcLock];

  strings = [TLDictionary dictionary];
  [strings gcLock];

  used_strings = [TLDictionary dictionary];
  [used_strings gcLock];

  identifiers = [TLSet set];
  [identifiers gcLock];

  unique_identifier_colon = unique_identifier (@":");

  [CO_OTMError init];
}

/* Find and read the tom unit definition (needed for the State class).
   After that, initialize the otm classes.  */
static void
otm_read_tom (void)
{
  OTMInstance *inst_common;
  struct
  {
    id <TLString> unit_name, instance_name;
    OTMInstance **inst;
  } instances[] =
    {
      {TOM_UNIT_NAME_TOM, TOM_CLASS_NAME_STRING, &tom_string_instance},
      {TOM_UNIT_NAME_TOM, TOM_CLASS_NAME_COMMON, &inst_common},
      {TOM_UNIT_NAME_TOM, TOM_CLASS_NAME_CONDITION, &tom_condition_instance},
      {TOM_UNIT_NAME_TOM, TOM_CLASS_NAME_CONDITION_CLASS,
       &tom_condition_class_instance},
      {0, 0, 0}
    };
  int i;

  ltt_finish_init ();

  for (i = 0; instances[i].unit_name; i++)
    {
      LTTUnit *u = [CO_LTTUnit unitNamed: instances[i].unit_name];
      LTTInstance *inst;

      inst = u ? [u instanceNamed: instances[i].instance_name] : nil;

      if (!inst)
	error (@"Can't find instance %@.%@", instances[i].unit_name,
	       instances[i].instance_name);
      else
	*instances[i].inst = [inst semantics];
    }

  common_main_ext = [[inst_common structure] extensionNamed: nil];

  tom_i_state_field_name = [ltt_ext_i_state outputExtensionFieldName];
  [tom_i_state_field_name gcLock];
  tom_c_state_field_name = [ltt_ext_c_state outputExtensionFieldName];
  [tom_c_state_field_name gcLock];

  [CO_OTMBasic init];

  void_expr = [[CO_OTMExpr gcAlloc] initWithType: basic_type[BT_VOID]];
  [void_expr gcLock];

  [CO_OTMMeta init];

  the_any_type = [[CO_OTMAnyType gcAlloc] init];
  [the_any_type gcLock];
  the_any_ref_type = [[CO_OTMAnyRefType gcAlloc] init];
  [the_any_ref_type gcLock];

  [CO_OTMBuiltinMethod init];
  [CO_OTMDynamicType init];

  nil_expr = [[CO_OTMForeignExpr gcAlloc]
	      initWithType: [ltt_instance_all semantics] string: @"0"];
  [nil_expr gcLock];

  any_type_expr = [[CO_OTMExpr gcAlloc] initWithType: the_any_type];
  [any_type_expr gcLock];
}

void
usage (int rc, char *s)
{
  if (s)
    formac (V_stderr_, @"%@: bad option/argument: %s\n", prog, s);

  formac (rc || s ? V_stderr_ : V_stdout_, @"otmc (%@), version %s\n\
usage: %@ options input [ output [ info ] ]\n\
options:\n\
 -dgc <n>	if n < 0, inhibit garbage collection\n\
		if n >= 0, set gc debugging to N\n\
 -h -?		print help on stdout and exit\n\
 -v		increase verbosity\n\
 --version	print version on stdout and exit\n\
 -u <unit>	name the UNIT to use (mandatory)\n\
 -1             generate code for single-unit resolution\n\
 -f		modify behaviour.  Flags currently implemented are:\n\
		-flookup-direct, -flookup-lookup, -flookup-send\n\
		-fatomic-gc -fincremental-gc\n\
		-freadable-c -ftrace-identifiers\n\
                -fno-checks -fno-post-checks -fno-pre-checks\n\
                -fcheck-extension-address -fno-conditions\n\
 -I <path>	add elements from the colon separated PATH to the load path\n\
 -precious	don't unlink output files in case of errors\n\
 -Wno-empty	do not warn about empty expression\n\
 -Wno-comment   do not warn about double-slash comments\n\
 -Wno-redeclare	... about unqualified redeclaration of undeferred method\n\
", prog, long_version, prog);

  exit (rc);
}

static void
dump_info (id <TLOutputStream> inf)
{
  id <TLEnumerator> e;
  OTMExtension *ext;
  LTTStringCST *str;
  LTTSelector *sel;

  formac (inf, @"((file-version %d)\n(otmc-version \"%s\")",
	  TOM_INFO_FILE_VERSION, short_version);

  formac (inf, @"\n(compile-date %#)\n", formac (nil, @"%@", [TLDate now]));

  formac (inf, @"(selectors");
  e = [CO_LTTSelector selectors];
  while ((sel = [e nextObject]))
    formac (inf, @"\n (%# %# %#)", [[sel lttName] internal],
	    [sel inArgumentTypes], [sel outArgumentTypes]);
  formac (inf, @")\n");

  [CO_OTMDynamicType dumpInfo: inf];

  formac (inf, @"(strings");
  e = [used_strings valueEnumerator];
  while ((str = [e nextObject]))
    formac (inf, @"\n (%# %#)", [str string], [[str lttName] internal]);
  formac (inf, @")");

  e = [extensions_output enumerator];
  while ((ext = [e nextObject]))
    [ext dumpInfo: inf];

  formac (inf, @")\n");
}

int
main (int argc, char **argv)
{
  int i, n, input = 0, output = 0, info = 0;
  id inf = nil;
  LTTUnit *unit;
  char *opt;

  otm_init (argc, argv);

  for (i = 1; i < argc; i++)
    if (argv[i][0] == '-')
      {
	BOOL ok = YES;

	switch (argv[i][1])
	  {
	  case '-':
	    if (!strcmp (argv[i], "--version"))
	      {
		if (flag_verbose)
		  printf ("tomc, version %s\n", long_version);
		else
		  printf ("%s\n", short_version);
		exit (0);
	      }
	    break;

	  case '1':
	    if (!strcmp (argv[i], "-1"))
	      flag_1 = 1;
	    else
	      ok = NO;
	    break;

	  case 'd':
	    if (!strncmp (argv[i], "-dgc", 4))
	      {
		opt = get_arg (&i, 4, argv, argc);
		if (opt)
		  {
		    int i = atoi (opt);
		    if (i < 0)
		      flag_inhibit_gc = 1;
		    else
		      {
			flag_inhibit_gc = 0;
			debug_gc = i;
		      }
		  }
	      }
	    else
	      ok = NO;
	    break;

	  case 'f':
	    if (!strcmp (argv[i], "-fatomic-gc"))
	      flag_atomic_gc = 1;
	    else if (!strcmp (argv[i], "-fchecks"))
	      flag_pre_checks = flag_post_checks = 1;
	    else if (!strcmp (argv[i], "-fconditions"))
	      flag_suppress_conditions = 0;
	    else if (!strcmp (argv[i], "-fincremental-gc"))
	      flag_atomic_gc = 0;
	    else if (!strcmp (argv[i], "-flookup-direct"))
	      flag_lookup = LOOKUP_DIRECT;
	    else if (!strcmp (argv[i], "-flookup-lookup"))
	      flag_lookup = LOOKUP_LOOKUP;
	    else if (!strcmp (argv[i], "-flookup-send"))
	      flag_lookup = LOOKUP_SEND;
	    else if (!strcmp (argv[i], "-fno-checks"))
	      flag_post_checks = flag_pre_checks = 0;
	    else if (!strcmp (argv[i], "-fno-conditions"))
	      flag_suppress_conditions = 1;
	    else if (!strcmp (argv[i], "-fno-post-checks"))
	      flag_post_checks = 0;
	    else if (!strcmp (argv[i], "-fno-pre-checks"))
	      flag_pre_checks = 0;
	    else if (!strcmp (argv[i], "-fpost-checks"))
	      flag_post_checks = 1;
	    else if (!strcmp (argv[i], "-fpre-checks"))
	      flag_pre_checks = 1;
	    else if (!strcmp (argv[i], "-freadable-c"))
	      flag_readable = 1;
	    else if (!strcmp (argv[i], "-fsuper-direct"))
	      flag_super = LOOKUP_DIRECT;
	    else if (!strcmp (argv[i], "-fsuper-lookup"))
	      flag_super = LOOKUP_LOOKUP;
	    else if (!strcmp (argv[i], "-ftrace-identifiers"))
	      flag_trace_identifiers = 1;
	    else if (!strcmp (argv[i], "-fcheck-extension-address"))
	      flag_check_extension_address = 1;
#if 0
	    /* send_super can not work.  */
	    else if (!strcmp (argv[i], "-fsuper-send"))
	      flag_super = LOOKUP_SEND;
#endif
	    else
	      ok = NO;
	    break;

	  case 'W':
	    if (!strcmp (argv[i], "-Wno-empty"))
	      flag_inhibit_empty = 1;
	    else if (!strcmp (argv[i], "-Wempty"))
	      flag_inhibit_empty = 0;
	    else if (!strcmp (argv[i], "-Wno-redeclare"))
	      flag_inhibit_unqualified_redeclare = 1;
	    else if (!strcmp (argv[i], "-Wredeclare"))
	      flag_inhibit_unqualified_redeclare = 0;
	    else if (!strcmp (argv[i], "-Wcomment"))
	      flag_inhibit_comment = 0;
	    else if (!strcmp (argv[i], "-Wno-comment"))
	      flag_inhibit_comment = 1;
	    else
	      ok = NO;
	    break;

	  case 'h':
	  case '?':
	    /* Allow any option starting with a `h' or `?' to be an
	       indication of the need for help.  */
	    usage (0, NULL);
	    break;

	  case 'p':
	    if (!strcmp (argv[i], "-precious"))
	      flag_precious = 1;
	    else
	      ok = NO;
	    break;

	  case 'u':
	    opt = get_arg (&i, 2, argv, argc);
	    if (opt)
	      top_unit_name = [CO_TLString stringWithCString: opt];
	    break;

	  case 'v':
	    if (!strcmp (argv[i], "-v"))
	      flag_verbose++;
	    else
	      ok = NO;
	    break;

	  case 'I':
	    opt = get_arg (&i, 2, argv, argc);
	    if (opt)
	      add_to_load_path ([[CO_TLString stringWithCString: opt]
				 componentsSeparatedByString: @":"]);
	    break;

	  default:
	    ok = NO;
	    break;
	  }

	if (!ok)
	  error (@"bad option: %s", argv[i]);
      }
    else if (!input)
      input = i;
    else if (!output)
      output = i;
    else if (!info)
      info = i;
    else
      error (@"%s: filename arguments already specified", argv[i]);

  if (!input)
    usage (1, NULL);

  if (!flag_lookup)
    flag_lookup = LOOKUP_DEFAULT;
  if (!flag_super)
    flag_super = LOOKUP_LOOKUP;

  if (!NUM_ERRORS && !top_unit_name)
    error (@"missing unit name");
  if (NUM_ERRORS)
    return 1;

  add_to_load_path ([TLVector vectorWithElements: DEFAULT_LOAD_PATH]);

  /* Finish initialization.  */
  otm_read_tom ();
  unit = ltt_find_unit (top_unit_name);
  if (!unit)
    return 1;
  if (flag_1)
    ltt_current_unit = unit;

  if (!NUM_ERRORS)
    {
      TLString *input_name = [CO_TLString stringWithCString: argv[input]];
      LTTFile *file = [unit fileNamed: input_name];

      input_basename = ltt_filename_without_extension ([input_name basename]);
      if (!file)
	error (@"unit `%@' has no file `%@'", [[unit lttName] internal],
	       input_name);
      else
	{
	  TLString *output_name
	    = (output ? [CO_TLString stringWithCString: argv[output]]
	       : [CO_TLString stringWithFormat: @"%@.c", input_basename]);
	  TLString *info_name
	    = (info ? [CO_TLString stringWithCString: argv[info]]
	       : [CO_TLString stringWithFormat: @"%@.%@", input_basename,
			      TOM_INFORMATION_SUFFIX]);

	  [input_basename gcLock];
	  [output_name gcLock];
	  [info_name gcLock];

	  TARGET_PATCH_FLAGS;

	  {
	    TLVector *items = [file items];

	    for (i = 0, n = [items length]; i < n; i++)
	      {
		LTTExtension *x = [items _elementAtIndex: i];
		LTTMeta *m = [x meta];

		if ([m classp])
		  m = [(LTTClass *) m instance];

		load_interface ((LTTInstance *) m);
	      }
	  }

	  /* It is highly unlikely that the interface has not yet been
             loaded... */
	  if (![file loadedInterface])
	    load_file (file, nil, YES, YES);

	  if (!NUM_ERRORS)
	    {
	      if (postponed_behaviour_supers || postponed_state_supers)
		do_postponed_supers (1);

	      if (postponed_state_supers || postponed_behaviour_supers)
		internal (@"postponed supers: %# %#",
			  postponed_state_supers, postponed_behaviour_supers);

	      if (postponed_aliases)
		do_postponed_aliases (1);

	      if (postponed_aliases)
		internal (@"postponed aliases: %#", postponed_aliases);

	      [CO_OTMMeta resolveIdentifiers];
	      if (num_identifiers)
		{
		  extern TLSet *ident_trace_objects;

		  if (!ident_trace_objects)
		    error (@"use -ftrace-identifiers for useful information");
		  else
		    error (@"unresolved identifiers: %@", ident_trace_objects);

		  internal (@"not all identifiers resolved (%d remaining)",
			    num_identifiers);
		}

	      if (!NUM_ERRORS)
		{
		  of = open_output_file (output_name);
		  inf = open_output_file (info_name);
		  if (!of || !inf)
		    return 1;

		  load_file (file, nil, NO, YES);
		  if (!NUM_ERRORS)
		    dump_info (inf);
		}
	    }

	  [(id) of close];
	  of = nil;

	  [(id) inf close];
	  inf = nil;

	  if (NUM_ERRORS && !flag_precious)
	    {
	      if (unlink ([output_name cString]) && errno != ENOENT)
		error (@"%@: %s", output_name, strerror (errno));
	      if (unlink ([info_name cString]) && errno != ENOENT)
		error (@"%@: %s", info_name, strerror (errno));
	    }

	  if (flag_verbose)
	    formac (V_stderr_,
		    @"total %d lines, %d error%s, %d warning%s\n",
		    total_num_lines,
		    total_num_errors, total_num_errors == 1 ? "" : "s",
		    total_num_warnings, total_num_warnings == 1 ? "" : "s");
	}
    }

  return !!NUM_ERRORS;
}
