<copyright> Losing objects in an archive.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1996, 1997 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: archiving.t,v 1.18 1998/05/08 10:12:15 tiggr Exp $</id>
    </copyright>

/******************** StreamEncoder ********************/

implementation class
StreamEncoder: Encoder

end;

implementation instance
StreamEncoder
{
  <doc> The stream to which we write.  </doc>
  OutputStream stream;
}

id
  initWithStream OutputStream s
{
  stream = s;

  = [super init];
}

protected State
  replacementObjectFor State object
{
  = [object replacementForStreamCoder self];
}

end;

/******************** StreamDecoder ********************/

implementation class
StreamDecoder: Decoder

end;

implementation instance
StreamDecoder
{
  <doc> The stream from which we read.  </doc>
  InputStream stream;
}

<doc> Designated initializer.  </doc>
id
  initWithStream InputStream s
{
  stream = s;
  = [super init];
}

end;

/******************** BinaryStreamEncoder ********************/

implementation class
BinaryStreamEncoder: StreamEncoder, BinaryEncoder

end;

implementation instance
BinaryStreamEncoder

<doc> Designated initializer.  </doc>
id
  initWithStream OutputStream s
{
  [super (BinaryEncoder) init];

  = [super initWithStream s];
}

<doc> Finish the graph.  </doc>
void
  finishEncodingRoot All object
{
  [stream flushOutput];
}

protected void
  writeByte byte b
{
  [stream write b];
}

protected void
  writeBytes (int, int) (start, length)
	from ByteArray r
{
  int written = [stream writeRange (start, length) from r];
  if (written != length)
    [[SelectorCondition for self class coding-condition
      message [[MutableByteString new]
	       print ("wrote ", written, " instead of ", length)]
      selector cmd] raise];
}

protected void
  writeBytes (pointer, int) (address, length)
{
  [stream writeBytes length from address];
}

end;

/******************** BinaryStreamDecoder ********************/

implementation class
BinaryStreamDecoder: StreamDecoder, BinaryDecoder, C

end;

implementation instance
BinaryStreamDecoder

<doc> Designated initializer.  </doc>
id
  initWithStream InputStream s
{
  [super (BinaryDecoder) init];

  = [super initWithStream s];
}

protected byte
  readByte
{
  = [stream read];
}

protected void
  readBytes int num
	 to pointer address
{
  // Slow.
  // Sun Dec 29 20:41:23 1996, tiggr@tricky.es.ele.tue.nl
  MutableByteArray buf = [MutableByteArray withCapacity num];
  int read = [stream readRange (0, num) into buf];

  if (read != num)
    [[SelectorCondition for self class coding-condition
      message [[MutableByteString new]
	       print ("read ", read, " instead of ", num)]
      selector cmd] raise];

  pointer from;
  (from, ) = [buf pointerToElements (0, -1)];

  memcpy (address, from, num);
}

end;

/******************** TextStreamEncoder ********************/

implementation class
TextStreamEncoder: StreamEncoder

end;

implementation instance
TextStreamEncoder

<doc> Output the top of the graph.  </doc>
void
  startEncodingRoot All object
{
  [stream print ("(graph (version ", [Coder version], ")")];
}

<doc> Finish the graph.  </doc>
void
  finishEncodingRoot All object
{
  [[stream print ')'] flushOutput];
}

<doc> Output the start of the {object}.  </doc>
class (State)
  startEncoding State object
{
  class (State) objclass = [super startEncoding object];
  Number n = perm_objects_done[object];

  if (!n)
    n = tmp_objects_done[object];

  [stream print (" (", perm_objects_done[objclass], " ", n)];
}

<doc> Finish the output of the {object}.  </doc>
void
  finishEncoding State object
{
  [stream print ')'];
}

<doc> Identify this class on the output {stream}, reporting its coding
    version.  </doc>
protected int
  identityForClass class (State) a_class
{
  int v = [super identityForClass a_class];

  [stream print (" (class ", v, " ", [a_class name],
		 " (version ", [a_class version], "))")];

  = v;
}

<doc> Output {()}, which is the notation for the {nil} object.  </doc>
void
  encodeNilObject
{
  [stream print " ()"];
}

void
  encodeReference int v
{
  [stream print (" (ref ", v, ")")];
}

void
  encode boolean v
{
  [stream print (" (boolean ", v, ")")];
}

void
  encode byte v
{
  [stream print (" (byte ", int (v), ")")];
}

void
  encode char v
{
  [stream print (" (char ", v, ")")];
}

void
  encode int v
{
  [stream print (" (int ", v, ")")];
}

void
  encode long v
{
  [stream print (" (long ", v, ")")];
}

void
  encode float v
{
  [stream print (" (float ", v, ")")];
}

void
  encode double v
{
  [stream print (" (double ", v, ")")];
}

void
  encode selector v
{
  [stream print (" (selector ", [Runtime nameOfSelector v], ")")];
}

void
  encodeBytes (int, int) (start, length)
	 from ByteArray r
{
  (start, length) = [r adjustRange (start, length)];
  [stream print (" (bytes ", length, ' ', '"')];

  // This is slow.  Better search for a `"' and print the string fast if
  // it does not contain `"'.
  // Fri May  8 10:17:34 1998, tiggr@gerbil.org
  int i;
  for (i = start; i < length; i++)
    {
      byte b = r[i];
      if (b == '"')
	[stream print ('\\', '"')];
      else if (b == '\\')
	[stream print ('\\', '\\')];
      else
	[stream print b];
    }

  [stream print ('"', ')')];
}

end;

/******************** TextStreamDecoder ********************/

<doc> This class is unimplemented.  </doc>
implementation class
TextStreamDecoder: StreamDecoder, XLTokens

end;

implementation instance
TextStreamDecoder
{
  <doc> The lexer actually doing the reading from our {stream}.  </doc>
  XL lexer;

  <doc> The current token, cached so we know when we are starting up, in
      which case the token is {XLT_EPSILON}.  </doc>
  int token;
}

<doc> Designated initializer of our super.  </doc>
id
  initWithStream InputStream s
{
  = [self initWithLexer [[XL alloc] initWithStream s]];
}

<doc> Designated initializer.  </doc>
id
  initWithLexer XL l
{
  (lexer, token) = (l, XLT_EPSILON);
  = [super initWithStream [lexer stream]];
}

Any
  decode
{
  if (token == XLT_EPSILON)
    {
      [self termToken XLT_PAR_OPEN];
      [self termSymbol "graph"];
    }

  for (;;)
    {
      [self termToken XLT_PAR_OPEN];
      if (token == XLT_PAR_CLOSE)
	return nil;
      if (token == XLT_INT)
	break;
      String us = [lexer matched];
      if ([us equal "class"])
	[self declareClass];
      else if ([us equal "ref"])
	return [self readReference];
      else
	[self skipList];
    }

  int class_num = int ([lexer int_value]);
  [self termToken XLT_INT];
  int obj_num = int ([lexer int_value]);
  [self termToken XLT_INT];

  = [self decodeObject [self reference class_num] as obj_num];

  [self termToken XLT_PAR_CLOSE];
}

byte
  decode
{
  [self termToken XLT_PAR_OPEN];
  [self termSymbol "byte"];
  = byte ([lexer int_value]);
  [self termToken XLT_INT];
  [self termToken XLT_PAR_CLOSE];
}

boolean
  decode
{
  [self termToken XLT_PAR_OPEN];
  [self termSymbol "boolean"];
  = !![lexer int_value];
  [self termToken XLT_INT];
  [self termToken XLT_PAR_CLOSE];
}

char
  decode
{
  [self termToken XLT_PAR_OPEN];
  [self termSymbol "char"];
  = char ([lexer int_value]);
  [self termToken XLT_INT];
  [self termToken XLT_PAR_CLOSE];
}

int
  decode
{
  [self termToken XLT_PAR_OPEN];
  [self termSymbol "int"];
  = int ([lexer int_value]);
  [self termToken XLT_INT];
  [self termToken XLT_PAR_CLOSE];
}

long
  decode
{
  [self termToken XLT_PAR_OPEN];
  [self termSymbol "long"];
  = [lexer int_value];
  [self termToken [lexer token] == XLT_INT ? XLT_INT : XLT_LONG];
  [self termToken XLT_PAR_CLOSE];
}

float
  decode
{
  [self termToken XLT_PAR_OPEN];
  [self termSymbol "float"];
  = float ([lexer float_value]);
  [self termToken XLT_FLOAT];
  [self termToken XLT_PAR_CLOSE];
}

double
  decode
{
  [self termToken XLT_PAR_OPEN];
  [self termSymbol "double"];
  = [lexer float_value];
  [self termToken XLT_FLOAT];
  [self termToken XLT_PAR_CLOSE];
}

(pointer, int) (contents, length)
  decodeBytes
{
  [self termToken XLT_PAR_OPEN];
  [self termSymbol "bytes"];
  length = int ([lexer int_value]);
  = [lexer readBytes length];
  [self termToken XLT_INT];
  [self termToken XLT_PAR_CLOSE];
}

protected void
  declareClass
{
  [self termSymbol "class"];
  int obj_id = int ([lexer int_value]);
  [self termToken XLT_INT];
  String name = [UniqueString with [lexer matched]];
  [self termToken XLT_SYMBOL];

  class (State) cls = [Runtime classNamed name];
  if (!cls)
    cls = [[SelectorCondition for name class unknown-class-condition
			      message nil selector cmd] signal];
  perm_objects_done[obj_id] = cls;

  [self termToken XLT_PAR_OPEN];
  [self termSymbol "version"];
  class_versions[cls] = [IntNumber with [lexer intValue]];
  [self termToken XLT_INT];
  [self termToken XLT_PAR_CLOSE];

  [self termToken XLT_PAR_CLOSE];
}

protected int
  nextToken
{
  = token = [lexer nextToken];
}

protected Any
  readReference
{
  [self termSymbol "ref"];
  int obj_id = int ([lexer int_value]);
  [self termToken XLT_INT];
  [self termToken XLT_PAR_CLOSE];
  = [self reference obj_id];
}

<doc> Read tokens up to and including the first top-level close
    parenthesis.  </doc>
protected void
  skipList
{
  int level;

  for (;;)
    {
      if (token == XLT_PAR_OPEN)
	level++;
      else if (token == XLT_PAR_CLOSE)
	{
	  if (!level--)
	    {
	      [self nextToken];
	      return;
	    }
	}
      else if (token == XLT_EOF)
	break;
      [self nextToken];
    }
}

protected void
  termSymbol String name
{
  if (token == XLT_SYMBOL)
    {
      String m = [lexer matched];
      if (![m equal name])
	[[SelectorCondition for self class coding-condition
	    message [[MutableByteString new]
		      print ("expected `", name, "' but encountered `", m, "'")]
			    selector cmd]
	  raise];
    }

  [self termToken XLT_SYMBOL];
}

protected void
  termToken int t
{
  if (token == XLT_EPSILON)
    [self nextToken];

  if (token != t)
    [[SelectorCondition for self class coding-condition
      message [[MutableByteString new]
	       print ("expected ", t, " but encountered ", token, " `",
		      [lexer matched], "'")]
      selector cmd]
     raise];

  [self nextToken];
}

end;
