<copyright> HashTable class.
    Written by <a href="mailto:tiggr@gerbil.org">Pieter J. Schoenmakers</a>

    Copyright &copy; 1996-1999 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    The HashTable and its subclasses modify the hash value returned by an
    object used as the key by using the multiplication method for hash
    functions as described in the excellent ``Introduction to Algorithms''
    by Thomas H. Cormen, Charles E. Leiserson, and Ronald L. Rivest, who
    attribute the use of the golden ratio (minus one) to Knuth (who else).

    The advantage of having the hashtable modify the hash value is that
    the hash functions implemented by the objects can be more or less
    trivial, especially in the case of pointers (or object addresses) and
    integers.

    <id>$Id: HashTable.t,v 1.29 1999/09/08 19:36:12 tiggr Exp $</id>
    </copyright>

/******************** HashTable ********************/

implementation class
HashTable: State, Keyed
{
  <doc> Thirty-two bits by which to multiply a hashvalue to make all bits,
      more or less, significant.

      This (unsigned) value is (1 << 32) * frac (0.5 + 0.5 * sqrt (5)).
      The value of which the fractional part is taken is the golden ratio.
      (0x9e3779b9).  </doc>
  const GOLDEN_BITS = -1640531527;
}

end;

implementation instance
HashTable
{
  <doc> The number of stored objects.  </doc>
  public int length;

  <doc> The 2log of the number of buckets.  </doc>
  int size_shift;

  <doc> The buckets.  </doc>
  MutableObjectArray buckets;
}

<doc> Designated initializer.  </doc>
id (self)
  init
{
  [self empty];
}

<doc> Remove all elements from the table.  </doc>
void
  empty
{
  (length, size_shift) = (0, 4);
  buckets = [MutableObjectArray withCapacity 1 << size_shift];
  [buckets resize 1 << size_shift];
}

<doc> Evaluate the {block} for each object element in this {HashTable}, by
    passing this method to the {BucketElement}s.  </doc>
void
  do Block block
{
  int i, n = 1 << size_shift;

  for (i = 0; i < n; i++)
    {
      BucketElement elt = buckets[i];
      if (elt != nil)
	[elt do block];
    }
}

<doc> Return the {key} if present, or {nil} otherwise.  </doc>
Any
  at All key
{
  int bucket = [key hash] * GOLDEN_BITS >>> (32 - size_shift);
  BucketElement elt = buckets[bucket];

  = !elt ? nil : [elt member key];
}

<doc> A different name for {at}.  </doc>
Any
  member All object
{
  return self[object];
}

<doc> Like {member}, but the elements are compared using the selector
    {cmp}.  </doc>
Any
  member All key
   equal selector cmp
{
  int bucket = [key hash] * GOLDEN_BITS >>> (32 - size_shift);
  BucketElement elt = buckets[bucket];

  = !elt ? nil : [elt member key equal cmp];
}

<doc> Like {member}, but the element is identified on reference equality.
    </doc>
Any
  memq All key
{
  int bucket = [key hash] * GOLDEN_BITS >>> (32 - size_shift);
  BucketElement elt = buckets[bucket];

  = !elt ? nil : [elt memq key];
}

void
  encodeUsingCoder Encoder coder
{
  if (![coder hasBeenCodedFor [HashTable self]])
    {
      [super (Keyed) encodeUsingCoder coder];
      [super (State) encodeUsingCoder coder];

      [coder encode length];
      [coder encode size_shift];
      [coder encode buckets];
    }
}

void
  initWithCoder Decoder coder
{
  if (![coder hasBeenCodedFor [HashTable self]])
    {
      [super (Keyed) initWithCoder coder];
      [super (State) initWithCoder coder];

      length = [coder decode];
      size_shift = [coder decode];
      buckets = [coder decode];
    }
}

<doc> Adjust the length of the hashtable, resizing if necessary.  </doc>
protected void
  adjust_length int inc
pre
  inc != 0
{
  length += inc;

  if (inc > 0 && length == 1 << size_shift + 1 && size_shift < 32)
    [self resize size_shift + 1];
  else if (inc < 0 && length == 1 << size_shift - 1 && size_shift > 4)
    [self resize size_shift - 1];
}

protected void
  resize int new_shift
{
  Indexed old_buckets = buckets;
  int i, n = 1 << size_shift;

  size_shift = new_shift;
  buckets = [MutableObjectArray withCapacity 1 << size_shift];
  [buckets resize 1 << size_shift];

  for (i = 0; i < n; i++)
    {
      BucketElement elt = old_buckets[i];
      if (elt != nil)
	[elt resizing_feed self];
    }
}

void
  resizing_add BucketElement elt
{
  int bucket = [elt rehash] * GOLDEN_BITS >>> (32 - size_shift);

  [elt resizing_add buckets[bucket]];
  buckets[bucket] = elt;
}

end;

/******************** HashTableContainer ********************/

<doc> The HashTableContainer class is just a HashTable which knows how to
    mark its elements as a container.  It is intended to be inherited by
    various class employing the HashTable class as a superclass for
    implementation reuse.  </doc>
implementation class
HashTableContainer: HashTable, Container

end;

implementation instance
HashTableContainer

void
  gc_container_mark_elements
{
  int i, n, len = 1 << size_shift;
  BucketElement elt, pe;

  /* Incidentally, we must also mark our BUCKETS, since containers are not
     automatically marked.  */
  [buckets gc_mark];

  for (i = 0; i < len; i++)
    {
      elt = buckets[i];
      if (elt != nil)
	{
	  (pe, n) = [elt gc_mark_values];
	  if (elt != pe)
	    buckets[i] = pe;
	  length -= n;
	}
    }
}

end;

/******************** MutableHashTable ********************/

implementation class
MutableHashTable: HashTableContainer, MutableKeyed

end;

implementation instance
MutableHashTable

<doc> Add {elt} to the receiving hashtable.  </doc>
void
  add BucketElement elt
{
  int inc = 1, bucket = [elt hash] * GOLDEN_BITS >>> (32 - size_shift);
  BucketElement bucket_elt = buckets[bucket];

  if (!bucket_elt)
    buckets[bucket] = elt;
  else
    inc = [bucket_elt addElement elt];

  if (inc != 0)
    [self adjust_length inc];
}

<doc> Remove {elt} from the receiving hashtable, if present.  </doc>
void
  remove BucketElement elt
{
  int bucket = [elt hash] * GOLDEN_BITS >>> (32 - size_shift);
  BucketElement bucket_elt = buckets[bucket];

  if (bucket_elt != nil)
    {
      BucketElement new_bucket_elt;
      int decrease;

      (new_bucket_elt, decrease) = [bucket_elt remove elt];
      if (decrease != 0)
	{
	  // Should we shrink?
	  // Mon Dec  1 13:57:34 1997, tiggr@natlab.research.philips.com
	  length -= decrease;
	  if (new_bucket_elt != bucket_elt)
	    buckets[bucket] = new_bucket_elt;
	}
    }
}

end;

/******************** HashTableEnumerator ********************/

implementation class
HashTableEnumerator: State, Enumerator

instance (id)
  with Indexed b
{
  = [[self alloc] init b];
}

end;

implementation instance
HashTableEnumerator
{
  <doc> The array the dictionary uses to store the buckets.  </doc>
  Indexed buckets;

  <doc> The next bucket index we shall use.  </doc>
  int next;

  <doc> The bucket element we're looking at.  </doc>
  BucketElement elt;
}

protected id
  init Indexed b
{
  buckets = b;

  = self;
}

<doc> Update {elt} to point to the next bucket element.  </doc>
protected boolean
  next
{
  if (elt != nil)
    elt = [elt next];

  if (!elt)
    {
      int n = [buckets length];

      while (!elt && next < n)
	elt = buckets[next++];
    }

  = elt != nil;
}

end;
