/* Interface to TLObject class
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLObject.h,v 1.2 1998/01/09 11:20:51 tiggr Exp $  */

/* Not strictly needed here, but present to avoid stdarg problems on hppa
   GCC 2.6.3.  */
#import <stdarg.h>
#import <objc/Object.h>
#import "config.h"
#import "policy.h"

#if DEBUG_GC
extern int debug_gc;
#endif

#if GNU_RUNTIME
#if INLINE_MSG_LOOKUP

#import <objc/sarray.h>

/* This need not be true, but that doesn't matter.  */
extern id __objc_word_forward (id, SEL, ...);
extern id __objc_block_forward (id, SEL, ...);
extern id nil_method (id, SEL, ...);

extern __inline__
#endif
#if CONST_MSG_LOOKUP || INLINE_MSG_LOOKUP
IMP objc_msg_lookup (id receiver, SEL op)
#endif
#if CONST_MSG_LOOKUP
  __attribute__ ((const))
#endif
#if INLINE_MSG_LOOKUP
{
  IMP result;
  if(receiver)
    {
      result = sarray_get(receiver->class_pointer->dtable, (sidx)op->sel_id);
      if (result == 0)
	{
	  const char *t = op->sel_types;
	  if (t && (*t == '[' || *t == '(' || *t == '{'))
	    result = (IMP)__objc_block_forward;
	  else
	    result = (IMP)__objc_word_forward;
	}
      return result;
    }
  else
    return nil_method;
}
#elif CONS_MSG_LOOKUP
;
#endif
#endif

/* Time intervals are doubles, after NSTimeInterval.  Not only does it
   give millisecond accuracy while counting millenia; it also gives
   femto-second accuracy in sub-second ranges and nano-second accuracy in
   ranges in order of a day.

   This should be in TLDate, but is also needed here...  */
typedef double time_interval;

/* Invoke the object's `eval', if it is not NIL.  Otherwise, return NIL.
   XXX Maybe this should actually eval Qnil?  */
#define EVAL(O)  ({id _o = (O); _o ? [_o eval] : nil;})

/* If the receiving object is NIL, pass control to `Qnil'.  */
#define EVAL_WITH_ARGS(O, A)  \
  ({id _o = (O); [(_o ? _o : Qnil) evalWithArguments: (A)];})

/* Invoke the object's `gcMark', if it is not NIL.  Side effects are
   allowed.  */
#define MARK(O)  ({ id _o = (O); if (_o) [_o gcMark]; _o; })

/* Assign to the instance variable A the object B.  If SELF is black, make
   sure the object B is at least gray.  Side effects are allowed.  It is not
   necessary for correctness that SELF is black, but it is faster to check
   this.  */
#define ASGN_IVAR(A, B)  \
  do {							\
      id _b = (B);					\
      if (_b && TL_COLOUR (self) == TLOC_BLACK)		\
	MARK (_b);					\
      (A) = _b;						\
  } while (0)

/* Protect the object B for being pointed to from a stack protection struct.
   Make sure B is at least gray.  */
#define ASGN_SPROT(B)  MARK (B)

/* Protect the object B being assigned to a static variable.  */
#define ASGN_STATIC(A, B)  \
  do { id _b = (B); MARK (_b); (A) = _b; } while (0)

/* Adjust the reference count of the receiving object, which plans to
   return something different from SELF, from within its `-init' method.
   The test on the reference count succeeds if this object was created
   with `+alloc'; it fails when created with `+gcAlloc'.  */
#define INIT_WILL_RETURN_NIL()  \
  do									\
    {									\
    } while (0)

/* If an object can't initialize and thus wants to return NIL from its init
   method, it should use this macro.  */
#define INIT_RETURN_NIL()  \
  do									\
    {									\
      INIT_WILL_RETURN_NIL ();						\
      return (nil);							\
    } while (0)

/* Use these macros to manipulate quitting behaviour.  They are already used
   by the `x' allocation routines, the garbage collection and the posing
   root classes, so, basically, all allocation stuff is already protected.  */
#define QUIT_DISABLE  \
  do { tl_quit_inhibit++; } while (0)

#define QUIT_ENABLE  \
  do { if (!--tl_quit_inhibit && tl_quit_request)		\
	 { tl_quit_request = 0; [TLObject error: "quit"]; }	\
     } while (0)

/* TL_QUIT_INHIBIT inhibits a quit.  TL_QUIT_REQUEST requests a quit.  See
   the QUIT_DISABLE and QUIT_ENABLE macros.  XXX These declarations should
   not be here but inside the QUIT macros.  However, NeXT/m68k GCC 2.6.3
   moans about redeclarations...  */
extern int tl_quit_request, tl_quit_inhibit;

/* The number of objects allocated since the last partial GC run.  */
extern unsigned long tlgc_alloc_since_partial;

/* The number of objects allocated since the last complete GC run.  This
   excludes TLGC_ALLOC_SINCE_PARTIAL objects from all allocated objects.  */
extern unsigned long tlgc_alloc_since_complete;

#define TLGC_TOTAL_ALLOC_SINCE_COMPLETE  \
  (tlgc_alloc_since_partial + tlgc_alloc_since_complete)

/* The number of allocated objects.  */
extern unsigned long tlgc_num_alloc;

/* The total number of objects, both allocated and free.  */
extern unsigned long tlgc_num_total;

/* The threshold for TLGC_ALLOC_SINCE_PARTIAL before `-[TLCons eval]' will
   initiate a GC run.  */
extern unsigned long tlgc_partial_threshold;

/* The time limit on a GC run initiated by running over
   TLGC_PARTIAL_THRESHOLD.  */
extern unsigned long tlgc_partial_time_limit;

/* The threshold for TLGC_ALLOC_SINCE_COMPLETE plus TLGC_ALLOC_SINCE_PARTIAL
   before `-[TLCons eval]' will initiate a GC run.  */
extern unsigned long tlgc_total_threshold;

/* The time limit on a GC run initiated by running over
   TLGC_TOTAL_THRESHOLD.  */
extern unsigned long tlgc_total_time_limit;

/* The amount of time (seconds) spent in GC since the last completion.  */
extern time_interval tlgc_this_time;

/* The same information, split in the three phases.  */
extern time_interval tlgc_this_protect, tlgc_this_mark, tlgc_this_sweep;

/* The amount of time (seconds) spent in total in all completed GC runs.  */
extern time_interval tlgc_total_time;

/* The same information, split in the three phases.  */
extern time_interval tlgc_total_protect, tlgc_total_mark, tlgc_total_sweep;

/* The number of completed and partial GC runs.  */
extern int tlgc_num_run_complete, tlgc_num_run_partial;

/* The hard limit on the number of objects.  If this limit is reached while
   one of the partial or total thresholds is reached, a run til completion
   is performed.  */
extern unsigned long tlgc_alloc_limit;

/* The number of times `gcMark' has been invoked.  */
extern unsigned long tlgc_num_mark;

/* Pointer to the function to create a new instance of the CLASS.  */
extern id (*tl_objc_object_alloc) (struct objc_class *class);

/* Pointer to the function to do a garbage collection run, limited to USEC
   micro seconds elapsed time.  */
extern id (*tl_garbage_collect) (unsigned long usec);

/* Pointer to the function invoked to perform custom garbage protection.  */
extern id (*tl_garbage_protect) (void);

enum tlobject_colour
{
  /* Gray is zero, since newly allocated objects are initialized to be all
     zero bits.  Newly allocated objects are gray to protect them from being
     white, in case they were allocated between partial sweeps.  */
  TLOC_GRAY = 0,

  TLOC_WHITE,
  TLOC_BLACK,

  /* This is a fake colour, used for counting dead objects.  */
  TLOC_PINK,

  /* The number of colours.  */
  TLOC_NUM
};

/* Accessor macros for TLObject fields.  */
#define TL_COLOUR(O)		((enum tlobject_colour) (O)->_tlobject_colour)
#define TL_COLOUR_SET(O, C)	((O)->_tlobject_colour = (C))
#define TL_LOCKED_P(O)		!!((O)->_tlobject_locked)
#define TL_LOCKED_SET(O)	((O)->_tlobject_locked = 1)
#define TL_LOCKED_CLEAR(O)	((O)->_tlobject_locked = 0)
#define TL_REFCOUNT(O)		((O)->_tlobject_refcount)
#define TL_RETAIN(O)		((O)->_tlobject_refcount += 1)
#define TL_RELEASE(O)		((O)->_tlobject_refcount -= 1)

/* If an object's reference count reaches TL_REFCOUNT_MAX, any subsequent
   release or retain will have no effect.  */
#define TL_REFCOUNT_MAX		65535

/* Forward declaration.  */
@class TLCons;

@interface TLObject: Object
{
  /* The colour of this object.  Newly allocated objects start out as gray.  */
  char _tlobject_colour;

  /* Whether this object should be on the locked list.  Only used when
     PARANOID_GCLOCK is on.  */
  char _tlobject_locked;

  /* The reference count of this object.  */
  unsigned short _tlobject_refcount;
}

/******************** allocation and garbage collection ********************/

/* Perform a mark & sweep garbage collection run.  */
+gc;
+gc: (unsigned long) usec;

/* Return a newly allocated instance which is not retained.  */
+gcAlloc;

/* Sent to objects which are still white at the end of a garbage collection
   run: They are deleted.  */
-(void) dealloc;

/* Protect an object against garbage collection: Make it one of the root
   objects.  */
-(void) gcLock;

/* Tell the receiving object that it should at least be gray.  If it is
   white, its colour will be set to gray.  */
-(void) gcMark;

/* Tell the receiving object that it will move from gray to black.  It
   should send all objects it knows a `gcMark'.  */
-(void) gcReference;

/* Remove the protection set by `gcLock'.  */
-(void) gcUnlock;

/******************** miscellaneous ********************/

/* Read back in canonical output.  */
+evalWithArguments: (TLCons *) args;

/* Return -1 if the receiving object is `less than' the object O, 0 if they
   are equal, 1 if the receiving object is `greater than' O.  */
-(int) compare: o;

@end
