/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: lti.m,v 1.28 1998/01/20 00:01:18 tiggr Exp $  */

#import "lti.h"

/* Keys in the info file.  */
TLSymbol *Qfile_version, *Qotmc_version, *Qcompile_date, *Qselectors;
TLSymbol *Qextension, *Qclass, *Qsupers, *Qmethods, *Qinstance, *Qfile;
TLSymbol *Qdynamic_selectors, *Qstrings, *Qvariables, *Qstatic, *Qsuper_refs;
TLSymbol *Qconstants, *Qlocal;

id CO_LTIClass, CO_LTIExtension, CO_LTIInstance, CO_LTIMeta;
id CO_LTIMethod, CO_LTISelector, CO_LTIStringCST, CO_LTIVariable;

/* Dictionary from actual selector to its dynamic definition.  */
TLDictionary *dynamic_selectors;

/* The `void load' selector.  */
LTISelector *load_selector;

/* All implementations of `load'.  */
TLCons *load_imps;

static struct
{
  id name;
  TLSymbol **sym;
} symbols[] =
{
  {@"class", &Qclass},
  {@"compile-date", &Qcompile_date},
  {@"constants", &Qconstants},
  {@"extension", &Qextension},
  {@"file-version", &Qfile_version},
  {@"file", &Qfile},
  {@"instance", &Qinstance},
  {@"local", &Qlocal},
  {@"methods", &Qmethods},
  {@"otmc-version", &Qotmc_version},
  {@"dynamic-selectors", &Qdynamic_selectors},
  {@"selectors", &Qselectors},
  {@"supers", &Qsupers},
  {@"static", &Qstatic},
  {@"strings", &Qstrings},
  {@"super-refs", &Qsuper_refs},
  {@"variables", &Qvariables},
  {0, 0}
};

void
lti_init (int argc, char **argv)
{
  static BOOL init = NO;
  int i;

  if (init)
    return;
  init = YES;

  CO_LTIClass = objc_get_class ("LTIClass");
  CO_LTIExtension = objc_get_class ("LTIExtension");
  CO_LTIMeta = objc_get_class ("LTIMeta");
  CO_LTIMethod = objc_get_class ("LTIMethod");
  CO_LTISelector = objc_get_class ("LTISelector");
  CO_LTIStringCST = objc_get_class ("LTIStringCST");

  /* XXX objc_get_class won't find these??!?!?!
     Mon Jan 27 17:57:10 1997, tiggr@akebono.ics.ele.tue.nl  */
  CO_LTIInstance = [LTIInstance self];
  CO_LTIVariable = [LTIVariable self];

  ltt_init (argc, argv);

  dynamic_selectors = [TLDictionary dictionary];
  [dynamic_selectors gcLock];

  for (i = 0; symbols[i].sym; i++)
    *symbols[i].sym = [TLSymbol symbolWithName: symbols[i].name];
}

void
lti_finish_init (void)
{
  load_selector = [[LTTSelector selectorWithName: @"v_load_r"
				inArgs: @"r" outArgs: @""] semantics];
  ltt_finish_init ();
}

LTIMeta *
lti_get_meta (TLString *unit_name, TLString *meta_name, BOOL classp)
{
  LTTMeta *m = ltt_get_meta (unit_name, meta_name, classp);

  return m ? [m semantics] : nil;
}

LTIMeta *
lti_retrieve_meta (TLCons *l, int qualified, int *was_qualified)
{
  TLString *unit_name, *meta_name;
  TLCons *in_l = l;
  BOOL classp = NO;
  LTIMeta *m;
  int q = 0;
  id first;

  DECONS (l, first, l);

  if (first == Qinstance || first == Qclass)
    {
      if (!qualified)
	warning (@"unexpectedly qualified meta indication: %#", in_l);
      else
	{
	  q = 1;
	  if (first == Qclass)
	    classp = YES;
	}

      DECONS (l, unit_name, l);
    }
  else
    {
      if (qualified > 0)
	warning (@"unexpectedly unqualified meta indication: %#", in_l);
      unit_name = first;
    }

  DECONS (l, meta_name, l);

  if (l)
    warning (@"excess ignored in meta indication: %#", l);
  if (!unit_name || ![unit_name stringp]
      || !meta_name || ![meta_name stringp])
    {
      error (@"malformed meta indication: %#", in_l);
      return nil;
    }

  m = lti_get_meta (unit_name, meta_name, classp);
  if (m && was_qualified)
    *was_qualified = q;
  return m;
}

/* Hard worker for lti_load_unit_info to grok extension definition.  */
static void
lti_do_extension (TLCons *ei)
{
  LTIExtension *ext = nil;
  TLString *ext_name;
  TLCons *super_def;
  LTTExtension *x;
  LTIMeta *meta;

  DECONS (ei, ext_name, ei);
  DECONS (ei, super_def, ei);

  meta = lti_retrieve_meta (super_def, 1, NULL);
  if (!meta)
    return;

  x = [[meta structure] extensionNamed: ext_name];
  if (!x)
    {
      error (@"%@ does not have an extension named `%@'",
	     ltt_meta_name ([meta structure]), ext_name);
      return;
    }

  push_report_context (formac (nil, @"in %@", ltt_ext_name (x)));
  ext = [x semantics];

  [ext digestInfo: ei];

  pop_report_context ();
}

void
lti_load_file_info (LTTFile *file)
{
  TLFILEStream *fs;
  TLCons *info;
  TLString *s;
  TLLLex *lx;
  FILE *f;

  if ([file loadedInfo])
    return;
  [file setLoadedInfo];

  s = (id) formac (nil, @"%@." TOM_INFORMATION_SUFFIX,
		   ltt_filename_without_extension ([file interfaceFilename]));
  f = fopen ([s cString], "r");
  if (!f)
    {
      error (@"%@: %s", s, ERRMSG);
      return;
    }

  PARSE_START_CONTEXT (s);

  push_report_context (formac (nil, @"at top level"));

  fs = [TLFILEStream streamWithFILE: f];
  lx = [TLLLex lexerWithStream: fs];
  info = [lx read];

  current_line = [[lx line] intValue];

  if (!info || ![info consp])
    error (@"bad info file");
  else if (info == (id) Qlex_eof)
    error (@"bad info file (EOF)");
  else
    {
      int file_version;
      BOOL ok = YES;

      while (info)
	{
	  id entry;

	  DECONS (info, entry, info);

	  if ([entry consp])
	    {
	      TLSymbol *key;
	      TLCons *rest;

	      DECONS (entry, key, rest);

	      if (key == Qfile_version)
		file_version = [[rest car] intValue];
	      else if (key == Qotmc_version)
		{
		  if (flag_verbose > 1)
		    formac (V_stderr_, @"%@: compiled by otmc version %@\n",
			    s, [rest car] );
		}
	      else if (key == Qcompile_date)
		{
		  if (flag_verbose > 1)
		    formac (V_stderr_, @"%@: compiled on %@\n", s, [rest car]);
		}
	      else if (key == Qselectors)
		{
		  while (rest)
		    {
		      id <TLString> sel_name, in_name, out_name;

		      DECONS (rest, entry, rest);
		      DECONS (entry, sel_name, entry);
		      DECONS (entry, in_name, entry);
		      DECONS (entry, out_name, entry);

		      [LTTSelector selectorWithName: sel_name
				   inArgs: in_name outArgs: out_name];
		    }
		}
	      else if (key == Qdynamic_selectors)
		{
		  while (rest)
		    {
		      TLCons *list;
		      id <TLString> sel_name, alias_name;
		      LTISelector *sel, *alias;

		      DECONS (rest, list, rest);
		      DECONS (list, sel_name, list);

		      sel = [[LTTSelector searchSelectorNamed: sel_name]
			      semantics];
		      if (!sel)
			ABORT ();

		      while (list)
			{
			  DECONS (list, alias_name, list);

			  alias = [[LTTSelector searchSelectorNamed:
						  alias_name] semantics];
			  if (!alias)
			    ABORT ();
			  [dynamic_selectors setObject: sel forKey: alias];
			}
		    }
		}
	      else if (key == Qstrings)
		{
		  while (rest)
		    {
		      id <TLString> string, name;

		      DECONS (rest, entry, rest);
		      DECONS (entry, string, entry);
		      DECONS (entry, name, entry);

		      [LTTStringCST stringWithString: string name: name];
		    }
		}
	      else if (key == Qextension)
		lti_do_extension (rest);
	      else
		ok = NO;
	    }
	  else
	    ok = NO;

	  if (!ok)
	    {
	      error (@"bad entry: %#", entry);
	      ok = YES;
	    }
	}
    }

  [fs close];
  pop_report_context ();

  PARSE_STOP_CONTEXT;
  PARSE_END_CONTEXT;
}

void
lti_load_unit_info (LTTUnit *u)
{
  id <TLEnumerator> e = [u files];
  LTTFile *file;

  push_report_context (formac (nil, @"in unit %#", [[u lttName] internal]));

  while ((file = [e nextObject]))
    lti_load_file_info (file);

  pop_report_context ();
}
