/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: gi.m,v 1.21 1998/04/22 12:35:28 tiggr Exp $  */

#import "gi.h"
#import <sys/stat.h>

/* Common declaration in case tl was compiled without DEBUG_GC.  */
int debug_gc;

/* Iff !0, we should not run GC.  */
int flag_inhibit_gc = 1;

/* The number of times `-v' was specified on the command line.  */
int flag_verbose;

/* Whether comments and documentation are to be saved.  */
int flag_comments, flag_documentation;

/* Iff !0, the output files are precious and are not deleted, even when
   errors are detected.  */
int flag_precious;

/* Iff !0, we're not to examine modification times, i.e. the interface files
   are always updated.  */
int flag_force;

/* Should we continue after en error for a file?  */
int flag_k;

/* Iff !NIL, this is the unit file we're writing.  */
id <TLOutputStream> uf;

/* The units the unit being created will need.  */
TLVector *units_used;

const char *
indent (int level)
{
  static const char s[] = "                                        ";
  int i = sizeof (s) - 1 - level;

  return i < 0 ? s : s + i;
} /* indent */

void
usage (int rc, char *s)
{
  if (s)
    formac (V_stderr_, @"%@: bad option/argument: %s\n", prog, s);

  formac (rc || s ? V_stderr_ : V_stdout_, @"gi (%@), version %s\n\
usage: %@ options [files]\n\
If the FILES are specified, the unit-file is written.\n\
options:\n\
 -dgc <n>	if n < 0, inhibit garbage collection\n\
		if n >= 0, set gc debugging to N\n\
 -F		force interface generation irrespective of modification times\n\
 -h -?		print help on stdout and exit\n\
 -k		continue with other files after error\n\
 -u unit	name the unit to be used/created (mandatory)\n\
 -v		increase verbosity\n\
 --version	print version on stdout and exit\n\
 -U <unit>	unit needed by this unit (when generating the unit file)\n\
 -f		modify behaviour.  Flags currently implemented are:\n\
		-fcomments -fdocumentation\n\
 -I <path>	add elements from the colon separated PATH to the load path\n\
 -precious	don't unlink output files in case of errors\n\
", prog, long_version, prog);

  exit (rc);
}

int
main (int argc, char **argv)
{
  TLString *time_stamp, *uf_name = nil;
  id unit_name = nil;
  TLVector *files = nil;
  LTTUnit *unit = nil;
  char *opt;
  int i;

  ltt_init (argc, argv);

  for (i = 1; i < argc; i++)
    if (argv[i][0] == '-' && !files)
      {
	BOOL ok = YES;

	switch (argv[i][1])
	  {
	  case '-':
	    if (!strcmp (argv[i], "--version"))
	      {
		if (flag_verbose)
		  printf ("gi, version %s\n", long_version);
		else
		  printf ("%s\n", short_version);
		exit (0);
	      }
	    break;

	  case 'd':
	    if (!strncmp (argv[i], "-dgc", 4))
	      {
		opt = get_arg (&i, 4, argv, argc);
		if (opt)
		  {
		    int i = atoi (opt);
		    if (i < 0)
		      flag_inhibit_gc = 1;
		    else
		      {
			flag_inhibit_gc = 0;
			debug_gc = i;
		      }
		  }
	      }
	    else
	      ok = NO;
	    break;

	  case 'F':
	    if (!strcmp (argv[i], "-F"))
	      flag_force = 1;
	    else
	      ok = NO;
	    break;

	  case 'f':
	    if (!strcmp (argv[i], "-fcomments"))
	      flag_comments = 1;
	    else if (!strcmp (argv[i], "-fdocumentation"))
	      flag_documentation = 1;
	    else if (!strcmp (argv[i], "-fno-comments"))
	      flag_comments = 0;
	    else if (!strcmp (argv[i], "-fno-documentation"))
	      flag_documentation = 0;
	    else
	      ok = NO;
	    break;

	  case 'h':
	  case '?':
	    /* Allow any option starting with a `h' or `?' to be an
	       indication of the need for help.  */
	    usage (0, NULL);
	    break;

	  case 'k':
	    flag_k = 1;
	    break;

	  case 'p':
	    if (!strcmp (argv[i], "-precious"))
	      flag_precious = 1;
	    else
	      ok = NO;
	    break;

	  case 'u':
	    opt = get_arg (&i, 2, argv, argc);
	    if (opt)
	      unit_name = [TLString stringWithCString: opt];
	    break;

	  case 'v':
	    if (!strcmp (argv[i], "-v"))
	      flag_verbose++;
	    else
	      ok = NO;
	    break;

	  case 'I':
	    opt = get_arg (&i, 2, argv, argc);
	    if (opt)
	      add_to_load_path ([[TLString stringWithCString: opt]
				 componentsSeparatedByString: @":"]);
	    break;

	  case 'U':
	    opt = get_arg (&i, 2, argv, argc);
	    if (opt)
	      {
		if (!units_used)
		  units_used = [TLVector vector];
		[units_used addElement: [TLString stringWithCString: opt]];
	      }
	    break;

	  default:
	    ok = NO;
	    break;
	  }

	if (!ok)
	  error (@"bad option: %s", argv[i]);
      }
    else
      {
	if (!files)
	  files = [TLVector vector];
	[files addElement: [TLString stringWithCString: argv[i]]];
      }

  if (!NUM_ERRORS && !unit_name)
    error (@"missing unit name");
  if (NUM_ERRORS)
    return 1;

  add_to_load_path ([TLVector vectorWithElements: DEFAULT_LOAD_PATH]);

  if (!files)
    {
      /* Finish initialization.  */
      ltt_finish_init ();
      unit = [LTTUnit findUnitNamed: unit_name];
      if (!unit)
	return 1;
    }
  else
    {
      uf_name = formac (nil, @"%@.%@", unit_name, TOM_UNIT_SUFFIX);
      uf = open_output_file (uf_name);
    }

  if (!NUM_ERRORS)
    {
      id <TLEnumerator> e = files ? [files enumerator] : [unit files];
      LTTFile *f;

      time_stamp = formac (nil, @"/* Generated by gi (%@)\n"
			   @"   version %s.  */\n", prog, long_version);

      [(id) e gcLock];
      [time_stamp gcLock];

      if (uf)
	{
	  formac (uf, @"%@\nunit %@\n", time_stamp, unit_name);
	  if (units_used)
	    {
	      int i, n = [units_used length];

	      for (i = 0; i < n; i++)
		{
		  TLString *s = [units_used _elementAtIndex: i];
		  formac (uf, i ? @", %@" : @"\nuses %@", s);
		}
	      if (n)
		formac (uf, @";\n");
	    }
	}

      while ((f = [e nextObject]) && (flag_k || !NUM_ERRORS))
	{
	  int previous_num_errors = NUM_ERRORS;
	  TLString *tfile, *jfile;
	  struct stat st, sj;
	  FILE *inf;

	  if (uf)
	    {
	      tfile = (id) f;
	      jfile = formac (nil, @"%@." TOM_INTERFACE_SUFFIX,
			     ltt_filename_without_extension ([tfile basename]));
	    }
	  else
	    {
	      jfile = [f interfaceFilename];
	      tfile = [f implementationFilename];
	    }

	  if (stat ([tfile cString], &st))
	    {
	      error (@"%@: %s", tfile, ERRMSG);
	      continue;
	    }

	  if (stat ([jfile cString], &sj))
	    sj.st_mtime = st.st_mtime - 1;
	  else if (sj.st_mtime >= st.st_mtime && !uf && !flag_force)
	    continue;

	  if ((sj.st_mtime < st.st_mtime || flag_force)
	      && unlink ([jfile cString]) && ERRNO != ENOENT)
	    {
	      error (@"unlink %@: %s", jfile, ERRMSG);
	      continue;
	    }

	  inf = fopen ([tfile cString], "r");
	  if (!inf)
	    {
	      error (@"open %@: %s", tfile, ERRMSG);
	      continue;
	    }

	  if (sj.st_mtime < st.st_mtime || flag_force)
	    {
	      of = open_output_file (jfile);
	      if (!of)
		{
		  fclose (inf);
		  continue;
		}

	      formac (of, @"%@", time_stamp);
	    }
	  else
	    of = nil;

	  if (uf)
	    formac (uf, @"\nfile %#\n{", tfile);

	  parse_file (tfile, unit, inf);

	  if (uf)
	    formac (uf, @"\n}\n");

	  if (of)
	    [(id) of close];

	  fclose (inf);

	  if (of && !flag_precious && NUM_ERRORS != previous_num_errors)
	    if (unlink ([jfile cString]) && ERRNO != ENOENT)
	      error (@"%@: %s", jfile, ERRMSG);
	}
    }

  if (uf)
    {
      formac (uf, @"\nend;\n");

      [(id) uf close];

      if (!flag_precious && NUM_ERRORS)
	if (unlink ([uf_name cString]) && ERRNO != ENOENT)
	  error (@"%@: %s", uf_name, ERRMSG);
    }

  return !!NUM_ERRORS;
}

@implementation LTTMeta (Semantics)

+(id) semanticsForInstance: (LTTInstance *) inst
{
  return nil;
} /* +semanticsForInstance: */

+(id) semanticsForClass: (LTTClass *) cls
{
  return nil;
} /* +semanticsForClass: */

@end

@implementation LTTExtension (Semantics)

+(id) semanticsForExtension: (LTTExtension *) ext
{
  return nil;
} /* +semanticsForExtension: */

@end

/* XXX YYY ZZZ  */
#ifdef SOMETHINGS_VERY_WRONG_WITH_THE_GNU_RUNTIME

@implementation TLObject (FooBar)

-equal: o
{
  return ![self compare: o] ? self : nil;
}

-consp { return (nil); }
-emptyp { return (nil); }
-floatp { return (nil); }
-integerp { return (nil); }
-lambdap { return (nil); }
-listp { return (nil); }
-nilp { return (nil); }
-sequencep { return (nil); }
-stackp { return (nil); }
-stringp { return (nil); }
-streamp { return (nil); }
-subroutinep { return (nil); }
-symbolp { return (nil); }

@end

#endif
