# Opt.pm -- Options parsing
##
##
package Opt;
$VERSION = '0.1';

# RCS Status      : $Id$
# Author          : James Theiler
# Created On      : Early July, 1998
# Last Modified By: James Theiler
# Last Modified On: 13 July 1998
# Update Count    : 1, and counting...
# Status          : Experimental


require Option;
  ## Option's are objects which contain information about
  ## individual options (name, help string, variable they
  ## refer to, and type), as well as methods for manipulating
  ## them, eg reading their value from a string, writing
  ## their values to strings appropriate for menus, optfiles,
  ## etc

  ## Opt, on the other hand, is not an object, but a package
  ## It "has a" list of Option objects, actually of object refs.
  ## As well as the list, it also has two hashes, various global
  ## (well, global to Opt) variables indicating current mode 
  ## Is menu being read, or is info really from command line?
  ## 
    
use Exporter ();
@ISA=qw(Exporter);
@EXPORT=qw(%OPT getoptions);
@EXPORT_OK=qw(%OPT getoptions register optreg optrega);

my $DEBUG=0;

## Globals
## 1. Global Constants (far more than we need in perl, but these match
##                      what is available in the C code)
@optTypes = qw(NUL
	       INT        UNSINT     SHORT      LONG
	       CHAR       INTLEVEL       
	       FLOAT      DOUBLE
	       FLAG       NEGFLAG    ABSFLAG    ABSNEGFLAG
               STRING     UNDELIM
	       VSTRING    CSTRING    UNDELIMV   UNDELIMC);

## 2. Global Variables

my @optList=();  ## optList is an array of Option's, each Option is an object
              ## that is defined in the Option.pm package.  

my %optHash=();  ## optHash is that same set of Option objects, 
              ## but hashed on the name.

%OPT=();      ## OPT is a hash of ${$opt->{value}}'s
              ## This is exported by default, and is meant to be
              ## a global variable in the calling perl script

@optUndelim=();      ## list of undelimited options

$progname="";        ## Name of the program that invoked opt
$optMainRoutine="";  ## Pointer to fcn which is run as "main" 
                     ## whenever "=" specified on menu
$optMenuMode=0;      ## Flag to specify whether opt is currently taking
                     ## options from menu input
$optEnvName="";      ## Name of environment variable from which option
                     ## defaults are defined

my $scalarPrefix="";		# if nonempty, eg if equal to "opt_", then
				# let $opt_xxx be variable associated with
				# option named xxx

my $pkg = (caller)[0];

sub useScalarPrefix {
    $scalarPrefix = shift || "opt_";
}

##################### Functions #############################
##
## Functions for registering options

## optreg registers an option; that is, it takes arguments,
## defines a new Option, and appends it to the list
sub optreg {
    my ($nptr,$type,$name,$descript)=@_;
    $longname="";
    &optrega($nptr,$type,$name,$longname,$descript);
}

sub optrega {
    my ($nptr,$type,$name,$longname,$descript)=@_;
    warn "optrega: not enough arguments" if not defined($descript);
    my $opt;
    if ($type =~ /FLAG/) {
	$opt = Option_FLAG->new($nptr,$type,$name,$longname,$descript);
    } else {
	$opt = Option->new($nptr,$type,$name,$longname,$descript);
    }

    ## each $opt goes into the list
    push @optList,$opt;

    $opt;  ## if it must return something, might as well be the new $opt.
}    

## define all the subroutines of the form
## optreg_INT, optreg_VSTRING, etc.
foreach $type (@optTypes) {
    eval "sub optrega_$type { ".
	"my (\$nptr,\$name,\$longname,\$descript)=\@_; ".
	    "\$type=\"$type\"; ".
		"\&optrega(\$nptr,\$type,\$name,\$longname,\$descript); ".
		    "}";
    eval "sub optreg_$type { ".
	"my (\$nptr,\$name,\$descript)=\@_; ".
	    "\$longname=\"\"; \$type=\"$type\"; ".
		"\&optrega(\$nptr,\$type,\$name,\$longname,\$descript); ".
		    "}";
    ## Having defined them, now export them!
    push @EXPORT_OK,"optreg_$type","optrega_$type";
}

sub register {
    ## Usage:
    ## There are a number of ways that an option can be registered; the
    ## easiest is 
    ##     Opt::register("-x");
    ## This will make 'x' a FLAG option; if '-x' is set on the command line
    ## then the variable $Opt::OPT{x} will be set to 1, otherwise it is zero.
    ## If you want 'x' to be another kind of option, eg INT or FLOAT or STRING,
    ## then you can use, for example,
    ##     Opt::register("-xSTRING");
    ## Long (aka POSIX) options are also permitted, these are registered
    ## with a call like
    ##     Opt::register("--xvalue");
    ## or  Opt::register("--xvalue=INT");
    
    ## A second and third argument are also accepted by Opt::register().
    ## You can assign a help string, eg
    ##     Opt::register("-x","Use method x")
    ## or a direct reference to the variable
    ##     Opt::register("-x",\$x);
    ## or both
    ##     Opt::register("-x","Use method x",\$x);

    my ($string,$help,$hashref) = @_;

    print STDERR "Registering $string\n" if $DEBUG;

    if (defined($help) and (ref($help) eq "HASH" or ref($help) eq "SCALAR")) {
	$hashref=$help;
	$help=undef;
    }
    unless (ref($hashref)) {
	## should check flag for variable prefix; if it exists
	## then don't use %OPT hash
	unless ($scalarPrefix) {
	    $hashref = \%OPT;
	}
    }
    
    unless ($string =~ /^-/) {
	warn "invalid argument to Opt::register [$string],";
    }
    
    if ($string =~ /^-(\w)$/) {
	print STDERR "case 1: -x\n" if $DEBUG;
	$name = $1;
	$longname = "";
	$type = "FLAG";
    } elsif ($string =~ /^-(\w)\s*([a-zA-Z]+)/) {
	print STDERR "case 2: -xINT == $string\n" if $DEBUG;
	$name = $1;
	$longname = "";
	$type = $2;
    } elsif ($string =~ /^--(\w[\w-]+)$/) { # permitting '-' in name, dangerous?
	print STDERR "case 3: --xvalue\n" if $DEBUG;
	$name = "";
	$longname = $1;
	$type = "FLAG";
    } elsif ($string =~ /^--(\w[\w-]+)\s*=\s*([a-zA-Z]+)/) {
	print STDERR "case 4: --xvalue=INT\n" if $DEBUG;
	$name = "";
	$longname = $1;
	$type = $2;
    } else {
	warn "Invalid option specification: [$string]\n";
	return;
    }
    $type = uc $type;

    ## for consistency with Getopt::Long
    $type = "INT" if $type eq "i";


    $help = ($longname || $name) unless $help;

    if ($scalarPrefix) {
	print STDERR "scalar prefix = $pkg"."::"."$scalarPrefix\n" if $DEBUG;
	my $varname = $name || $longname;
	print STDERR "varname=$varname" if $DEBUG;
	$varname =~ s/\W/_/g;	# make into a valid variable name
	print STDERR "=$varname" if $DEBUG;
	$varname = $pkg."::".$scalarPrefix."_".$varname;
	print STDERR "=$varname\n" if $DEBUG;
	print STDERR "\$vref = \\\$".$varname,"\n" if $DEBUG;
	eval "\$vref = \\\$".$varname;
	if ($DEBUG) {
	    eval "\$".$varname."=17";
	    print STDERR "vref=",$$vref,"\n";
	}
    } else {
	$vref = (ref($hashref) eq "HASH" ? \$hashref->{$name} : $hashref);
    }    

    &optrega($vref,$type,$name,$longname,$help);

}

## Opt::getoptions is a single function call that does it all; it
## registers options according to its argument list (each argument
## is an anonymous array which is a list of arguments to be sent
## to "Opt::uniregister); finally, Opt::opt is called.
sub getoptions_usage {
    my $msg = shift;
    print STDERR "getoptions error: $msg\n";
    print STDERR "Usage: arguments are anon arrays\n";
    print STDERR "Opt::getoptions(['-x'],          ## traditional flag\n";
    print STDERR "                ['-ySTRING'],    ## traditional parameter\n";
    print STDERR "                ['--zvalue=INT'] ## POSIX parameter\n";
    print STDERR "                ['--etc','Help for etc',\\\$etc]\n";
    die "\n";
    
}
sub getoptions {

    my $argref;
    while ($argref = shift) {
	getoptions_usage "Invalid option specification" 
	    if (ref($argref) ne "ARRAY");
	register(@$argref);
    }
    @ARGV = opt($0,@ARGV);
}
	
########################################### Hooks

sub optMain {
    $optMainRoutine = shift;
}
sub optEnv  {
    $optEnvName = shift;
    print STDERR "Env=$optEnvName\n" if $DEBUG;
}

########################################### Internal functions

sub opt_help {

    print "Usage: $progname [options]\n";
    print "To invoke the menu, type:\n";
    print "       $progname --menu\n";
    print "The options are:\n";

    foreach $opt (@optList) {
	my $line=" ";
	if ($$opt{name} ne "") {
	    $line .= "-$$opt{name}";
	    $line .= ($$opt{longname}) ? ", " : "  ";
	} else {
	    $line .= "    ";
	}
	if ($$opt{longname} ne "") {
	    $line .= "--$$opt{longname} ";
	}
	$line .= " " x (20-length($line));
	$line .= "<" . $$opt{type} . ">";
	$line .= " " x (35-length($line));
	$line .= $$opt{descript};
	print $line,"\n";
    }
    print "--\n";
}
sub opt_tofile {
    my $file = shift;
    if ($file eq '%') {     
	my $prog = $progname;
	$prog =~ s,\.[^/]*$,,;	# remove trailing extensions
	$file = "$prog.opt"; 
    }

    print STDERR ">>>> output to file: $file\n" if $DEBUG;
    open(OUT,">$file") || warn "Cant open file [$file]: $!\n";
    foreach $opt (@optList) {
	my $line="";
	if ($$opt{name} ne "") {
	    $line .= "-$$opt{name}";
	    $line .= " " unless ($$opt{type}=~/FLAG/ or $$opt{type}=~/LEVEL/);
	} elsif ($$opt{longname} ne "") {
	    $line .= "   --$$opt{longname}=";
	}
	$line .= $opt->stringval;
	print OUT $line;
	print OUT (" "x(30-length($line))," ; ",$$opt{descript})
	    if ($$opt{descript});
	print OUT "\n";
    }
    close OUT;
}
sub opt_startmenu {
    $optMenuMode = ($optMenuMode ? 0 : 1);
    &opt_writemenu if $optMenuMode;
    while ($optMenuMode) {
	my @argv = &opt_frommenu();
	opt_process(@argv);
    }
}
sub opt_writemenu {
    foreach $opt (@optList) {
	my $line="";
	if ($$opt{name} ne "") {
	    $line .= "$$opt{name} ";
	    $line .= $$opt{descript} if $$opt{descript};
	} elsif ($$opt{longname} ne "") {
	    $line .= "   --$$opt{longname}";
	}
	$line .= " "x(30-length($line));
	$line .= $opt->menuval;
	print "$line\n";
    }
}
sub opt_frommenu {
    print "->";
    my $line = <STDIN>;
    chomp($line);
    &opt_writemenu unless $line;
    $line = '-' . $line if $line =~ /^\w/;
    @argv = &line2argv($line);
}
    
sub opt_fromfile {
    my $file = shift;
    if ($file eq '@') {     
	my $prog = $progname;
	$prog =~ s,\.[^/]*$,,;	# remove trailing extensions
	$file = "$prog.opt"; 
    }
    print STDERR ">>>> atsign $file\n";
    open(IN,$file) || warn "Cant open file: $!\n";
    while (<IN>) {
	chomp;
	## remove all the ;'s
	next if /^\s*;/;
	s/;.*$//;
	## now call opt recursively, using the line itself
	## as the new argv;
	&opt_process(&line2argv($_));
    }
    close IN;
}
sub line2argv {
    ## used by both &opt_fromfile, and &opt_frommenu;
    use Text::ParseWords;
    my $line = shift;
    @argv = quotewords("[ \t\n\r\f]+",0,$line);
}
    
## opt reads a command-line argument list (passed as the argument)
## parses the options, sets variables, etc.; it returns that unprocessed
## portion of the argument list
sub opt {

    print STDERR "EXPORT=",join(":",@EXPORT),"\n" if $DEBUG;

    $progname = shift;
    if (defined($_[0]) and $_[0] eq "--help") {
	&opt_help();
	exit;
    }
    ## Initialize
    %optHash=();
    @optUndelim=();		# is there any occasion where this
				# needs to be reset??
    foreach $opt (@optList) {
        ## put both names into the hash 
        $optHash{$opt->{name}}     = $opt if $opt->hasname;
        $optHash{$opt->{longname}} = $opt if $opt->haslongname;

	## put $opt in list of undelimited options
	push @optUndelim,$opt if $opt->{type} =~ /UNDELIM/;
    }

    ## Check to see if there is an environment variable
    if (exists $ENV{$optEnvName}) {
	&opt_process(line2argv($ENV{$optEnvName}));
    }
    ## Now process the arguments on the command line	    

    &opt_process(@_);
}
sub opt_process {
    my @argv = @_;		# this copies the arg list; thus there is
				# no "destruction" of @ARGV
    my @retargv = ();
    while ($_=shift @argv) {

	## a plain '--' ends argument processing
	last if /^--$/;

	if (/^=/ and $optMenuMode) {
	    my @arg = @argv;
	    &$optMainRoutine(@arg);
	    1 while shift @argv;
	    next;
	}
	if (/^\$$/ || /^--menu$/) {
	    &opt_startmenu();
	    next;
	}
	if (/^\.$/) {
	    exit;
	}
	if (/^@/) {
	    my ($file) = /^@(.*)/;
	    &opt_fromfile($file);
	    next;
	}

	if (/^%/) {
	    my ($file) = /^%(.*)/;
	    &opt_tofile($file);
	    next;
	}

	## all options start with a '-'
	## anything that doesn't is added to the returned argv
	## this is arguably too "permissive", as it allows 
	## options to be parsed after the main arguments, eg
	## 'cat file -n' works the same way as 'cat -n file'.
	## if you don't like opt to be so permissive, then
	## instead do something like:
	## push @retargv,$_ while (shift @argv);
	unless(/^-/) {
	    ## If there are any UNDELIM opts waiting to be filled,
	    ## then fill them here.  Note we have ruled out the
	    ## use of undelim strings that start with '-'
	    if (@optUndelim) {
		$opt = shift @optUndelim;
	    } else {
		push @retargv,$_;
		next;
	    }
	}
	
        ##                             name, rest
	##                            ------------
	## Cases here 1/ -xval     ==> x,    val
        ##            2/ --xval    ==> xval, undef
	##            3/ --xval=   ==> xval, ""
	##            4/ --xval=8  ==> xval, 8

	## Strip the leading '-';
	s/^-//;

	my ($name,$rest) = /^(.)(.*)$/;   ## works for Case 1

	if ($name eq "-") {
	    ## ie, if Case 2, 3, or 4

	    ## this works for Case 2
	    $name = $rest;
	    $rest = undef;
	}
	my $eqsign=0;                     ## is there an '=' in the string?
	if ($name =~ /=/) {
	    ## ie, if Case 3 or 4
	    $eqsign=1;
	    ($name,$rest) = ($name =~ /([^=]+)=(.*)/);
	}

	print STDERR ">> name=$name, rest=$rest\n" if $DEBUG;
	## Make sure that $name is a valid name
	if ($name ne "" and not exists $optHash{$name}) {
	    warn "Invalid option: $name\n";
	    next;
	} 

	## Go ahead and get the reference to the object
	my $opt = $optHash{$name};

	## If there was an equal sign, then straightforward assignment
	if ($eqsign) {
	    $opt->setval( $rest );
	    next;
	}

	## Check to see if it's a flag;
	## Note this doesn't work for '-x+'
	my $type=$$opt{type};
	if ($type =~ /FLAG/ or $type =~ /INTLEVEL/) {
	    ## First check for cases: -x+ or -x-
	    ## Hmmm, do we want to consider -x0 and -x1 ??
	    if ($rest =~ /^([+-])/) {
		($val,$rest) = ($rest =~ /^([+-])(.*)$/);
		$opt->setval( $val );
	    }
	    ## Then do INTLEVEL case
	    elsif ($type =~ /INTLEVEL/) {
		$opt->increment;
	    }
	    ## Check for ABS flags
	    elsif ($type =~ /ABS/) {
		$opt->setval( 1 );
	    } 
	    ## If none of these, then it's a TOGGLE flag
	    else {
		$opt->toggle;
	    }
	    ## if we had -xyz, and -x was a flag, then
	    ## put "-yz" as the next option 
	    unshift @argv, "-".$rest if $rest ne "";

	    next;
	}

	## If it's not a flag
	$rest = shift @argv if (not defined($rest) or $rest eq "");
	$opt->setval($rest);

    }
    push @retargv,@argv if @argv;
    return @retargv;
}
1;
__END__


=head1 NAME

Opt - parsing of command line options, and more...

=head1 SYNOPSIS

    use Opt;
    Opt::getoptions(["-x","Help string for x",\$x],
                    ["-yINT"],
                    ["--zvalue=STRING"], 
                    ["--longvar",\%LocalHashOfOptions]
                   );


and then the values associated with those options will be assigned
to variables   
C<$x>, C<$OPT{y}>, C<$OPT{zvalue}>, and C<$LocalHashOfOptions{longvar}>.

=head1 DESCRIPTION

The B<Opt> module is a library whose purpose is to simplify the task of
parsing command line arguments to set parameter values for a B<Perl>
routine.  B<Opt> also gets parameter values from environment variables, 
option files, and a builtin rudimentary interactive menu.
It is much like the
standard Getopt modules, but there are a few more bells and whistles.
It is mostly meant to be a clone of the similarly named 'opt' for C
programs (in fact, this version of F<Opt.pm> was probably distributed
with the larger C library).  

B<Opt> supports simultaneous use of traditional, bundled, and long (aka POSIX) 
options

=over 4

=item -x

turns on the 'x' feature of your code; in this traditional 
specification, the option specifier 'x' is only permitted to be
one character, usually alphabetic of either case, but Opt.pm also
permits other characters, such as the digits 0-9, so you can specify
'-2' to indicate that you want a two-sided statistical test.  Most
punctuation would not be permitted.

=item -x -y

turns on the 'x' and 'y' features

=item -xy

turns on both 'x' and 'y' features; putting both options together like
this is called "bundling".

=item -z 3.14159, 

specifies that the number associated with 'z' should have value 3.14159.

=item -z3.14159

does the same thing, even though there is no space between the 'x' and
'3.14159'; that space is optional.

=item --xflag

turns on the flag associated with "xflag"; note the two hyphens and 
that "xflag" is not constrained to be a single character.

=item --zvalue 3.14159, --zvalue=3.14159

are both permitted, but "--zvalue3.14159" is not allowed.

=back

as well as some extra bells and whistles specific to B<Opt>.

=over 4

=item @file.opt

specifies that options are read from a file named "file.opt".

=item %file.opt

tells the program to write all the current values of all the 
options to a file called file.opt.  If you want a "file.opt" template
as a starting point which you might then edit by hand, you can always
get one by typing C<program %file.opt .>; using a period ('.') as
an argument causes the program to exit gracefully.

B<Aside:>
Hmmm, this could be a problem for a program in which '.' might be a
reasonable argument, eg a program that was expecting a directory name.
Perhaps, we should instead use something like '---exit' instead?

=item --help

is a special flag which, if it is the first in a command line,
tells the program to write a help message and exit.  The message is
of a "standard" format (standard to 'opt' that is) which can be parsed
by other programs, such as 'tkopt' which instantly turns any opt-enabled
program to be GUI too.

=item --menu

pops you into an interactive menu which allows you to see what parameters
are available, what they do, and what their current values are.  you can then
reset those parameters to what you want and run the program.

=back

=head1 FUNCTIONS

=over 4

=item C<getoptions(["--var=>I<TYPE>C<","Help string",\$var], ['-x'], ... )>

In principle, C<getoptions> is the only function you need to know about
in order to use the B<Opt> package.  Each argument "registers" a single
option, which essentially sets up a table that associates each option with attributes specifying
what kind of option it is, and which variable in the B<Perl> script it
corresponds to.  After all the options are registered, the C<@ARGV> array
is parsed, and the variables are all set accordingly.  This is the
one-function-does-all invocation; you may prefer to call its individual
components: for instance, the example in the L<SYNOPSIS> is equivalent to:

    Opt::register("-x","Help string for x",\$x);
    Opt::register("-yINT");
    Opt::register("--zvalue=STRING");
    Opt::register("--longvar",\%LocalHashOfOptions);
    @ARGV=Opt::opt($0,@ARGV);

The longer form is a little closer to what the interface would look
like in the C version of B<opt>.  You may prefer this form
if, say, you want to perform argument processing on an array other than @ARGV.

=item register("--var=TYPE","Help string",\$var)

This function registers a single option of type TYPE, named "var", and
associates it with the variable C<$var>.  Invocations of "--var=value"
on the command line will lead to setting C<$var="value"> in the
B<Perl> script.  The help string is optional, as is the C<$var>
reference.  If the variable reference is not specified, then a
variable C<$OPT{var}> is created, and it is associated with the
option.  TYPE is one of: C<NUL>, C<INT>, C<UNSINT>, C<SHORT>, C<LONG>,
C<CHAR>, C<INTLEVEL>, C<FLOAT>, C<DOUBLE>, C<FLAG>, C<NEGFLAG>,
C<ABSFLAG>, C<ABSNEGFLAG>, C<STRING>, or C<UNDELIM>.

=item optreg(\$var,"TYPE",'v',"Help string")

This is another version of the C<register> function.  It is arguably
not as intuitive as C<register>, but it more closely matches the way
that options are registered in the C version of B<opt>.  This is not
the most convenient way to register a function, but it mimics the C
opt version.  This form only registers one-character option names.
Alternative forms include:

=over 4

=item optrega(\$var,"TYPE",'v',"var","Help string")

Provides two names for the variable C<$var>, a long name ("var") and a
short name ('v').  As an implementation issue, all other registration
functions, including the ones below as well as C<register> and even
C<getoptions>, all call C<optrega> to actually register the option.

=item optregc(\$var,"TYPE",'v')

only provides a short single-character name ('v') for the variable C<$var>.

=back

Each of these functions also has a form C<optreg_TYPE(\$var,'v',"Help string")>,
in which the TYPE is not a string argument but is part of the function name.

=item @argvnew=opt($0,@argv)

After all the options are registered, the function C<opt> does the
actual parsing of command line as given in the array C<@argv>.
Here C<$0> is the name of the program.  Note that C<opt> does
"nondestructive" argument processing; so the argument C<@argv> is unaltered
by the call to C<opt>.  The result C<@argvnew> is the list of arguments that
are unprocessed by C<opt>.  In typical usage, you would write
C<@ARGV=Opt::opt($0,@ARGV);>

=head2 HOOKS

Hooks are functions which are written by the B<Perl>
application programmer (but not by the B<Opt> developer), which B<Opt> calls
at certain points in its processing.  Among them are:

=over 4
 
=item C<optMain(\&MyMain);>

specifies that the function C<&MyMain> will be run whenever the "=" is
invoked on the opt menu.

=item C<optEnv("ENV_VARIABLE");>

specifies that the environment variable can be used to specify default
options.  eg, if C<ENV_VARIABLE='-x+ -y-'>, then the default value for
'x' will be TRUE, and for 'y' will be FALSE.

=item C<useHash(\%OPTION);>

specifies that options registered without an explicitly associated
reference will be associated with the hash %OPTION.  That is,

    %OPTION=();
    useHash(\%OPTION);
    Opt::register("-x");
    
will associate the option '-x' with the variable C<$OPTION{x}>.
By default, the package does an equivalent of C<useHash(\%OPT)>,
unless that is changed by this function or by the C<useScalar>
function.

=item C<useScalarPrefix("opt_");>

specifies that options registered without an explicitly associated
reference will be associated with the scalar string C<$opt_>I<xxx>,
where I<xxx> is the name of the option.

=item C<useInitFile("~/.optrc")>

still not implemented (not in the C package either), but it would be
a nice idea.  Equivalent to "%~/.optrc" as the first line of the command
line arguments.  Should maybe be a list of dot files, so that they are
used in series, eg C<UseDotFile("/etc/optrc","~/.optrc");>

=back

=head1 BUGS

Given how full-featured and how much more mature the B<Getopts::Long> 
package is, the very existence of this package is arguably a bug.  
The I<justification> is that
by making it look (to a user of the program) just like the B<C> version of
B<opt>, a program such as B<tkopt> will be useable on both B<C> and
B<Perl> routines.  
It (this package, B<Opt>) does have a few more bells and whistles,
including: support for option files, help strings, and an interactive 
menu, as well as the ability to read default parameters from an
environment variable or a dot file.

The real I<reason> I wrote B<Opt> was to learn
about packages and object-oriented programming in B<Perl>.

=over 4

=item *

Do not currently support setting C<$opt_>I<xxx> variables, as is
done in the standard C<GetOpts> packages.  Well, it sort of does,
but it is not exporting the variable names: you have to ask for
C<$Opt::opt_>I<xxx> to get your variable.

It is pushed onto the @EXPORT array, but I think it's too late!

=item *

Not all of the B<C-opt> functionality is implemented.

=item *

Some extra functionality, not provided in B<C-opt>, is implemented

=item *

Don't be fooled by the F<Makefile.PL> file; it's just a stub until
I figure out how to make a real one.  However, at the moment, all you
have to do to install you just copy F<Opt.pm> and F<Option.pm> into a 
directory in your B<PERLLIB> list.

=back

=head1 COPYRIGHT

Copyright (C) 1998, James Theiler; email: jt@lanl.gov

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

=cut
