/*	File:	  pending.c
 *      Author:	  Lars Ole Andersen (lars@diku.dk)
 *		  Peter Holst Andersen (txix@diku.dk)
 *                Arne John Glenstrup (panic@diku.dk)
 *                Henning Makholm (makholm@diku.dk)
 *                Jens Peter Secher (jpsecher@diku.dk)
 *      Created:  Tue Jul 27 10:36:33 1993
 *      Content:  C-Mix speclib: pending list.
 *
 *      Copyright  1998. The TOPPS group at DIKU, U of Copenhagen.
 *      Redistribution and modification are allowed under certain
 *      terms; see the file COPYING.cmix for details.
 */
  
#include <stdio.h>
#define cmixSPECLIB_SOURCE
#include "speclib.h"

#ifndef PGENLABEL
/* these definitions is carried out if this file is compiled by itself:
 * when it is included from vpending.cc, that file supplies suitable
 * definitions.
 */
#define PGENLABEL unsigned
#define PENDFUNC(x) x
#endif

typedef struct Pend_node {
  PGENLABEL label;		/* Specialization label */
  cmixLabelTag reslab;		/* Residual label       */
  cmixSavedState state;		/* Copied state         */
  struct Pend_node *next;	/* List link            */
} Pend_node ;

typedef struct Pend_stack {
  Pend_node *seen, *pend;	/* Seen and pend lists  */
  struct Pend_stack *next;	/* Next in stack        */
} Pend_stack ;

/* Pending stack */
static Pend_stack *cur_pend = NULL ;
/* recentSetSIze is the number of pending points at front of the list
 * that are "recent", i.e, cmixPending() has not been called since they
 * were added
 */
static int recentSetSize = 0;

/*======================================================================*
 * Pending list routines						*
 *======================================================================*/

/* Insert specialization point into current pending list and return label */
cmixLabelTag
PENDFUNC(cmixPendinsert)(PGENLABEL lab,
                         cmixDataObject locals[], cmixMemoWhat what)
{
    Pend_node *p;

    /* Check to see whether it's already here */
    for (p = cur_pend->pend; p != NULL; p = p->next)
        if (p->label == lab &&
            cmixCmpLightState(p->state))
              return p->reslab;
    for (p = cur_pend->seen; p != NULL; p = p->next)
	if (p->label == lab &&
	    cmixCmpLightState(p->state))
              return p->reslab;

    /* Not found: allocate new node and insert */
    p = NEW(Pend_node);
    p->label = lab;
    p->reslab = cmixMakeLabel() ;
    p->state = cmixCopyLightState(locals, what);
    p->next = cur_pend->pend;
    cur_pend->pend = p;
    recentSetSize++;
    return p->reslab;
}

/* Check pending list and return next specialization point if non-empty */
PGENLABEL
PENDFUNC(cmixPending)(void)
{
  /* Return 0 if current pending list is empty */
  if (cur_pend->pend == NULL)
    return NULL;
  else {
    /* Move node from pending list to seen list, restore memory configuration
     * and return label.
     */
    Pend_node *p = cur_pend->pend;
    cur_pend->pend = p->next;
    p->next = cur_pend->seen;
    cur_pend->seen = p;

    /* Only waste time on restoring if the node is not recent */
    if (!recentSetSize)
      cmixRestoreState(p->state);
    /* Empty the set of recent nodes */
    recentSetSize = 0;

    cmixLabel(p->reslab);
    return p->label;
  }
}

/*======================================================================*
 * Pending stack routines						*
 *======================================================================*/

/* Push current pending and seen lists onto pending stack and initialize new */
void
PENDFUNC(cmixPushPend)(void)
{
    Pend_stack *p = NEW(Pend_stack);
    p->pend = p->seen = NULL ;
    p->next = cur_pend;
    cur_pend = p;
    recentSetSize = 0;
}

/* Pop pending and seen lists from pending stack and make them current */
void
PENDFUNC(cmixPopPend)(void)
{
    Pend_node *p, *p1;
    Pend_stack *q;
    
    /* Check to see whether the stack is empty */
    if (cur_pend == NULL)
	cmixFatal("Pending: pop on empty pending stack");

    /* Check memusage before we free memory */
    cmixCheckMemusage();

    /* Delete the seen lists. (pending is already empty)
     * When the seen list is deleted, copied objects are deleted as well
     */
    for (p = cur_pend->seen; p; p = p1) {
	p1 = p->next;
        cmixFreeState(p->state);
	FREE(p);
    }

    /* Restore */
    q = cur_pend;
    cur_pend = cur_pend->next;
    FREE(q);
    recentSetSize = 0;
}
