/* Authors:  Jens Peter Secher (jpsecher@diku.dk)
 *           Arne Glenstrup (panic@diku.dk)
 *           Henning Makholm (makholm@diku.dk)
 * Content:  C-Mix gegen subsystem: memoization-related stuff
 * History:  Derived from theory by Peter Holst Andersen and Lars Ole Andersen.
 *
 * Copyright  1999. The TOPPS group at DIKU, U of Copenhagen.
 * Redistribution and modification are allowed under certain
 * terms; see the file COPYING.cmix for details.
 */

#include <cmixconf.h>
#include "gegen.h"
#include "options.h"

void
GegenMemo::EmitMemoizer(const char *name1,const char *name2)
{
  ForCascade fc(MemoBaseType,up,3);
  if ( MemoUserType ) {
    up.ost << "if (!cmix_follow" << UserMemo[MemoUserType]
           << "(cmixCallback,cmixCopy," ;
    if ( MemoUserType != fc.t2->user_def() )
      up.ost << "(struct " << up.names[MemoUserType] << "*)" ;
    up.ost << '&' ;
  } else {
    up.ost << "if (!cmixCallback(cmixCopy," ;
  }
  up.ost << name1 << name2 << fc << ")) return 0;\n" ;
  fc.close() ;
}

bool
GegenMemo::AnyPointersToFollow(C_Type *t)
{
  MemoBaseType = t ;
    
  C_Type *rt = t->array_contents_type() ;
  switch(rt->tag) {
  default:
    return false ;
  case StructUnion:
    MemoUserType = rt->user_def() ;
    while ( MemoUserType->isUnion ) {
      C_Type *fmt
        = MemoUserType->instances.front()->user_types().front();
      if ( fmt->tag != StructUnion )
        return false ;
      MemoUserType = fmt->user_def() ;
    }
    return UserMemo[MemoUserType] != 0 ;
  case Pointer:
    MemoUserType = NULL ;
    return up.spectime(rt->ptr_next()) ;
  }
}

class CompareBindingTimes : public TypeCompareCallback {
  BTresult const &bt;
public:
  CompareBindingTimes(BTresult const &b) : bt(b) {}
  virtual bool deem_different(C_Type const&a,C_Type const&b) const {
    return bt.Dynamic(a) != bt.Dynamic(b);
  }
};

void
GegenMemo::AlocMemodata(C_Decl *d)
{
  if ( !up.spectime(d) )
    return ;
  // XXX if no functions or basic blocks ever need to memoise
  // the object we can bail out now - but we don't because that
  // information is not yet available.

  IdNumber[d] = ++IdCount ;

  if ( !AnyPointersToFollow(d->type) )
    return ;
  // Try to find an object whose type is the same as this one's.
  foreach(prev,memofun_leaders,Plist<C_Decl>)
    if ( d->type->equal(*prev->type,CompareBindingTimes(up.bt))) {
      MemoNumber[d] = MemoNumber[*prev] ;
      return ;
    }

  // This binding-time type has not been seen before. Create
  // a new pointer following function
  MemoNumber[d] = ++fcncounter ;
  memofun_leaders.push_back(d);
  up.oksection("Memoization helper functions")
    << "static int cmix_follow" << fcncounter << "\n"
    "    (cmixPointerCallback cmixCallback, "
    "cmixDataObjectCopy **cmixCopy, void *cmixVoid, "
    "unsigned cmixI) {\n  "
    << EmitType(d->type,"(*const cmixSrc)",d->qualifiers)
    << " = (" << EmitType(d->type,"(*)",d->qualifiers) << ")cmixVoid ;\n"
    "  while ( cmixI-- )\n" ;
  EmitMemoizer("cmixSrc[cmixI]","") ;
  up.ost << "  return 1;\n"
    "}\n" ;
}

void
GegenMemo::follow_functions()
{
  // Helper functions for static struct type
  // .. only emitted when there's actually any pointers to follow
  // .. their numbers are recorded in the UserMemo array
  foreach(u,up.pgm.usertypes,Plist<C_UserDef>) {
    if ( u->isUnion || up.bt.Dynamic(u->instances.front()) )
      continue ;
    bool begun = false ;
    foreach(j,**u,C_UserDef) {
      if ( !AnyPointersToFollow(j.type()) )
        continue ;
      if ( !begun ) {
        begun = true ;
        UserMemo[*u] = ++fcncounter ;
        up.oksection("Memoization helper functions")
          << "static int\ncmix_follow" << fcncounter
          << "(cmixPointerCallback cmixCallback, "
          "cmixDataObjectCopy **cmixCopy, struct "
          << up.names[*u] << " *cmixSrc)\n{\n" ;
      }
      EmitMemoizer("cmixSrc->",up.names[j.name()]) ;
    }
    if ( begun )
      up.ost << "   return 1;\n}\n" ;
  }
    
  // Make ID numbers and memoization helper functions
  foreach(v,up.pgm.globals,Plist<C_Decl>)
    // Spectime global vars are memoized unless they are
    // - VarExtSpectime (hence the 'dangerous spectime:' syntax
    //   for declaring them), or
    // - never side-effected at all (constant tables etc.)
    //   (in which case they can only point to other global
    //   variables, so there is no need to follow pointers
    //   in them either).
    if ( v->varmode() != VarExtSpectime &&
         up.pa.sideEffectedObjects.find(*v) )
      AlocMemodata(*v);
  foreach(vv,up.pgm.heap,Plist<C_Decl>) {
    assert(vv->type->tag==Array);
    AlocMemodata(vv->members().front());
  }
  foreach(f,up.pgm.functions,Plist<C_Decl>) {
    foreach(p,f->fun_params(),Plist<C_Decl>)
      AlocMemodata(*p);
    foreach(v,f->fun_locals(),Plist<C_Decl>)
      AlocMemodata(*v);
  }

  // Clear the temporary data structures that were used to implement
  // sharing of functions for like types.
  memofun_leaders.clear();
  UserMemo.clear();
}

void
GegenMemo::InitDO(C_Decl *d,bool local)
{
  if ( IdNumber[d] ) {
    up.ost << "\n\t{" << IdNumber[d]-1 << ',' ;
    if ( local )
      up.ost << '0' ;
    else
      up.ost << '&' << up.names[d] ;
    up.ost << ",sizeof " << up.names[d]
           << ", cmix_follow" << MemoNumber[d] << "}," ;
  }
}

void
GegenMemo::emit_globalmemodata()
{
  up.ost << "static cmixDataObject cmix_globals[] = {" ;
  foreach(i,up.pgm.globals,Plist<C_Decl>)
    InitDO(*i,false);
  up.ost << " {0} };\n" ;
}

void
GegenMemo::emit_globalmemocode()
{
  up.ost << "  /* memoisation selector strings\n";
  foreach(fun,up.pgm.functions,Plist<C_Decl>) {
    up.ost << "   * (";
    emit_funmemo(*fun);
    up.ost << ")\n" ;
  }
  foreach(gen,up.pgm.generators,Plist<C_Decl>) {
    up.ost << "   * entry point " << gen->get_name() << ": (";
    emit_funmemo(*gen);
    up.ost << ")\n" ;
  }
  up.ost << "   */\n" ;

  up.ost << "  cmixPushGlobals(cmix_globals);\n" ;
}

unsigned
GegenMemo::emit_localmemodata(C_Decl *fun,bool funmemo, bool bbmemo)
{
  unsigned localcounter, totalcounter ;
  Plist<char const> addresses ;
  foreach(v,fun->fun_locals(),Plist<C_Decl>)
    if ( IdNumber[*v] )
      addresses.push_back(up.names[*v]) ;
  localcounter = addresses.size() ;
  foreach(vv,fun->fun_params(),Plist<C_Decl>)
    if ( IdNumber[*vv] )
      addresses.push_back(up.names[*vv]) ;
  totalcounter = addresses.size() ;

  if ( totalcounter || bbmemo || funmemo ) {
    up.ost << "  cmixDataObject cmix_locals[] = {" ;
    foreach(v,fun->fun_locals(),Plist<C_Decl>)
      InitDO(*v,true);
    foreach(vv,fun->fun_params(),Plist<C_Decl>)
      InitDO(*vv,true);
    up.ost << " {0} };\n" ;
  }

  if ( funmemo )
    up.ost << "  cmixDataObject * const cmix_params = cmix_locals + "
           << localcounter << ";\n" ;

  unsigned u = 0 ;
  foreach(n,addresses,Plist<char const>)
    up.ost << "  cmix_locals[" << u++ << "].obj = &" << *n << ";\n" ;
  
  if ( totalcounter )
    up.ost << "  cmixPushLocals(cmix_locals);\n" ;
  return totalcounter ;
}
        
void
GegenMemo::emit_pendinsert(C_Decl *fun,C_BasicBlock *bb,unsigned bbnumber)
{
  if ( use_assigned_gotos )
    up.ost << "cmixPendinsert_v(&&cmixL" << bb->Numbered_ID ;
  else
    up.ost << "cmixPendinsert(" << bbnumber ;
  up.ost << ",cmix_locals," ;
  df.EmitMemoParam(bb,IdNumber,IdCount,up.ost);
  up.ost << ')' ;
}

void
GegenMemo::emit_funmemo(C_Decl *fun)
{
  up.ost << '"' << up.names[fun] << "\",cmix_params, " ;
  df.EmitMemoParam(fun,IdNumber,IdCount,up.ost);
}

GegenMemo::GegenMemo(GegenEnv &env,DFresult const &DF)
  : up(env), df(DF),
    fcncounter(0), UserMemo(0), IdNumber(0), IdCount(0), MemoNumber(0)
{}
