/* -------------------------------------------------------------------- */
/* SMS Client, send messages to mobile phones and pagers		*/
/*									*/
/* cimd.c								*/
/*									*/
/*  Copyright (C) 1997,1998,1999 Angelo Masci				*/
/*									*/
/*  This library is free software; you can redistribute it and/or	*/
/*  modify it under the terms of the GNU Library General Public		*/
/*  License as published by the Free Software Foundation; either	*/
/*  version 2 of the License, or (at your option) any later version.	*/
/*									*/
/*  This library is distributed in the hope that it will be useful,	*/
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of	*/
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU	*/
/*  Library General Public License for more details.			*/
/*									*/
/*  You should have received a copy of the GNU Library General Public	*/
/*  License along with this library; if not, write to the Free		*/
/*  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.	*/
/*									*/
/*  You can contact the author at this e-mail address:			*/
/*									*/
/*  angelo@styx.demon.co.uk						*/
/*  ???									*/
/*  mattias								*/
/*									*/
/* -------------------------------------------------------------------- */
/* $Id$
   -------------------------------------------------------------------- */

#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "common/common.h"
#include "logfile/logfile.h"
#include "driver.h"
#include "comms/comms.h"
#include "error.h"
#include "resource/resource.h"
#include "ascii.h"

/* -------------------------------------------------------------------- */

static struct cimd_env
{
	DRIVER_DEFAULT_ENV def;

	/* Place any extended driver	*/ 
	/* variables here 		*/

	char	*login,
		*password,
		*message_terminator;

	long 	login_delay;
	

} driver_env;


/* -------------------------------------------------------------------- */

static 	RESOURCE resource_list[] = 
	{
		{ RESOURCE_STRING,  "SMS_comms_params", 	0, 0, NULL, 0,  "8N1",        0, 	&(driver_env.def.comms_params)  	},
		{ RESOURCE_STRING,  "SMS_centre_number", 	0, 0, NULL, 0,  "0708222901", 0, 	&(driver_env.def.centre_number)  	},
		{ RESOURCE_NUMERIC, "SMS_baud", 		0, 0, NULL, 0,  NULL,         9600, 	&(driver_env.def.baud)  		},
		{ RESOURCE_NUMERIC, "SMS_deliver_timeout", 	0, 0, NULL, 0,  NULL,         45,   	&(driver_env.def.deliver_timeout)  	},
		{ RESOURCE_NUMERIC, "SMS_timeout", 		0, 0, NULL, 0,  NULL,         10,   	&(driver_env.def.timeout)  		},
		{ RESOURCE_NUMERIC, "SMS_write_timeout", 	0, 0, NULL, 0,  NULL,         10,   	&(driver_env.def.write_timeout)  	},
		{ RESOURCE_NUMERIC, "SMS_max_deliver", 		0, 0, NULL, 0,  NULL,         1,    	&(driver_env.def.max_deliver)  		},
		{ RESOURCE_STRING,  "SMS_login", 		0, 1, NULL, 0,  "EURO_CIMD",  0,   	&(driver_env.login)  			},
		{ RESOURCE_STRING,  "SMS_password", 		0, 1, NULL, 0,  "2062",       0,    	&(driver_env.password)  		},
		{ RESOURCE_STRING,  "SMS_message_terminator", 	0, 0, NULL, 0,  "\n",         0,    	&(driver_env.message_terminator)  	},
		{ RESOURCE_NUMERIC, "SMS_login_delay", 		0, 0, NULL, 0,  NULL,         2000000, 	&(driver_env.login_delay)  		},
		{ RESOURCE_NULL,     NULL, 			0, 1, NULL, 0,  NULL,         0,    	NULL  					}
	};

/* -------------------------------------------------------------------- */

#define DELIVERTIMEOUT 		(driver_env.def.deliver_timeout)
#define TIMEOUT 		(driver_env.def.timeout)
#define WRITETIMEOUT 		(driver_env.def.write_timeout)

/* -------------------------------------------------------------------- */

#define MSGTERM			(driver_env.message_terminator)
#define FD			(driver_env.def.fd)

/* -------------------------------------------------------------------- */

static int CIMD_login(void);
static int CIMD_sendmessage(char *msisdn, char *message);
static int CIMD_send_disconnect(void);
static void CIMD_hangup(void);
static void calculate_checksum(char *checksum, char *ptr);

/* -------------------------------------------------------------------- */
/* Generate a checksum for message pointed to by 'ptr'			*/
/* Place resulting 2 Character checksum in buffer pointed to		*/
/* by 'checksum' add an addition '\0' character to terminate 		*/
/* the string.								*/
/* -------------------------------------------------------------------- */
static void calculate_checksum(char *checksum, char *ptr)
{
	int 	sum, 
		high, 
		low;


	sum = 0;
	while (*ptr != '\0')
	{	sum += *ptr++;
		sum &= 0x00FF; /* truncate */
	}

	/* get high and low byte */

	high = (sum >> 4) & 0x000F;
	low  = (sum     ) & 0x000F;

	/* make hex chars and put them in message */
	if (high <10)
	{	checksum[0] = '0' + high;
	} 
	else
	{	checksum[0] = 'A' + (high -10);
	}

	if (low <10)
	{	checksum[1] = '0' + low;
	} 
	else 
	{	checksum[1] = 'A' + (low -10);
	}

	/* set end of string for checksum */
	checksum[2] = '\0';
}


/* -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- */
static int CIMD_login(void)
{
	char 	buf[MAX_RESPONSE_BUFSIZE], 
		message[512], 
		checksum[3];


	/* expect login */
	if( expstr(FD, buf, "login: ", MAX_RESPONSE_BUFSIZE, TIMEOUT) )
	{
		lprintf(LOG_STANDARD, "No login request\n");
		CIMD_hangup();
		return -1;
	}

	lprintf(LOG_STANDARD, "Received login request\n");
	lprintf(LOG_STANDARD, "Sending Service Login...\n");

	/* send "sc3cimd1" to login to service */
	twrite(FD, "sc3cimd1\n", sms_strlen("sc3cimd1\n"), WRITETIMEOUT);

	/* expect to be logged in to the service */
	if (expstr(FD, buf, "CIMD", MAX_RESPONSE_BUFSIZE, TIMEOUT) )
	{
		lprintf(LOG_STANDARD, "No CIMD Service Detected\n");
		CIMD_hangup();
		return -1;
	}

	/* read the rest of the line */
	if (expstr(FD, buf, "\n", MAX_RESPONSE_BUFSIZE, TIMEOUT) )
	{
		lprintf(LOG_STANDARD, "No CIMD Service Detected\n");
		CIMD_hangup();
		return -1;
	}

	lprintf(LOG_STANDARD, "CIMD Service Detected\n");

	sms_usleep(driver_env.login_delay);

	/* build the login packet */
	sms_snprintf(message, 512, "%c%s%c%s%c%s%c", S_STX,
		                                     "01",
		                                     S_HT, 
		                                     driver_env.login, 
		                                     S_HT,
		                                     driver_env.password, 
			   	                     S_HT);

	/* calculate the checksum */
	calculate_checksum(checksum, message);

	/* complete the packet */
	sms_snprintf(message, 512, "%s%s%c%s",message, checksum, S_ETX, MSGTERM);
	
	/* send */
	twrite(FD, message, sms_strlen(message), WRITETIMEOUT);

	/* read a line */
	if (expstr(FD, buf, "\n", MAX_RESPONSE_BUFSIZE, TIMEOUT) )
	{
		lprintf(LOG_STANDARD, "No response\n");
		lprintf(LOG_STANDARD, "CIMD Login failed\n");
		CIMD_hangup();
		return -1;
	}

	/* check for: <STX><ACK><TAB>C:01<TAB>chksum<ETX><LF> */
	if ( !( (buf[0] == S_STX) && (buf[1] == S_ACK) ) )
	{
		lprintf(LOG_STANDARD, "CIMD Login failed\n");
		CIMD_hangup();
		return -1;
	}	
	
	lprintf(LOG_STANDARD, "Completed CIMD Login\n");
	return 0;
}

/* -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- */
static int CIMD_sendmessage(char *msisdn, char *message)
{
	char 	buf[MAX_RESPONSE_BUFSIZE],
		packet[512],
		checksum[3];


	/* build packet */
	sms_snprintf(packet, 512, "%c%s%c%s%c%s%c", S_STX,
	                                            "03", 
	                                            S_HT, 
	                                            msisdn, 
	                                            S_HT, 
	                                            message,
	                                            S_HT);

	/* calculate checksum */
	calculate_checksum(checksum, packet);

	/* build the rest of the packet */
	sms_snprintf(packet, 512, "%s%s%c%s",packet, checksum, S_ETX, MSGTERM);
	
	/* send packet */
	twrite(FD, packet, sms_strlen(packet), WRITETIMEOUT);

	/* check response */
	if (expstr(FD, buf, "\n", MAX_RESPONSE_BUFSIZE, DELIVERTIMEOUT))
	{
		/* timed out */
		lprintf(LOG_STANDARD, "No Message Delivery Response\n");
		CIMD_hangup();
		return -1;
	}

	/* check for: <STX><ACK><TAB>C:03<TAB>chksum<ETX><LF> */
	if ( (buf[0] == S_STX) && (buf[1] == S_ACK) )
	{
		lprintf(LOG_STANDARD, "Received Message Delivery Response\n");
	} 
	else 
	{	lprintf(LOG_STANDARD, "No Message Delivery Response\n");
		CIMD_hangup();
		return -1;
	}
	
	return 0;
}

/* -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- */
static int CIMD_send_disconnect(void)
{

	char 	buf[MAX_RESPONSE_BUFSIZE],
		message[512],
		checksum[3];


	/* build logout packet */
	sms_snprintf(message, 512, "%c%s%c", S_STX, "02", S_HT); 

	/* calculate checksum */
	calculate_checksum(checksum, message);

	/* build the rest of the packet */
	sms_snprintf(message, 512, "%s%s%c%s",message, checksum, S_ETX, MSGTERM);
	
	/* send packet */
	twrite(FD, message, sms_strlen(message), WRITETIMEOUT);

	/* check response */
	if (expstr(FD, buf, "\n", MAX_RESPONSE_BUFSIZE, TIMEOUT))
	{
		/* timed out */
		lprintf(LOG_STANDARD, "No Disconnection Response\n");
		CIMD_hangup();
		return -1;
	}

	/* check for: <STX><Ack><TAB>C:02<TAB>time. Sess:time.Sub:n (M)<TAB>chksum<ETX><LF> */
	if ((buf[0] == S_STX) && (buf[1] == S_ACK) )
	{
		lprintf(LOG_STANDARD, "Received Disconnection Response\n");
	} 
	else 
	{	lprintf(LOG_STANDARD, "No Disconnection Response\n");
		CIMD_hangup();
		return -1;
	}
	
	return 0;
}

/* -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- */
static void CIMD_hangup(void)
{
	default_hangup((DRIVER_DEFAULT_ENV *)(&driver_env));
}

/* -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- */
DEVICE_ENTRY cimd_device = {

	"CIMD",
	"1.0",
	resource_list,
	(DRIVER_DEFAULT_ENV *)(&driver_env),

	default_init,
	default_main,
	default_validate_numeric_id,
	default_dial,
	default_hangup,
	CIMD_send_disconnect,
	default_multiple_counted_deliver,
	CIMD_sendmessage,
	CIMD_login
};
