/* -*-	Mode:C++ -*- */

/*
 * process_scan_freebsd.cc
 * Copyright (C) 1999 by John Heidemann
 * $Id: process_scan_freebsd.cc,v 1.10 1999/09/06 18:16:16 johnh Exp $
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 */


#ifdef USE_PROCESS_SCAN_FREEBSD

#include <assert.h>

#include <time.h>

extern "C" {
#include <stdio.h>
#include <err.h>
#include <string.h>
#include <vis.h>
#include <limits.h>
#include <sys/param.h>
#include <sys/sysctl.h> // KERN_PROC_ALL
#include <sys/types.h>
#include <sys/user.h>
#include <fcntl.h>
#include <kvm.h>
}

#include "process_scan.hh"
#include "main.hh"


class process_scan_freebsd : public process_scan {
private:
	process_scan_freebsd(const process_scan_freebsd&);
	process_scan_freebsd& operator=(const process_scan_freebsd&);

protected:
	static kvm_t *kvm_;
	struct kinfo_proc *kp_;
	int nentries_;
	struct kinfo_proc *cur_kp_;

	int pages_to_kb(int pages);
	int ticks_to_msec(int pages);

	char *get_cmd();

public:
	process_scan_freebsd();
	virtual ~process_scan_freebsd();

	virtual bool next();
	virtual process_model *birth();
	virtual void life(process_model *pm);

	virtual int cur_pid() { return int(cur_kp_->kp_proc.p_pid); }
	virtual int cur_uid() { return int(cur_kp_->kp_eproc.e_pcred.p_ruid); }
};



process_scan *
process_scan::open_platform()
{
	// someday may fall back if /proc isn't available
	return (process_scan*)new process_scan_freebsd();
}

kvm_t *process_scan_freebsd::kvm_ = NULL;

process_scan_freebsd::process_scan_freebsd() :
	process_scan(),
	kp_(NULL),
	nentries_(0),
	cur_kp_(NULL)
{
	char errbuf[_POSIX2_LINE_MAX];
	if (!kvm_) {
		kvm_ = kvm_openfiles(NULL, NULL, NULL, O_RDONLY, errbuf);
		if (!kvm_)
			die("error opening kvm (probably need setgid kmem\n");
	};
	kp_ = kvm_getprocs(kvm_, KERN_PROC_ALL, 0, &nentries_);
	assert(kp_ != NULL);
	cur_kp_ = &kp_[-1];
}

process_scan_freebsd::~process_scan_freebsd()
{
	// freebsd kvm lib handles overwriting
}

int
process_scan_freebsd::pages_to_kb(int pages)
{
	ENTRY_TRACE(__FILE__,__LINE__);
	static int pages_per_kb = 0;
	if (!pages_per_kb)
		pages_per_kb = getpagesize() / 1024;
	return pages * pages_per_kb;
}

int
process_scan_freebsd::ticks_to_msec(int ticks)
{
	// xxx:
#ifndef HZ
#define HZ 100
#endif
	ENTRY_TRACE(__FILE__,__LINE__);
	return ticks * 1000 / HZ;
}

#if 0
// this stuff is straight from FreeBSD's ps:
// (But we don't need it because we only want p_comm, not the whole cmd line.)
static char *
shquote(char **argv)
{
	char **p, *dst, *src;
	static char buf[4*ARG_MAX];		/* XXX */

	if (*argv == 0) {
		buf[0] = 0;
		return (buf);
	}
	dst = buf;
	for (p = argv; (src = *p++) != 0; ) {
		if (*src == 0)
			continue;
		strvis(dst, src, VIS_NL | VIS_CSTYLE);
		while (*dst)
			dst++;
		*dst++ = ' ';
	}
	/* Chop off trailing space */
	if (dst != buf)
		dst--;
	*dst = '\0';
	return (buf);
}

static char *
cmdpart(char *arg0)
{
	char *cp;

	return ((cp = strrchr(arg0, '/')) != NULL ? cp + 1 : arg0);
}

static char *
fmt_argv(char **argv, char *cmd, int maxlen)
{
	int len;
	char *ap, *cp;

	if (argv == 0 || argv[0] == 0) {
		if (cmd == NULL)
			return ("");
		ap = NULL;
		len = maxlen + 3;
	} else {
		ap = shquote(argv);
		len = strlen(ap) + maxlen + 4;
	}
	if ((cp = malloc(len)) == NULL)
		return (NULL);
	if (ap == NULL)
		sprintf(cp, " (%.*s)", maxlen, cmd);
	else if (strncmp(cmdpart(argv[0]), cmd, maxlen) != 0)
		sprintf(cp, "%s (%.*s)", ap, maxlen, cmd);
	else
		(void) strcpy(cp, ap);
	return (cp);
}

typedef char **(*fn_t) __P((kvm_t *, const struct kinfo_proc *, int));

static char *
fmt(kvm_t *kd, fn_t fn, struct kinfo_proc *ki_p, char *comm, int maxlen)
{
	char *s;

	if ((s =
	    fmt_argv((*fn)(kd, ki_p, 80), comm, maxlen)) == NULL)
		err(1, NULL);
	return (s);
}
#endif

char *
process_scan_freebsd::get_cmd()
{
#if 0
#define CMD_MAX 32
	char cmd[CMD_MAX];

	return fmt(kvm_, kvm_getargv, cur_kp_, cur_kp_->kp_proc.p_comm,
		    MAXCOMLEN);
#else
	return cur_kp_->kp_proc.p_comm;
#endif
}

bool
process_scan_freebsd::next()
{
	cur_kp_++;
	return cur_kp_ < &kp_[nentries_];
}

process_model *
process_scan_freebsd::birth()
{
	process_model *pm = new process_model(int(cur_kp_->kp_proc.p_pid));
	// fill in init-time only bits:
	pm->set_uid(int(cur_kp_->kp_eproc.e_pcred.p_ruid));
	pm->set_nice(cur_kp_->kp_proc.p_nice);
	pm->set_start_time(0);  // xxx: not in bsd?
	life(pm);
	return pm;
}

void
process_scan_freebsd::life(process_model *pm)
{
	// OS-specific bits:
	pm->set_utime(ticks_to_msec(cur_kp_->kp_proc.p_uticks));
	pm->set_stime(ticks_to_msec(cur_kp_->kp_proc.p_sticks));
	// xxx: skip p_iticks

	int size, resident;

#if __FreeBSD__ < 3
	size = UPAGES + cur_kp_->kp_eproc.e_vm.vm_tsize
		+ cur_kp_->kp_eproc.e_vm.vm_dsize
		+ cur_kp_->kp_eproc.e_vm.vm_ssize;
	pm->set_virtual_size(pages_to_kb(size));
#else
	// under FreeBSD 3.0 and higher, ps uses this value, which is
	// kept in bytes
	size = UPAGES + cur_kp_->kp_eproc.e_vm.vm_map.size/1024;
	pm->set_virtual_size(size);
#endif
	resident = UPAGES + cur_kp_->kp_eproc.e_vm.vm_rssize;

	pm->set_resident(pages_to_kb(resident));

	// xxx: need some kind of hair to get this
	pm->set_cmd(get_cmd());
}


#endif /* USE_PROCESS_SCAN_FREEBSD */
