/*

------------------------------------------------------------------------------

A license is hereby granted to reproduce this software source code and
to create executable versions from this source code for personal,
non-commercial use.  The copyright notice included with the software
must be maintained in all copies produced.

THIS PROGRAM IS PROVIDED "AS IS". THE AUTHOR PROVIDES NO WARRANTIES
WHATSOEVER, EXPRESSED OR IMPLIED, INCLUDING WARRANTIES OF
MERCHANTABILITY, TITLE, OR FITNESS FOR ANY PARTICULAR PURPOSE.  THE
AUTHOR DOES NOT WARRANT THAT USE OF THIS PROGRAM DOES NOT INFRINGE THE
INTELLECTUAL PROPERTY RIGHTS OF ANY THIRD PARTY IN ANY COUNTRY.

Copyright (c) 1995, 1996, John Conover, All Rights Reserved.

Comments and/or bug reports should be addressed to:

    john@johncon.com (John Conover)

------------------------------------------------------------------------------

searchpath.c, recursive descent directory handler

int searchpath (char *name, ELEMENT *postfix_stack, BMHPATTERN *pattern_stack);

    open the path, name, and if it is a directory, call this routine,
    recursively, to descend through a directory hierarchy-if it is a
    file, search the file for relevance

    since this routine is called recursively, a shutdown procedure is
    implemented by placing all of the directory information in a
    DIRECTORY structure-this structure is allocated as an auto, and
    pushed on a static stack, directory_stack; when the routine exits
    normally, the DIRECTORY structure is poped off of the directory
    stack-if, for example under interrupt conditions, the program is
    aborted, the module relclose() (which was installed as an
    interrupt handler,) is called, which in turn calls a routine,
    int_searchpath(), to pop the remaining directories off of the
    directory stack, closing each one sequentially

    note that there is a test for the directory path name length; this
    is a safety provision to prevent continuously searching a circular
    linked directory, created with a shell construct like: mkdir
    my_dir;cd my_dir;ln -s my_subdir ../my_dir

    note that directory or files with names beginning with a '.' are
    ignored

The algorithm is as follows:

    auto allocate and set a directory structure on the machine stack

    attempt to open the directory

        if the directory opens

            while more paths exist under this directory

                concatenate the path name

                if the path name is too long, error

                call this routine, recursively, to open the pat

            close the directory

        else if the directory does not open

            assume it is a file, and attempt to process the file

                if the file has relevance

                    allocate a relevance structure on the relevance stack

Usage is a call with the path name, the postfix stack, and the pattern
stack-the postfix stack was derived in postfix.c, and the pattern stack
in bmhsearch.c, for example:

    if ((postfix_stack = postfix (infix_string, tokens)) != (ELEMENT *) 0)
    {

        if ((pattern_stack = bmhcompile_postfix (postfix_stack)) != (BMHPATTERN *) 0)
        {

            if (searchpath (path_name, postfix_stack, pattern_stack) != 0)
            {
                print_relevance ();
            }

        }

    }

For a detailed description of the applications of the directory
related POSIX functions used, see "POSIX Programmer's Guide," Donald
A. Lewine, Editor: Dale Dougherty, O'Reilly & Associates, Inc., 103
Morris Street, Suite A, Sebastopol, CA 95472, 1991, ISBN
0-937175-73-0, Chapter 4, pp 63.

The argument name is the name of the top level path, the argument
postfix_stack is the postfix notation stack derived from postfix.c,
and the argument pattern_stack is the search stack, compiled in
bmhsearch.c.

The last construct, before memory is deallocated, in this function, is
a switch construct to handle certain types of file system exceptions,
for example, failure to open a file or directory-in which case it may
be desirable to continue execution instead of handling the error, and
exiting-the function of this construct is to provide flexibility in
file system exception handling. This behavior is controlled at compile
time by defining, or not defining, FSE_CONTINUE. If FSE_CONTINUE is
defined, then the program will continue during these, and only these,
specific file system exceptions. If FSE_CONTINUE is not defined, the
program will shutdown, and exit with any file system exception.

Zero is returned if successful, non-zero if not.

The structure RELEVANCE is defined in searchpath.h.

To test this module, compile the module source with -DTEST_SEARCHPATH

$Revision: 1.1 $
$Date: 1995/12/23 23:17:08 $
$Id: searchpath.c,v 1.1 1995/12/23 23:17:08 john Exp $
$Log: searchpath.c,v $
Revision 1.1  1995/12/23 23:17:08  john
Added switch statement to end of function searchpath () to permit continuation of execution during file system exceptions, specifically failure to open a file, or failure to open a directory.
Changes to files: Makefile, searchpath.c, searchfile.c, message.c, version.c-specifically to control program behavior under certain file system exceptions; specifics for the GNU gcc compiler, and ANSI C cosmetic formalities.

 * Revision 1.0  1995/04/22  05:13:18  john
 * Initial revision
 *

*/

#include "rel.h"

#ifndef LINT /* include rcsid only if not running lint */

static char rcsid[] = "$Id: searchpath.c,v 1.1 1995/12/23 23:17:08 john Exp john $"; /* module version */
static char rcsid_h[] = SEARCHPATH_H_ID; /* module include version */

#endif

#define DIRECTORY_CHARACTER '/' /* directory delimiter character */
#define DIRECTORY_STRING "/" /* directory delimiter string */

RELEVANCE *relevance_stack = (RELEVANCE *) 0; /* reference to relevance stack */

typedef struct directory_struct /* directory structure for a recursion */
{
    DIR *dirp; /* reference to the directory for a recursion */
    char *name; /* reference to name of the directory for a recursion */
    struct directory_struct *next; /* reference to next structure in the list of directory structures */
} DIRECTORY;

static DIRECTORY *directory_stack = (DIRECTORY *) 0; /* reference to the directory stack */

#ifdef __STDC__

int searchpath (char *name, ELEMENT *postfix_stack, BMHPATTERN *pattern_stack)

#else

int searchpath (name, postfix_stack, pattern_stack)
    char *name;
    ELEMENT *postfix_stack;
    BMHPATTERN *pattern_stack;

#endif

{
    char path[MAXPATHLEN + 1]; /* path name register, with space for a null character EOS terminator */

    int retval = URODR_ERR, /* return value, assume error opening directory */
        value;

    struct dirent *dire; /* reference to directory path */

    RELEVANCE *element; /* reference to element in relevance stack */

    DIRECTORY direct, /* directory structure for this recursion */
              *directory; /* reference to direct for PUSH() and POP() operations */

    directory = &direct; /* reference the directory structure for this recursion */
    directory->dirp = (DIR *) 0; /* null the reference to the directory for this recursion */
    directory->name = name; /* reference the name of the directory for this recursion */
    PUSH (directory_stack, directory); /* push the directory structure on the directory stack */

    if ((directory->dirp = opendir (name)) != 0) /* open the top level directory */
    {
        retval = NO_ERROR; /* assume no errors */

        while ((dire = readdir (directory->dirp)) != 0) /* for each file or directory in this directory: */
        {

            if (dire->d_name[0] != '.') /* if the filename begins with a '.' ignore it */
            {
                (void) strcpy (path, name); /* start concatinating the top level directory name */

                if (path[strlen (path) - 1] != DIRECTORY_CHARACTER) /* last character in top directory name a path delimiter? */
                {
                    (void) strcat (path, DIRECTORY_STRING); /* no, make it so */
                }

                if (strlen (path) + strlen (dire->d_name) > MAXPATHLEN) /* path name is too long */
                {
                    retval = URODR_ERR, /* yes, assume error opening directory */
                    message (retval, name); /* print the error */
                    break; /* stop processing the top level directory */
                }

                (void) strcat (path, dire->d_name); /* concatinate the file or directory name with the top level name */

                if ((retval = searchpath (path, postfix_stack, pattern_stack)) != NO_ERROR) /* and decend into it */
                {
                    break; /* some sort of error occured when decending into the directory, return */
                }

            }

        }

        if (closedir (directory->dirp) != 0) /* close the directory */
        {
            retval = URCDR_ERR; /* couldn't close the directory, assume error closing directory */
            message (retval, name); /* print the error */
        }

    }

    else
    {

        if (errno == ENOTDIR) /* couldn't open the top level directory, is it not a directory? */
        {

            if ((retval = searchfile (name, pattern_stack)) == NO_ERROR) /* yes, it is not a directory, assume it is a file */
            {

                if ((value = postfix_eval (postfix_stack)) > 0) /* evaluate the relevance of the file, is it relevant? */
                {
                    retval = URMEM_ERR; /* yes, assume error allocating memory */

                    if ((element = (RELEVANCE *) memalloc (sizeof (RELEVANCE))) != (RELEVANCE *) 0) /* allocate a structure */
                    {

                        if ((element->name = (char *) memalloc (strlen (name) + 1)) != (char *) 0) /* allocate the file name */
                        {
                            (void) strcpy (element->name, name); /* save the file name */
                            element->count = value; /* save the relevance count */
                            PUSH (relevance_stack, element); /* push the relevance element on the relevance stack */
                            retval = NO_ERROR; /* assume no errors */
                        }

                    }

                    if (retval != NO_ERROR) /* pending error? */
                    {
                        message (retval, name); /* yes, print the error */
                    }

                }

            }

        }

        else
        {
            message (retval, name); /* couldn't open the top level directory, it is a directory, print the error */
        }

    }

#ifdef FSE_CONTINUE

    /*

    any/all file system errors have been printed for this
    directory-the following switch construct is for handling certain
    of the errors, for example, error opening the directory, or, error
    opening a file in the directory, in which case, under certain
    circumstances, it may be desirable to continue execution; for
    continuation under file system exceptions, the variable, retval,
    should be set to NO_ERROR for these specific exceptions

    exceptions that can be handled are: URFIL_ERR, (error opening
    file,) URLCK_ERR, (error locking file,) URSTA_ERR, (error stating
    file,) URMEM_ERR, (error allocating memory,) URRED_ERR, (error
    reading from file,) URUCK_ERR, (error unlocking file,) URCLS_ERR,
    (error closing file,) all from searchfile.c, and, URODR_ERR,
    (error opening directory,) URCDR_ERR, (error closing directory,)
    URMEM_ERR, (error allocating memory,) all from this file,
    searchpath.c-NO_ERROR is returned if there were no errors
    encountered in searchfile.c or searchpath.c

    */

    switch (retval) /* error return, if any? (any error messages have already been printed) */
    {

        case URODR_ERR: /* error opening directory? */

            retval = NO_ERROR; /* yes, assume no errors */
            break;

        case URFIL_ERR: /* error opening file? */

            retval = NO_ERROR; /* yes, assume no errors */
            break;

        default:

            break; /* any other errors, or no errors, continue */

    }

#endif

    (void) POP (directory_stack); /* pop the directory structure off the directory stack */
    return (retval); /* return any errors */
}

/*

void int_searchpath (void);

since searchpath()  is called recursively, a shutdown procedure is
necessary to close any open directories-this routine should be installed
as part of the interrupt handling process

The algorithm is as follows:

    while there are still DIRECTORY structures on the directory stack,
    pop the structures off of the stack, closing each directory

There are no arguments, and no return value from this function

*/

#ifdef __STDC__

void int_searchpath (void)

#else

void int_searchpath ()

#endif

{

    while (directory_stack != (DIRECTORY *) 0) /* for each directory structure on the directory stack */
    {

        if (directory_stack->dirp != (DIR *) 0) /* directory open? */
        {

            if (closedir (directory_stack->dirp) != 0) /* close the directory */
            {
                message (URCDR_ERR, directory_stack->name); /* couldn't close the directory, print the error */
            }

        }

        (void) POP (directory_stack); /* pop the directory structure off the directory stack */
    }

}

#ifdef TEST_SEARCHPATH

/*

simple exerciser for testing searchpath (); open and input the file
specified on the command line, and read it-the search argument must be
in uppercase; ignore the:

declared global, could be static
    relevance_stack     searchpath.c(xxx)
    searchpath          searchpath.c(yyy)

from lint

*/

#ifdef __STDC__

int main (int argc, char *argv[])

#else

int main (argc, argv)
    int argc;
    char *argv[];

#endif

{
    unsigned char tokens[2 * BUFSIZ]; /* buffer containing tokens from infix notation string */

    int retval = URARG_ERR, /* return value, assume argument error */
        file_ctr; /* file counter */

    ELEMENT *postfix_stack; /* reference to postfix stack */

    BMHPATTERN *pattern_stack; /* reference to pattern stack */

    RELEVANCE *file; /* reference to element in relevance stack */

    if (argc > 2) /* enough arguments? */
    {
        retval = NO_ERROR; /* assume no errors */

        if (make_uppercase () != (unsigned char *) 0) /* setup the uppercase array */
        {

            if ((postfix_stack = postfix (argv[1], tokens)) != (ELEMENT *) 0) /* translate first argument to postfix notation */
            {

                if ((pattern_stack = bmhcompile_postfix (postfix_stack)) != (BMHPATTERN *) 0)
                {

                    for (file_ctr = 2; file_ctr < argc; file_ctr++) /* for each file specified */
                    {

                        if ((retval = searchpath (argv[file_ctr], postfix_stack, pattern_stack)) != NO_ERROR) /* search the file */
                        {
                            break; /* if any errors quit */
                        }

                    }

                    if (relevance_stack != (RELEVANCE *) 0) /* yes, anything on the relevance stack? */
                    {
                        file = relevance_stack; /* reference the relevance stack */

                        while (file != (RELEVANCE *) 0) /* for each file on the relevance stack */
                        {
                            (void) printf ("%s = %d\n", file->name, file->count); /* print the file name */
                            file = file->next; /* next file on the relevance stack */
                        }

                    }

                }

            }

        }

    }

    else
    {
        message (retval, argv[0]); /* not enough arguments, print the error */
        retval = NO_ERROR; /* assume no error */
    }

    exit (retval); /* return any errors */

#ifdef LINT /* include only if running lint */

    return (0); /* for LINT formality */

#endif

}

#endif
