/*

------------------------------------------------------------------------------

A license is hereby granted to reproduce this software source code and
to create executable versions from this source code for personal,
non-commercial use.  The copyright notice included with the software
must be maintained in all copies produced.

THIS PROGRAM IS PROVIDED "AS IS". THE AUTHOR PROVIDES NO WARRANTIES
WHATSOEVER, EXPRESSED OR IMPLIED, INCLUDING WARRANTIES OF
MERCHANTABILITY, TITLE, OR FITNESS FOR ANY PARTICULAR PURPOSE.	THE
AUTHOR DOES NOT WARRANT THAT USE OF THIS PROGRAM DOES NOT INFRINGE THE
INTELLECTUAL PROPERTY RIGHTS OF ANY THIRD PARTY IN ANY COUNTRY.

Copyright (c) 1995, John Conover, All Rights Reserved.

Comments and/or bug reports should be addressed to:

    john@johncon.com (John Conover)

------------------------------------------------------------------------------

message.c, print an error message

void message (int fil, char *ancillary);

    prints message indicated by the error code, fil, called to lookup
    the message to be printed, print it to stderr, and print any
    ancillary message, perhaps a file name, referenced by the
    character reference, ancillary

    the array, message_array[], is an array of type MESSAGE_STRUCT
    elements, each element containing a mnemonic error type, the error
    number of the message (in ascending order,) both of which must be
    defined and match an entry in message.h-in addition to a character
    string that will be printed to stderr

    the first 50 error messages, numbered 0 to 49, are system related
    messages, and may be omitted if desired, or customized to a
    particular system by looking for the errors in signal.h, etc.

The algorithm is as follows:

    if fil is zero, just return

    if fil is greater than the the size of the message array, assume
    an unknown error

    print the message array element implicitly addressed by fil

    if the ancillary message reference is not null, print the message
    referenced by the ancillary message reference

Usage is a call with an error code, and an optional ancillary message,
for example:

    char *filename;

    message (URFIL_ERR,filename);

would print the message for a file that failed to open, or

    message (URMEM_ERR, (char *) 0);

would print the message for a memory allocation error

The argument, fill, represents the error code that corresponds to the
message to be printed, and ancillary references an additional string
that will be appended to the error code

There is no return value from this function

To test this module, compile the module source with -DTEST_MESSAGE

$Revision: 1.1 $
$Date: 1995/12/23 23:14:31 $
$Id: message.c,v 1.1 1995/12/23 23:14:31 john Exp $
$Log: message.c,v $
Revision 1.1  1995/12/23 23:14:31  john
Changed mnemonic and message elements of MESSAGE_STRUCT structure to const char to comply with formal ANSI C requirements for statically declared character strings.
Changes to files: Makefile, searchpath.c, searchfile.c, message.c, version.c-specifically to control program behavior under certain file system exceptions; specifics for the GNU gcc compiler, and ANSI C cosmetic formalities.

 * Revision 1.0	 1995/04/22  05:13:18  john
 * Initial revision
 *

*/

#include "rel.h"

#ifndef LINT /* include rcsid only if not running lint */

static char rcsid[] = "$Id: message.c,v 1.1 1995/12/23 23:14:31 john Exp john $"; /* module version */
static char rcsid_h[] = MESSAGE_H_ID; /* module include version */

#endif

#ifdef __STDC__

typedef struct message_struct /* message structure, one for each error message */
{
    const char *mnemonic; /* error type */
    int number; /* error number */
    int arguments; /* ancillary arguments to be printed */
    const char *message; /* message to be printed on exit with error */
} MESSAGE_STRUCT;

#else

typedef struct message_struct /* message structure, one for each error message */
{
    char *mnemonic; /* error type */
    int number; /* error number */
    int arguments; /* ancillary arguments to be printed */
    char *message; /* message to be printed on exit with error */
} MESSAGE_STRUCT;

#endif

int max_interrupts = 49; /* maximum value of system interrupts */

static char unknown[] = "unknown error"; /* unknown error message */

static MESSAGE_STRUCT message_array[] = /* system interrupts, 0 through 49, system dependent messages, and found in signal.h */
{
    {"NO_ERROR", 0, 0, "No errors"},
    {"SIGHUP", 1, 0, "hangup"},
    {"SIGINT", 2, 0, "interrupt"},
    {"SIGQUIT", 3, 0, "quit"},
    {"SIGILL", 4, 0, "illegal instruction"},
    {"SIGTRAP", 5, 0, "trace trap"},
    {"SIGIOT", 6, 0, "IOT instruction"},
    {"SIGEMT", 7, 0, "EMT instruction"},
    {"SIGFPE", 8, 0, "floating point exception"},
    {"SIGKILL", 9, 0, "kill"},
    {"SIGBUS", 10, 0, "bus error"},
    {"SIGSEGV", 11, 0, "segmentation violation"},
    {"SIGSYS", 12, 0, "bad argument to system call"},
    {"SIGPIPE", 13, 0, "write on a pipe with no one to read it"},
    {"SIGALRM", 14, 0, "alarm clock"},
    {"SIGTERM", 15, 0, "software termination signal from kill"},
    {"SIGUSR1", 16, 0, "user defined signal 1"},
    {"SIGUSR2", 17, 0, "user defined signal 2"},
    {"SIGCLD", 18, 0, "child status change"},
    {"SIGPWR", 19, 0, "power-fail restart"},
    {"SIGWINCH", 20, 0, "window size change"},
    {"SIGURG", 21, 0, "urgent socket condition"},
    {"SIGPOLL", 22, 0, "pollable event occured"},
    {"SIGSTOP", 23, 0, "stop"},
    {"SIGTSTP", 24, 0, "user stop requested from tty"},
    {"SIGCONT", 25, 0, "stopped process has been continued"},
    {"SIGTTIN", 26, 0, "background tty read attempted"},
    {"SIGTTOU", 27, 0, "background tty write attempted"},
    {"SIGVTALRM", 28, 0, "virtual timer expired"},
    {"SIGPROF", 29, 0, "profiling timer expired"},
    {"SIGXCPU", 30, 0, "exceeded cpu limit"},
    {"SIGXFSZ", 31, 0, "exceeded file size limit"},
    {"UNKNOWN", 32, 0, unknown},
    {"UNKNOWN", 33, 0, unknown},
    {"UNKNOWN", 34, 0, unknown},
    {"UNKNOWN", 35, 0, unknown},
    {"UNKNOWN", 36, 0, unknown},
    {"UNKNOWN", 37, 0, unknown},
    {"UNKNOWN", 38, 0, unknown},
    {"UNKNOWN", 39, 0, unknown},
    {"UNKNOWN", 40, 0, unknown},
    {"UNKNOWN", 41, 0, unknown},
    {"UNKNOWN", 42, 0, unknown},
    {"UNKNOWN", 43, 0, unknown},
    {"UNKNOWN", 44, 0, unknown},
    {"UNKNOWN", 45, 0, unknown},
    {"UNKNOWN", 46, 0, unknown},
    {"UNKNOWN", 47, 0, unknown},
    {"UNKNOWN", 48, 0, unknown},
    {"UNKNOWN", 49, 0, unknown},

    /* program specific errors */

    {"URISG_ERR", 50, 0, "error installing signal handler"},
    {"URIGN_ERR", 51, 0, "error disabling interrupts"},
    {"URRSG_ERR", 52, 0, "error restoring default interrupts"},
    {"URMEM_ERR", 53, 0, "error allocating memory"},
    {"URPAR_ERR", 54, 0, "error in grouping operators"},
    {"URSYN_ERR", 55, 0, "error in syntax"},
    {"URFIL_ERR", 56, 1, "error opening file, %s"}, /* requires ancillary filename argument */
    {"URLCK_ERR", 57, 1, "error locking file, %s"}, /* requires ancillary filename argument */
    {"URRED_ERR", 58, 1, "error reading from file, %s"}, /* requires ancillary filename argument */
    {"URUCK_ERR", 59, 1, "error unlocking file, %s"}, /* requires ancillary filename argument */
    {"URCLS_ERR", 60, 1, "error closing file, %s"}, /* requires ancillary filename argument */
    {"URSTA_ERR", 61, 1, "error stating file, %s"}, /* requires ancillary filename argument */
    {"URODR_ERR", 62, 1, "error opening directory, %s"}, /* requires ancillary filename argument */
    {"URCDR_ERR", 63, 1, "error closing directory, %s"}, /*requires ancillary filename argument */
    {"URARG_ERR", 64, 1, "usage: %s [-v] pattern filename(s)"} /* requires ancillary program name argument */
};

static size_t array_size = sizeof (message_array) / sizeof (MESSAGE_STRUCT); /* size of message structure array */

#ifdef __STDC__

void message (int fil, char *ancillary)

#else

void message (fil, ancillary)
    int fil;
    char *ancillary;

#endif

{
    MESSAGE_STRUCT *message_ref; /* message array element reference */

    if (fil != 0) /* requested message signify no error? */
    {

	if (fil < 0) /* fil negative? */
	{
	    fil = -fil; /* yes, use the absolute value */
	}

	if (fil >= (int) array_size) /* requested message in bounds of array? */
	{
	    fil = 49; /* assume unknown error */
	}

	message_ref = &message_array[fil]; /* yes, reference the message */

	if (message_ref->arguments == 0) /* any ancillary reference to additional error message? */
	{
	    (void) fprintf (stderr, message_ref->message); /* print the message */
	}

	else
	{

	    if (ancillary == (char *) 0) /* protect from programming errors */
	    {
		ancillary = unknown;
	    }

	    (void) fprintf (stderr, message_ref->message, ancillary); /* print the message with the ancillary message */
	}

	(void) fprintf (stderr, "\n"); /* terminate the message */
    }

}

#ifdef TEST_MESSAGE

/*

simple exerciser for testing message (); get a number from stdin, and
print the corresponding error message-ignore the:

declared global, could be static
    message		message.c(xxx)

from lint

*/

#include <stdio.h>
#include <stdlib.h>
#include "message.h"

#ifdef __STDC__

int main (int argc, char *argv[])

#else

int main (argc, argv)
    int argc;
    char *argv[];

#endif

{
    char buffer[BUFSIZ], /* input buffer */
	 *ancillary = (char *) 0; /* ancillary message reference */

    if (argc > 1) /* any argument? */
    {
	ancillary = argv[1]; /* assume argument is ancillary message */
    }

    while (gets (buffer) != 0) /* input the number */
    {
	message (atoi (buffer), ancillary); /* convert the input buffer to a number, and print the corresponding error message */
    }

    exit (0); /* always return 0 */

#ifdef LINT /* include only if running lint */

    return (0); /* for LINT formality */

#endif

}

#endif
