/*
 * $Id: zle.h,v 3.1.2.8 1997/05/31 06:04:08 hzoli Exp $
 *
 * zle.h - header file for line editor
 *
 * This file is part of zsh, the Z shell.
 *
 * Copyright (c) 1992-1997 Paul Falstad
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and to distribute modified versions of this software for any
 * purpose, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 *
 * In no event shall Paul Falstad or the Zsh Development Group be liable
 * to any party for direct, indirect, special, incidental, or consequential
 * damages arising out of the use of this software and its documentation,
 * even if Paul Falstad and the Zsh Development Group have been advised of
 * the possibility of such damage.
 *
 * Paul Falstad and the Zsh Development Group specifically disclaim any
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose.  The software
 * provided hereunder is on an "as is" basis, and Paul Falstad and the
 * Zsh Development Group have no obligation to provide maintenance,
 * support, updates, enhancements, or modifications.
 *
 */

#ifndef _ZLE_H
#define _ZLE_H

#include "comp.h"

#ifdef ZLEGLOBALS
#define ZLEXTERN
#else
#define ZLEXTERN extern
#endif

typedef struct widget *Widget;
typedef struct thingy *Thingy;

/* widgets (ZLE functions) */

typedef void (*ZleIntFunc) _((void));

struct widget {
    int flags;		/* flags (see below) */
    Thingy first;	/* `first' thingy that names this widget */
    union {
	ZleIntFunc fn;	/* pointer to internally implemented widget */
	char *fnnam;	/* name of the shell function for user-defined widget */
    } u;
};

#define WIDGET_INT	(1<<0)    /* widget is internally implemented */
#define ZLE_MENUCMP	(1<<1)    /* DON'T invalidate completion list */
#define ZLE_YANK	(1<<3)
#define ZLE_LINEMOVE	(1<<4)    /* command is a line-oriented movement */
#define ZLE_LASTCOL     (1<<5)    /* command maintains lastcol correctly */
#define ZLE_KILL	(1<<6)
#define ZLE_KEEPSUFFIX	(1<<9)    /* DON'T remove added suffix */

/* thingies */

struct thingy {
    HashNode next;	/* next node in the hash chain */
    char *nam;		/* name of the thingy */
    int flags;		/* TH_* flags (see below) */
    int rc;		/* reference count */
    Widget widget;	/* widget named by this thingy */
    Thingy samew;	/* `next' thingy (circularly) naming the same widget */
};

/* DISABLED is (1<<0) */
#define TH_IMMORTAL	(1<<1)    /* can't refer to a different widget */

/* Hashtable of thingies.  Enabled nodes are those that refer to widgets. */
ZLEXTERN HashTable thingytab;

/* Are references to earlier history lines permitted?  == 0 if       *
 * editing or reading a standalone line, such as in vared or select. */
ZLEXTERN int histallowed;

/* size of line buffer */
ZLEXTERN int linesz;

/* location of mark */
ZLEXTERN int mark;

/* last character pressed */
ZLEXTERN int c;

/* the binding for this key */
ZLEXTERN Thingy bindk;

/* insert mode/overwrite mode flag */
ZLEXTERN int insmode;

#ifdef HAVE_SELECT
/* cost of last update */
ZLEXTERN int cost;

/* Terminal baud rate (from the BAUD parameter) */
ZLEXTERN int baud;
#endif

/* number of lines displayed */
ZLEXTERN int nlnct;

/* Most lines of the buffer we've shown at once with the current list *
 * showing.  == 0 if there is no list.  == -1 if a new list has just  *
 * been put on the screen.  == -2 if refresh() needs to put up a new  *
 * list.                                                              */
ZLEXTERN int showinglist;

/* Non-zero if ALWAYS_LAST_PROMPT has been used, meaning that the *
 * screen below the buffer display should not be cleared by       *
 * refresh(), but should be by trashzle().                        */
ZLEXTERN int clearflag;

/* flags associated with last command */
ZLEXTERN int lastcmd;

/* Column position of vi ideal cursor.  -1 if it is unknown -- most *
 * movements and changes do this.                                   */
ZLEXTERN int lastcol;

/* != 0 if we're getting a vi range */
ZLEXTERN int virangeflag;

/* kludge to get cw and dw to work right */
ZLEXTERN int wordflag;

/* != 0 if we're killing lines into a buffer, vi-style */
ZLEXTERN int vilinerange;

/* last named command done */
ZLEXTERN Thingy lastnamed;

/* != 0 if we're done editing */
ZLEXTERN int done;

/* current history line number */
ZLEXTERN int histline;

/* the last line in the history (the current one), metafied */
ZLEXTERN char *curhistline;

/* the status line, and its length */
ZLEXTERN char *statusline;
ZLEXTERN int statusll;

/* !=0 if a complete added a suffix at the end of a completion */
ZLEXTERN int addedsuffix;

/* 1 if we expect special keys after completing a parameter name.  2 if *
 * the parameter name is in ${...}.                                     */
ZLEXTERN int complexpect;

/* The current history line and cursor position for the top line *
 * on the buffer stack.                                          */
ZLEXTERN int stackhist, stackcs;

/* != 0 if we are in the middle of a menu completion */
ZLEXTERN int menucmp;

/* != 0 if we are making undo records */
ZLEXTERN int undoing;

/* last vi change buffer, for vi change repetition */
ZLEXTERN int vichgbufsz, vichgbufptr, vichgflag;
ZLEXTERN char *vichgbuf;

/* point where vi insert mode was last entered */
ZLEXTERN int viinsbegin;

/* command modifier prefixes */

struct modifier {
    int flags;		/* MOD_* flags (see below) */
    int mult;		/* repeat count */
    int tmult;		/* repeat count actually being edited */
    int vibuf;		/* vi cut buffer */
};

#define MOD_MULT  (1<<0)   /* a repeat count has been selected */
#define MOD_TMULT (1<<1)   /* a repeat count is being entered */
#define MOD_VIBUF (1<<2)   /* a vi cut buffer has been selected */
#define MOD_VIAPP (1<<3)   /* appending to the vi cut buffer */

/* current modifier status */

ZLEXTERN struct modifier zmod;
#define zmult (zmod.mult)

/* Current command prefix status.  This is normally 0.  Prefixes set *
 * this to 1.  Each time round the main loop, this is checked: if it *
 * is 0, the modifier status is reset; if it is 1, the modifier      *
 * status is left unchanged, and this flag is reset to 0.  The       *
 * effect is that several prefix commands can be executed, and have  *
 * cumulative effect, but any other command execution will clear the *
 * modifiers.                                                        */

ZLEXTERN int prefixflag;

/* != 0 if there is a pending beep (usually indicating an error) */

ZLEXTERN int feepflag;

/* undo system */

struct change {
    struct change *prev, *next;	/* adjacent changes */
    int flags;			/* see below */
    int hist;			/* history line being changed */
    int off;			/* offset of the text changes */
    char *del;			/* characters to delete (metafied) */
    char *ins;			/* characters to insert (metafied) */
};

#define CH_NEXT (1<<0)   /* next structure is also part of this change */
#define CH_PREV (1<<1)   /* previous structure is also part of this change */

/* head of the undo list, and the current position */

ZLEXTERN struct change *changes, *curchange;

/* list of pending changes, not yet in the undo system */

ZLEXTERN struct change *nextchanges, *endnextchanges;

/* the line before last mod (for undo purposes) */

ZLEXTERN char *lastline;
ZLEXTERN int lastlinesz, lastll;

/* the last vi search */

ZLEXTERN char *visrchstr;
ZLEXTERN int visrchsense;

/* table of known thingies */

extern struct thingy thingies[];
#define Th(X) (&thingies[X])

#include "zle_things.h"

/* opaque keymap type */

typedef struct keymap *Keymap;

/* tables of default key bindings */

extern int emacsbind[], metabind[], viinsbind[], vicmdbind[];

/* currently selected keymap, and its name */

ZLEXTERN Keymap curkeymap;
ZLEXTERN char *curkeymapname;

typedef void (*KeyScanFunc) _((char *, Thingy, char *, void *));

#define invicmdmode() (!strcmp(curkeymapname, "vicmd"))

/* Cut/kill buffer type.  The buffer itself is purely binary data, *
 * not NUL-terminated.  len is a length count.  flags uses the     *
 * CUTBUFFER_* constants defined below.                            */

struct cutbuffer {
    char *buf;
    size_t len;
    char flags;
};

typedef struct cutbuffer *Cutbuffer;

#define CUTBUFFER_LINE 1   /* for vi: buffer contains whole lines of data */

/* Primary cut buffer */

ZLEXTERN struct cutbuffer cutbuf;

/* Emacs-style kill buffer ring */

#define KRINGCT 8
ZLEXTERN struct cutbuffer kring[KRINGCT];
ZLEXTERN int kringnum;

/* Vi named cut buffers.  0-25 are the named buffers "a to "z, and *
 * 26-34 are the numbered buffer stack "1 to "9.                   */

ZLEXTERN struct cutbuffer vibuf[35];

/* prototypes */

#define GLOBAL_PROTOTYPES

#include "zle_hist.pro"
#include "zle_keymap.pro"
#include "zle_main.pro"
#include "zle_misc.pro"
#include "zle_move.pro"
#include "zle_params.pro"
#include "zle_refresh.pro"
#include "zle_thingy.pro"
#include "zle_tricky.pro"
#include "zle_utils.pro"
#include "zle_vi.pro"
#include "zle_word.pro"

#undef GLOBAL_PROTOTYPES

#endif /*!_ZLE_H*/
