/************************************************************************/
/* sccw - Soundcard CW for Linux   Steven J. Merrifield			*/
/*			 	   VK3ESM  sjm@ee.latrobe.edu.au	*/
/*				   http://livewire.ee.latrobe.edu.au	*/
/*									*/
/* Feb 92 - First MS-DOS release using PC speaker (Turbo Pascal)	*/
/* Oct 92 - Added punctuation support, varying character delay		*/
/* Aug 95 - Added mouse support for sending practice			*/
/* Jul 96 - Added soundcard support using FM chip			*/
/* Aug 96 - Complete rewrite in ANSI-C, first Linux release	 	*/
/************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h> 
#include <asm/io.h>
#include <time.h>
#include "sccw.h"

extern int AdLib_found();
extern int Play_sound();

/* If no config file is found in the users' home directory, then these */
/* defaults will be used instead */

int Morse_speed=10;	/* Speed in WPM */
int Morse_gap=3;	/* Spacing between each letter */
int Num_grps=5;		/* Number of 5 character groups */
int Grp_type=1;		/* Type of character set for random groups */
int AdLib_tone = 700;	/* Frequency of tone */
int AdLib_vol = 32;	/* Soundcard volume */

char *Code[] = 
		{".-","-...","-.-.","-..",".","..-.","--.",
		"....","..",".---","-.-",".-..","--","-.","---",
		".--.","--.-",".-.","...","-","..-","...-",
		".--","-..-","-.--","--..",  /* A..Z */
		"-----",".----","..---","...--","....-",
		".....","-....","--...","---..","----.", /* 0..9 */
		"..--..","-..-.",""}; /*  q-mark, slant, space */

/************************************************************************/
/* Convert the string into sound					*/
/************************************************************************/
void Code_2_snd(int a)
{
	int i;
	double dit,dah,wait;

	dit = (double)(60.0/(Morse_speed*50.0)) * 1000000.0; 	/* delay in us (PARIS method) */
	dah = 3 * dit;
	wait = Morse_gap * dit;

	for (i=0;i<strlen(Code[a]);i++)
	{
		if (Code[a][i] == '.') 
		{
			/* printf("."); */
			Play_sound(AdLib_tone,dit,AdLib_vol);
		}
		if (Code[a][i] == '-')
		{
			/* printf("-"); */
			Play_sound(AdLib_tone,dah,AdLib_vol);
		}
		usleep(dit);	/* delay between each dit or dah */
	}
	usleep(wait);	/* delay between each each character */
}

void Press_return()
{
	int ch;
	fprintf(stdout,"Press <ENTER> to continue...");
	ch = fgetc(stdin);
}

/************************************************************************/
/* Display the character on the screen after sending it in cw 		*/
/************************************************************************/
void Show_code(int a)
{
	if (a < 26) fprintf(stdout,"%c",a+65);
	if ((a >= 26) && (a < 36)) fprintf(stdout,"%c",a+22);
	if (a == 36) fprintf(stdout,"?");
	if (a == 37) fprintf(stdout,"/");
	if (a == 38) fprintf(stdout," ");
}


/************************************************************************/
/* Send random groups							*/
/************************************************************************/
void Send_groups(int Num_groups, int Send_Num)
{
	int i,j,r=1;
	for (j=1;j<=Num_groups;j++)
	{
		for (i=1;i<=5;i++)
		{
			if (Send_Num == 1) r = rand() % 26;	/* A..Z only */
			if (Send_Num == 2) r = rand() % 36;	/* A..Z, 0..9 */
			if (Send_Num == 3) r = rand() % 38;	/* A..Z, 0..9, ? / */
			Code_2_snd(r);
			Show_code(r);
		}
		fprintf(stdout," ");	/* Space between each 5 char grp */
		if ((j % 10) == 0) fprintf(stdout,"\n");	/* 10 grps per line */
	}
	fprintf(stdout,"\n");
	Press_return();
}

/************************************************************************/
/* Read a file, and translate it into morse				*/
/************************************************************************/
int Send_file()
{
	char fname[255];
	FILE *f;
	int ch;
	fprintf(stdout,"Enter filename : ");
	fscanf(stdin,"%s",fname);
	if ((f = fopen(fname,"rt")) == NULL)
	{
		fprintf(stderr,"Error opening file.\n");
		getchar();	/* flush RETURN from buffer */
		Press_return();
		return(1);
	}
	while ((ch = getc(f)) != EOF)
	{
		if (isalpha(ch)) ch = toupper(ch); 
		if (((ch >= 'A') && (ch <= 'Z')) ||
			((ch >= '0' ) && (ch <= '9')) ||
			(ch == '?') || (ch == '/') || (ch == ' ') ||
			(ch == 0x0A))
		{
			if (ch == 0x0A)  printf("\n");
			else
			{
				if ((ch >= 'A') && (ch <= 'Z')) ch = ch - 65;
				if ((ch >= '0') && (ch <= '9')) ch = ch - 22;
				if (ch == '?') ch = 36;
				if (ch == '/') ch = 37;
				if (ch == ' ') ch = 38;
				Code_2_snd(ch);
				Show_code(ch);
			}
		}
	}
	fclose(f);
	getchar();	/* flush RETURN from buffer */
	Press_return();
	return(0);
}

/************************************************************************/
/* Setup the type of character set used in the random character 	*/
/* generation								*/		
/************************************************************************/
int Ask_grp_type()
{
	int choice;
	fprintf(stdout,"\nChoose the set of random characters...\n");
	fprintf(stdout,"  1. Letters A to Z only\n");
	fprintf(stdout,"  2. Letters A to Z, digits 0 to 9\n");
	fprintf(stdout,"  3. Letters A to Z, digits 0 to 9, question mark, slant\n");
	fprintf(stdout,"Enter your choice : ");
	fscanf(stdin,"%d",&choice);
	return(choice);
}

/************************************************************************/
/* Setup the number of groups sent in the random character generation 	*/
/************************************************************************/
int Ask_num_grps()
{
	int choice;
	fprintf(stdout,"Enter number of 5 character groups : ");
	fscanf(stdin,"%d",&choice);
	return(choice);
}

/************************************************************************/
/* Setup the frequency of the transmitted tone				*/
/************************************************************************/
int Ask_freq()
{
	int choice;
	fprintf(stdout,"Enter frequency (100-6000) : ");
	fscanf(stdin,"%d",&choice);
	return(choice);
}

/************************************************************************/
/* Set the volume of the soundcard. This also depends on the mixer	*/
/* setting, so you must alter two volumes - this one only controls the	*/
/* FM chip output, use something like "setmixer" to adjust the overall	*/
/* soundcard volume.							*/
/************************************************************************/
int Ask_vol()
{
	int choice;
	fprintf(stdout,"Enter volume (0-64) : ");
	fscanf(stdin,"%d",&choice);
	return(choice);
}

/************************************************************************/
/* Set the speed of the transmitted morse				*/
/************************************************************************/
int Ask_speed()
{
	int choice;
	fprintf(stdout,"Enter speed (3-50) : ");
	fscanf(stdin,"%d",&choice);
	return(choice);
}

/************************************************************************/
/* Set the delay between each character. 				*/
/* Normally, this is set to 3 dits, but sometimes it is helpful to	*/
/* increase this for learning purposes.					*/
/************************************************************************/
int Ask_delay()
{
	int choice;
	fprintf(stdout,"Enter delay between each character : ");
	fscanf(stdin,"%d",&choice);
	return(choice);
}

/************************************************************************/
/* This is the nerve center. Nearly everything can be adjusted here	*/ 
/************************************************************************/
int menu()
{
	int choice;
	/* This is a cheap and nasty way of clearing the screen without */
	/* using (n)curses or worrying about /etc/termcap. I know it's 	*/
	/* not the ideal way of doing things, but it works... 8-)	*/
	system("clear");	
	fprintf(stdout,"==========================================================\n");
	fprintf(stdout,"Soundcard CW for Linux  v1.0  Steven J. Merrifield, VK3ESM\n");
	fprintf(stdout,"==========================================================\n");
	fprintf(stdout,"1. Set the speed, currently = %d\n",Morse_speed);
	fprintf(stdout,"2. Set the frequency, currently = %d\n",AdLib_tone);
	fprintf(stdout,"3. Set the volume, currently = %d\n",AdLib_vol);
	fprintf(stdout,"4. Set the delay value, currently = %d\n",Morse_gap);
	fprintf(stdout,"5. Set the character set for random groups, currently = %d\n",Grp_type);
	fprintf(stdout,"6. Set the number of groups, currently = %d\n",Num_grps);
	fprintf(stdout,"7. Receive random character groups.\n");
	fprintf(stdout,"8. Receive a file.\n");
	fprintf(stdout,"9. QUIT\n");
	fprintf(stdout,"==========================================================\n");
	fprintf(stdout,"Enter your choice : ");
	fscanf(stdin,"%d",&choice);
	getchar();	/* flush <return> from buffer */
	switch(choice)
	{
		case 1: Morse_speed = Ask_speed(); break;
		case 2: AdLib_tone = Ask_freq(); break;
		case 3: AdLib_vol = Ask_vol(); break;
		case 4: Morse_gap = Ask_delay(); break;
		case 5:	Grp_type = Ask_grp_type(); break;
		case 6: Num_grps = Ask_num_grps(); break;
		case 7: Send_groups(Num_grps,Grp_type); break;
		case 8: Send_file(); break;
		case 9: return(0); break;
	}		
	return(1);
}

/************************************************************************/
/* Read the previous values from the users home directory		*/
/************************************************************************/
int read_config()
{
	FILE *cf;
	char home_dir[255];
	
	strcpy(home_dir,getenv("HOME"));
	strcat(home_dir,"/.sccw");

	cf = fopen(home_dir,"rt");
	if (cf== NULL) return(1);	/* file doesn't exist  - use builtin settings */

	fscanf(cf,"%d",&Morse_speed);
	fscanf(cf,"%d",&AdLib_tone);
	fscanf(cf,"%d",&AdLib_vol);
	fscanf(cf,"%d",&Morse_gap);
	fscanf(cf,"%d",&Num_grps);
	fclose(cf);
	return(0);
}	

/************************************************************************/
/* Write the current settings to a config file in the users home dir	*/
/************************************************************************/
int write_config()
{
	FILE *cf;
	char home_dir[255];

	strcpy(home_dir,getenv("HOME"));
	strcat(home_dir,"/.sccw");	

	cf = fopen(home_dir,"wt");
	if (cf == NULL) return(1);

	fprintf(cf,"%d\n",Morse_speed);
	fprintf(cf,"%d\n",AdLib_tone);
	fprintf(cf,"%d\n",AdLib_vol);
	fprintf(cf,"%d\n",Morse_gap);
	fprintf(cf,"%d\n",Num_grps);
	fclose(cf);
	return(0);
}

/************************************************************************/
/* Setup the IO ports, and fire up the menu, then reset the IO ports on	*/ 
/* exit. Note that this requires the setuid bit to be set, since we are	*/
/* playing with the hardware directly.					*/
/************************************************************************/
int main(int argc, char *argv[])
{
	time_t *t=NULL,seed;
	ioperm(reg_port,17,1);
	ioperm(data_port,17,1);
	if (!(AdLib_found))
	{
		fprintf(stderr,"\nAdLib/SoundBlaster card not found.\n");
		exit(1);
	}
	seed = time(t);		/* Set random value depending on time */
	srand(seed);		/* So the random char set is always different */
	read_config();
	while (menu()) {};
	ioperm(reg_port,17,0);
	ioperm(data_port,17,0);
	if (write_config()) fprintf(stderr,"\nError writing to config file.\n");
	return(0);
}


