/*
 * Worldvisions Weaver Software:
 *   Copyright (C) 1997 Worldvisions Computer Technology, Inc.
 *
 * Reads /etc/ppp/pap-secrets, and can add entries if need be.
 *
 * Created:	Feb 7 1998		D. Coombs
 *   
 */


#include "wvpapsecrets.h"
#include <assert.h>
#include <ctype.h>

//**************************************************
// class WvPapSecrets public functions
//**************************************************

WvPapSecrets::WvPapSecrets()
/**************************/
{
    opened_ok = open_file();
}

WvPapSecrets::~WvPapSecrets()
/***************************/
{
    if( pap_file.isok() && isok() ) {
	pap_file.close();
    }
}

a_pap_secret *	WvPapSecrets::get_secret( WvString username )
/***********************************************************/
{
    a_pap_secret *	secret;
    char *		from_file;

    if( isok() == false ) {
        return( NULL );
    }

    if( pap_file.isok() == false ) {
        open_file();
        if( pap_file.isok() == false ) {
            return( NULL );
        }
    }

    lseek( pap_file.getfd(), 0, SEEK_SET );

    secret = new a_pap_secret;
    if( !secret ) {
    	return( NULL );
    }

    from_file = pap_file.getline(0);
    while( from_file ) {
	if( parse_secret_line( from_file, secret ) == true ) {
	    // We got a useful line, see if it's the one we want.
	    if( secret->local == username ) {
	    	return( secret );
	    }
	}
	from_file = pap_file.getline(0);
    }

    delete( secret );

    // It wasn't there...
    return( NULL );
}

bool WvPapSecrets::put_secret( a_pap_secret secret )
/**************************************************/
{
    unsigned	bytes;
    char	line_to_write[ WVSTREAM_BUF_SIZE ];

    if( isok() == false ) {
    	return( false );
    }

    if( pap_file.isok() == false ) {
        open_file();
        if( pap_file.isok() == false ) {
            return( false );
        }
    }

    lseek( pap_file.getfd(), 0, SEEK_END );
    sprintf( line_to_write, "%s\t%s\t%s\t%s\n", secret.local.str,
    						secret.remote.str,
    						secret.password.str,
    						secret.rest.str );
    terminate_string( line_to_write, '\n' );
    bytes = pap_file.write( line_to_write, strlen( line_to_write ) );
    return( bytes == strlen( line_to_write ) );
}

bool WvPapSecrets::put_secret( WvString	username, WvString password )
/*******************************************************************/
{
    a_pap_secret	secret;

    secret.local = username;
    secret.remote = "wvdial";
    secret.password = password;
    secret.rest = "";

    return( put_secret( secret ) );
}

void WvPapSecrets::del_secret( WvString username )
/************************************************/
{
    if( isok() == false ) {
    	return;
    }

    WvFile		old_file;
    a_pap_secret	secret;
    char *		from_file;
    char		to_file[ WVSTREAM_BUF_SIZE ];

    pap_file.close();
    rename( PAP_SECRETS, PAP_SECRETS ".old" );
    old_file.open( PAP_SECRETS ".old", O_RDONLY );
    open_file();

    // We need to set the permissions on the new file....
    chmod( PAP_SECRETS, PAP_FILE_PERMS );

    old_file.setbuf( WVSTREAM_BUF_SIZE );
    pap_file.setbuf( WVSTREAM_BUF_SIZE );
    from_file = old_file.getline(0);
    while( from_file ) {
    	strcpy( to_file, from_file );
    	terminate_string( to_file, '\n' );
    	if( parse_secret_line( from_file, &secret ) == true ) {
    	    // We got a useful line, see if we want it gone... :)
    	    if( ! ( secret.local == username ) ) {
    	    	pap_file.write( to_file, strlen( to_file ) );
    	    }
    	} else {
    	    pap_file.write( to_file, strlen( to_file ) );
    	}
    	from_file = old_file.getline(0);
    }
    old_file.close();
    unlink( PAP_SECRETS ".old" );
}

bool WvPapSecrets::isok()
/***********************/
{
    return( opened_ok );
}



//**************************************************
// class WvPapSecrets private functions
//**************************************************

static char *	next_word( char * str )
/*************************************/
{
    // are we null?
    if( *str == '\0' ) {
    	return( NULL );
    }

    // skip to the end of the current word.
    while( *(++str) != '\0' && !isspace( *str ) );

    // did we hit a null?
    if( *str == '\0' ) {
    	return( NULL );
    }

    // skip any whitespace.
    while( *(++str) != '\0' && isspace( *str ) );

    // return str, whether it's null or not.
    return( str );
}

bool WvPapSecrets::parse_secret_line( char *		secret_line,
				      a_pap_secret *	secret )
/**************************************************************/
{
    assert( secret != NULL );

    secret->local = "";
    secret->remote = "";
    secret->password = "";
    secret->rest = "";

    char *	p = trim_string( secret_line );

    // See if this line is a comment.
    if( *p == '#' ) {
    	return( false );
    }

    char *	w2;
    char *	w3;
    char *	w4;

    w2 = next_word( p );
    if( !w2 ) return( false );

    w3 = next_word( w2 );
    if( !w3 ) return( false );

    w4 = next_word( w3 );
    // might not be anything after third word though!

    w2[ -1 ] = '\0'; trim_string( p );
    w3[ -1 ] = '\0'; trim_string( w2 );
    if( w4 ) {
    	w4[ -1 ] = '\0';
    }
    trim_string( w3 );
    if( w4 ) {
    	trim_string( w4 );
    }

    secret->local = p;
    secret->remote = w2;
    secret->password = w3;
    if( w4 ) {
	secret->rest = w4;
    } else {
    	secret->rest = "";
    }

    return( true );
}

bool WvPapSecrets::open_file()
/****************************/
{
    pap_file.open( PAP_SECRETS, O_CREAT | O_EXCL | O_RDWR );
    if( pap_file.isok() == false ) {
    	// couldn't create file, it must already be there.
    	pap_file.open( PAP_SECRETS, O_RDWR );
    	if( pap_file.isok() == false ) {
    	    // okay, problem.
    	    opened_ok = false;
    	    return( false );
    	}
    } else {
    	// well, if we DID create it we should set the permissions...
    	chmod( PAP_SECRETS, PAP_FILE_PERMS );
    }

    pap_file.setbuf( WVSTREAM_BUF_SIZE );
    return( true );
}

