/*
    resolve.c - NWDSResolveName() API demo
    Copyright (C) 1999  Petr Vandrovec

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    Revision history:

	0.00  1999			Petr Vandrovec <vandrove@vc.cvut.cz>
		Initial revision.

	1.00  1999, November 20		Petr Vandrovec <vandrove@vc.cvut.cz>
		Added license.

 */

#include "config.h"

#include <ncp/nwcalls.h>
#include <ncp/nwnet.h>

#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <wchar.h>

#include <libintl.h>
#define _(X) gettext(X)

static char *progname;

static void
usage(void)
{
	fprintf(stderr, _("usage: %s [options]\n"), progname);
}

static void
help(void)
{
	printf(_("\n"
	         "usage: %s [options]\n"), progname);
	printf(_("\n"
	       "-h              Print this help text\n"
	       "-v value        Context DCK_FLAGS value (default 0)\n"
	       "-o object_name  Name of object (default [Root])\n"
	       "-c context_name Name of current context (default OrgUnit.Org.Country)\n"
	       "-C confidence   DCK_CONFIDENCE value (default 0)\n"
	       "-S server       Server to start with (default CDROM)\n"
	       "\n"));
}

int dumpSAddr(unsigned char* buff, size_t l, size_t* pos) {
	u_int32_t val;
	size_t x = *pos;
	
	if (l < x + 4) {
		goto dump;
	}
	val = DVAL_LH(buff, x);
	x += 4;
	printf("Server addresses: %d\n", val);
	while (val-- > 0) {
		u_int32_t type;
		u_int32_t lskip;
				
		if (l < x + 4) {
			goto dump;
		}
		type = DVAL_LH(buff, x);
		x += 4;
		printf("-> type: %d\n", type);
		if (l < x + 4) {
			goto dump;
		}
		type = DVAL_LH(buff, x);
		x += 4;
		printf("   length: %d, ", type);
		lskip = ((type + 3) & ~3) - type;
		while (type--) {
			if (l <= x) {
				goto dump;
			}
			printf("%02X ", buff[x++]);
		}
		printf("\n");
		if (lskip) {
			printf("   skip: ");
			while (lskip--) {
				if (l <= x) {
					goto dump;
				}
				printf("%02X ", buff[x++]);
			}
			printf("\n");
		}
	}
	*pos = x;
	return 0;
dump:;
	printf("end of buffer\n");
	*pos = x;
	return 1;
}

int main(int argc, char *argv[]) {
	NWDSCCODE dserr;
	NWDSContextHandle ctx;
	NWCONN_HANDLE conn, conn2;
	char* objectname = "[Root]";
	char* context = "OrgUnit.Org.Country";
	char* server = "CDROM";
	int opt;
	u_int32_t ctxflag = 0;
	u_int32_t confidence = 0;
	NWObjectID id;
	

	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
	
	progname = argv[0];

	NWCallsInit(NULL, NULL);
	NWDSInitRequester();
	
	dserr = NWDSCreateContextHandle(&ctx);
	if (dserr) {
		fprintf(stderr, "NWDSCretateContextHandle failed with %s\n", strnwerror(dserr));
		return 123;
	}
	while ((opt = getopt(argc, argv, "h?o:c:v:S:C:")) != EOF)
	{
		switch (opt)
		{
		case 'C':
			confidence = strtoul(optarg, NULL, 0);
			break;
		case 'o':
			objectname = optarg;
			break;
		case 'c':
			context = optarg;
			break;
		case 'v':
			ctxflag = strtoul(optarg, NULL, 0);
			break;
		case 'h':
		case '?':
			help();
			goto finished;
		case 'S':
			server = optarg;
			break;
		default:
			usage();
			goto finished;
		}
	}

	ctxflag |= DCV_XLATE_STRINGS;
	
	dserr = NWDSSetContext(ctx, DCK_FLAGS, &ctxflag);
	if (dserr) {
		fprintf(stderr, "NWDSSetContext(DCK_FLAGS) failed: %s\n",
			strnwerror(dserr));
		return 123;
	}
	dserr = NWDSSetContext(ctx, DCK_NAME_CONTEXT, context);
	if (dserr) {
		fprintf(stderr, "NWDSSetContext(DCK_NAME_CONTEXT) failed: %s\n",
			strnwerror(dserr));
		return 122;
	}
	dserr = NWDSSetContext(ctx, DCK_CONFIDENCE, &confidence);
	if (dserr) {
		fprintf(stderr, "NWDSSetContext(DCK_CONFIDENCE) failed: %s\n",
			strnwerror(dserr));
		return 122;
	}
	{
		static const u_int32_t add[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};
		dserr = NWDSSetTransport(ctx, 16, add);
		if (dserr) {
			fprintf(stderr, "NWDSSetTransport failed: %s\n",
				strnwerror(dserr));
			return 124;
		}
	}
	{
		struct ncp_conn_spec connsp;
		long err;
		
		memset(&connsp, 0, sizeof(connsp));
		strcpy(connsp.server, server);
		conn = ncp_open(&connsp, &err);
		if (!conn) {
			fprintf(stderr, "ncp_open failed with %s\n",
				strnwerror(err));
			return 111;
		}
	}
	dserr = NWDSAddConnection(ctx, conn);
	if (dserr) {
		fprintf(stderr, "Cannot bind connection to context: %s\n",
			strnwerror(dserr));
	}
	dserr = NWDSResolveName(ctx, objectname, &conn2, &id);
	if (dserr) {
		fprintf(stderr, "Resolve failed with %s\n", 
			strnwerror(dserr));
	} else {
		char sname[1000];

		printf("ID: %08X\n", id);
		printf("Conn: %p\n", conn2);
		dserr = NWDSGetServerDN(ctx, conn2, sname);
		if (dserr)
			fprintf(stderr, "Get Server DN failed with %s\n",
				strnwerror(dserr));
		else
			printf("Server: %s\n", sname);
		dserr = NWDSMapIDToName(ctx, conn2, id, sname);
		if (dserr)
			fprintf(stderr, "Cannot convert ID to name: %s\n",
				strnwerror(dserr));
		else
			printf("Object: %s\n", sname);

		NWCCCloseConn(conn2);
	}
	NWCCCloseConn(conn);
	dserr = NWDSFreeContext(ctx);
	if (dserr) {
		fprintf(stderr, "NWDSFreeContext failed with %s\n", strnwerror(dserr));
		return 121;
	}
finished:
	return 0;
}
	
