/***************************************************************************
                          levels.c  -  description
                             -------------------
    begin                : Thu Sep 6 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "lbreakout.h"
#include "config.h"
#include "list.h"
#include "file.h"
#include "levels.h"

List *levels = 0; /* dynamic list containing the levels */
char **levelset_names = 0; /* name of all valid levelsets */
int levelset_count = 0;
char **levelset_home_names = 0; /* home directory sets */
int levelset_home_count;
extern Config config;

/*
====================================================================
Locals
====================================================================
*/

/*
====================================================================
Publics
====================================================================
*/

/*
====================================================================
Free current levels if any and load all levels saved
in path/fname.
Return Value: True if successful.
====================================================================
*/
int levels_load( char *path, char *fname )
{
    Level *level;
    FILE *file;

    /* delete old levels */
    if ( levels ) levels_delete();
    /* init list */
    levels = list_create( LIST_AUTO_DELETE, level_delete );
    list_reset( levels );
    /* open file */
    if ( ( file = file_open( path, fname, FILE_READ ) ) == 0 ) return 0;
    /* read levels */
    while( ( level = level_load( file ) ) != 0 ) list_add( levels, level );
    fclose( file );
    /* if no levels loaded at all delete levels */
    if ( list_empty( levels ) ) {
        levels_delete();
        fprintf( stderr, "Ups! No valid levels found in '%s/%s'.\n", path, fname );
        return 0;
    }
    printf( "Loaded %i levels from '%s/%s'.\n", levels->count, path, fname );
    return 1;
}
/*
====================================================================
Delete levels
====================================================================
*/
void levels_delete()
{
    if ( !levels ) return;
    list_delete( levels );
    levels = 0;
}
/*
====================================================================
Get a specific level from list.
Return Value: Level pointer containing information of this level or
NULL if there is no level
====================================================================
*/
Level* levels_get_first()
{
    return list_first( levels );
}
Level* levels_get_next( Level *current_level )
{
    return list_get( levels, list_check( levels, current_level ) + 1 );
}
/*
====================================================================
Return id of this level in list.
====================================================================
*/
int levels_get_id( Level *level )
{
    return list_check( levels, level );
}
/*
====================================================================
Load level from current file position.
====================================================================
*/
int next_line( FILE *file, char **buffer )
{
    /* lines with an # are comments: ignore them */
    if ( !fgets( *buffer, 1023, file ) ) return 0;
    if ( (*buffer)[strlen(*buffer) - 1] == 10 )
        (*buffer)[strlen(*buffer) - 1] = 0;
/*    if ( (*buffer)[0] == '#' )
        next_line( file, buffer ); */
    return 1;
}
Level* level_load( FILE *file )
{
    Level *level = 0;
    char *buffer = calloc( 1024, sizeof( char ) );
    int i, j;

    /* file handle ok? */
    if ( !file ) { free( buffer ); return 0; }
    /* get level mem */
    level = calloc( 1, sizeof( Level ) );
    /* read entries */
    /* level: */
    if ( !next_line( file, &buffer ) ) goto failure;
    if ( !strequal( "Level:", buffer ) ) goto failure;
    /* author */
    if ( !next_line( file, &buffer ) ) goto failure;
    level->author = strdup( buffer );
    /* level name */
    if ( !next_line( file, &buffer ) ) goto failure;
    level->name = strdup( buffer );
    /* bricks: */
    if ( !next_line( file, &buffer ) ) goto failure;
    if ( !strequal( "Bricks:", buffer ) ) goto failure;
    /* load bricks */
    for ( i = 0; i < EDIT_HEIGHT; i++ ) {
        if ( !next_line( file, &buffer ) ) goto failure;
        if ( strlen( buffer ) < EDIT_WIDTH ) goto failure;
        for ( j = 0; j < EDIT_WIDTH; j++ )
            level->bricks[j][i] = buffer[j];
    }
    /* extras: */
    if ( !next_line( file, &buffer ) ) goto failure;
    if ( !strequal( "Bonus:", buffer ) ) goto failure;
    /* load extras */
    for ( i = 0; i < EDIT_HEIGHT; i++ ) {
        if ( !next_line( file, &buffer ) ) goto failure;
        if ( strlen( buffer ) < EDIT_WIDTH ) goto failure;
        for ( j = 0; j < EDIT_WIDTH; j++ )
            level->extras[j][i] = buffer[j];
    }
    /* return level */
    free( buffer );
    return level;
failure:
    level_delete( level );
    free( buffer );
    return 0;
}
/*
====================================================================
Create an empty level
====================================================================
*/
Level* level_create_empty( char *author, char *name )
{
    int i, j;
    Level *level = calloc( 1, sizeof( Level ) );
    level->author = strdup( author );
    level->name = strdup( name );
    /* empty arena */
    for ( i = 0; i < EDIT_WIDTH; i++ )
        for ( j = 0; j < EDIT_HEIGHT; j++ ) {
            level->extras[i][j] = (char)EX_NONE;
            level->bricks[i][j] = (char)-1;
        }
    return level;
}
/*
====================================================================
Delete level pointer.
====================================================================
*/
void level_delete( void *level_ptr )
{
    Level *level = (Level*)level_ptr;
    if ( !level ) return;
    if ( level->author ) free( level->author );
    if ( level->name ) free( level->name );
    free( level );
}

/*
====================================================================
Get/delete list of all valid levelsets in data directory or
~/.lbreakout/levels.
====================================================================
*/
void levelsets_load_names()
{
    List *names;
    Text *text = 0;
    char level_dir[512];
    char file_name[64];
    int i;
    /* clear previous list */
    if ( levelset_names ) levelsets_delete_names();
    /* create dynamic list */
    names = list_create( LIST_NO_AUTO_DELETE, NO_CALLBACK );
    /* parse home directory */
    sprintf( level_dir, "%s/%s/lbreakout2-levels", (getenv( "HOME" )?getenv( "HOME" ):"."), CONFIG_DIR_NAME );
    text = get_file_list( level_dir, 0, level_dir );
    for ( i = 0; i < text->count; i++ ) {
        /* filter stuff */
        if ( text->lines[i][0] == '*' ) continue;
        /* add */
        sprintf( file_name, "~%s", text->lines[i] );
        list_add( names, strdup( file_name ) );
    }
    delete_text( text );
    /* create static levelset_home_names */
    levelset_home_count = names->count + 1;
    levelset_home_names = calloc( levelset_home_count, sizeof( char* ) );
    levelset_home_names[0] = strdup( "<CREATE SET>" );
    for ( i = 1; i < levelset_home_count; i++ )
        levelset_home_names[i] = strdup( list_get( names, i - 1 ) + 1 );
    /* parse source directory */
    sprintf( level_dir, "%s/levels", SRC_DIR );
    text = get_file_list( level_dir, 0, level_dir );
    for ( i = 0; i < text->count; i++ ) {
        /* filter stuff */
        if ( text->lines[i][0] == '*' ) continue;
        if ( strequal( text->lines[i], "Makefile.am" ) ) continue;
        if ( strequal( text->lines[i], "Makefile.in" ) ) continue;
        if ( strequal( text->lines[i], "Makefile" ) ) continue;
        /* add */
        list_add( names, strdup( text->lines[i] ) );
    }
    delete_text( text );
    /* create static list levelset_names */
    levelset_count = names->count;
    levelset_names = calloc( levelset_count, sizeof( char* ) );
    for ( i = 0; i < levelset_count; i++ )
        levelset_names[i] = list_get( names, i );
    /* adjust config */
    if ( config.levelset_count != levelset_count ) {
        config.levelset_id = 0;
        config.levelset_count = levelset_count;
    }
    if ( config.levelset_home_count != levelset_home_count ) {
        config.levelset_home_id = 0;
        config.levelset_home_count = levelset_home_count;
    }
    /* delete list */
    list_delete( names );
}
void levelsets_delete_names()
{
    int i;
    if ( levelset_names ) {
        for ( i = 0; i < levelset_count; i++ )
            if ( levelset_names[i] ) free( levelset_names[i] );
        free( levelset_names );
        levelset_names = 0;
    }
    if ( levelset_home_names ) {
        for ( i = 0; i < levelset_home_count; i++ )
            if ( levelset_home_names[i] ) free( levelset_home_names[i] );
        free( levelset_home_names );
        levelset_home_names = 0;
    }
}
