/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#include <librevenge-stream/librevenge-stream.h>
#include <libmwaw/libmwaw.hxx>
#include <librvngabw/librvngabw.hxx>

#include "ABWStringDocumentHandler.hxx"
#include "UsageHelper.hxx"

#define TOOLNAME "mwaw2abw"

class ABWOutputFileHelper
{
public:
	//! constructor
	ABWOutputFileHelper(const char *outFileName) : mOutFileName(outFileName) {}
	//! destructor
	~ABWOutputFileHelper() {}
	//! try to convert a document
	bool convertTextDocument(librevenge::RVNGInputStream &input)
	{
		writerperfectrvngabw::ABWStringDocumentHandler contentHandler;
		librvngabw::ABWTextGenerator collector(&contentHandler);
		// embedded document created by libmwaw
		collector.registerEmbeddedImageHandler("image/mwaw-odg", &handleEmbeddedMWAWGraphicImage);
		collector.registerEmbeddedObjectHandler("image/mwaw-ods", &handleEmbeddedMWAWSpreadsheetObject);
		// basic picture
		collector.registerEmbeddedImageHandler("image/pict", &handleEmbeddedBasicGraphicImage);
		collector.registerEmbeddedImageHandler("image/x-pict", &handleEmbeddedBasicGraphicImage);
		if (MWAWDocument::MWAW_R_OK != MWAWDocument::parse(&input, &collector))
			return false;
		if (mOutFileName)
		{
			FILE *fhandle = fopen(mOutFileName, "wb");
			if (!fhandle)
				return false;
			fprintf(fhandle, "%s\n", contentHandler.getData().cstr());
			fclose(fhandle);
		}
		else
			printf("%s\n", contentHandler.getData().cstr());
		return true;
	}
	//! returns true if the input corresponds to a text document
	static bool isSupportedFormat(librevenge::RVNGInputStream &input)
	{
		MWAWDocument::Type type;
		MWAWDocument::Kind kind;
		MWAWDocument::Confidence confidence = MWAWDocument::isFileFormatSupported(&input, type, kind);
		if (confidence != MWAWDocument::MWAW_C_EXCELLENT)
		{
			fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Mac Classic document.\n");
			return false;
		}
		if (kind != MWAWDocument::MWAW_K_TEXT)
		{
			fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Mac Classic text document.\n");
			return false;
		}

		return true;
	}
private:
	//! try to convert MWAW created picture in svg
	static bool handleEmbeddedMWAWGraphicImage(const librevenge::RVNGBinaryData &data, librevenge::RVNGBinaryData &output)
	{
		librevenge::RVNGStringVector svgOutput;
		librevenge::RVNGSVGDrawingGenerator generator(svgOutput, "");
		bool result = MWAWDocument::decodeGraphic(data, &generator);
		if (!result || svgOutput.empty() || svgOutput[0].empty())
			return false;

		output.clear();
		const char *svgHeader = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>\n"
		                        "<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.1//EN\""
		                        " \"http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd\">\n";
		output.append((unsigned char *)svgHeader, strlen(svgHeader));
		output.append((unsigned char *)svgOutput[0].cstr(), strlen(svgOutput[0].cstr()));
		return true;
	}
	/** try to convert embedded picture in svg.

		\note if possible, we convert them. If not, we returns the current picture...
	 */
	static bool handleEmbeddedBasicGraphicImage(const librevenge::RVNGBinaryData &data, librevenge::RVNGBinaryData &output)
	{
		librevenge::RVNGStringVector svgOutput;
		librevenge::RVNGSVGDrawingGenerator generator(svgOutput, "");
		if (MWAWDocument::MWAW_R_OK != MWAWDocument::parse(data.getDataStream(), &generator) ||
		        svgOutput.empty() || svgOutput[0].empty())
		{
			output=data;
			return true;
		}
		output.clear();
		const char *svgHeader = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>\n"
		                        "<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.1//EN\""
		                        " \"http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd\">\n";
		output.append((unsigned char *)svgHeader, strlen(svgHeader));
		output.append((unsigned char *)svgOutput[0].cstr(), strlen(svgOutput[0].cstr()));
		return true;
	}
	//! try to convert MWAW created spreadsheet in a table
	static bool handleEmbeddedMWAWSpreadsheetObject(const librevenge::RVNGBinaryData &data, librvngabw::ABWGenerator &generator)
	{
		librvngabw::ABWSpreadsheetToTableGenerator exporter(generator);
		return MWAWDocument::decodeSpreadsheet(data, &exporter);
	}
	//! the ouput file name
	char const *mOutFileName;
};

int printUsage(char const *name)
{
	UsageHelper usage(name, "converts a Mac document to AbiWord.");
	usage.addToDescription("If OUTPUT is missing, the result is printed to standard output.\n");
	usage.addToDescription("The list of recognized formats can be found at\n");
	usage.addToDescription("http://sourceforge.net/p/libmwaw/wiki/Home/\n");
	usage.addToDescription("(at least the list of formats recognized by the lattest version of libmwaw).\n");
	usage.addStdoutOption();
	return usage.printUsage();
}

int main(int argc, char *argv[])
{
	if (argc < 2)
		return printUsage(TOOLNAME);

	char *szInputFile = 0;
	char *szOutFile = 0;
	bool stdOutput = false;

	for (int i = 1; i < argc; i++)
	{
		if (!strcmp(argv[i], "--stdout"))
			stdOutput = true;
		else if (!strcmp(argv[i], "--version"))
			return UsageHelper::printVersion(TOOLNAME);
		else if (!strcmp(argv[i], "--help"))
			return printUsage(TOOLNAME);
		else if (!szInputFile && strncmp(argv[i], "--", 2))
			szInputFile = argv[i];
		else if (szInputFile && !szOutFile && strncmp(argv[i], "--", 2))
			szOutFile = argv[i];
		else
			return printUsage(TOOLNAME);
	}

	if (!szInputFile)
		return printUsage(TOOLNAME);

	if (szOutFile && stdOutput)
		szOutFile = 0;

	librevenge::RVNGFileStream input(szInputFile);
	if (!ABWOutputFileHelper::isSupportedFormat(input))
		return 1;
	ABWOutputFileHelper helper(szOutFile);
	bool ok=helper.convertTextDocument(input);
	if (!ok)
	{
		fprintf(stderr, "ERROR : Couldn't convert the document\n");
		return 1;
	}

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
