///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// This code presents various solvers
//
// also print matlab code for non-regression test purpose, i.e.
//
// usage:
//    makefish 4 | solve_iterative_tst | octave -q
//  machine-precision independent computation:
//    makefish 4 | solve_iterative_tst -ndigits 12 | octave -q
//
# include "rheolef/skit.h"
using namespace rheolef;
using namespace std;

extern int solve_bicg     (const csr<Float>&, const vec<Float>&);
extern int solve_bicgstab (const csr<Float>&, const vec<Float>&);
extern int solve_cg       (const csr<Float>&, const vec<Float>&);
extern int solve_cgs      (const csr<Float>&, const vec<Float>&);
extern int solve_cheby    (const csr<Float>&, const vec<Float>&);
extern int solve_gmres    (const csr<Float>&, const vec<Float>&);
extern int solve_ir       (const csr<Float>&, const vec<Float>&);
extern int solve_qmr      (const csr<Float>&, const vec<Float>&);

bool do_trunc;

Float eps1 = 1e-7;

template <class T>
inline T my_round (const T& x) { return eps1*T(int(x/eps1+0.5)); }

template <class Iterator>
void
round_value (Iterator first, Iterator last)
{
    while (first != last) {
        *first = my_round(*first);
	first++;
    }
}
void check (const char* method, int max_iter, const csr<Float>& a, 
            const vec<Float>& x, const vec<Float>& b)
{
    if (do_trunc) {
        vec<Float> x1 = x;
        round_value(x1.begin(), x1.end());
        cout << "x=" <<  x1 << ";"    << endl;
    } else {
        cout << "x=" <<  x << ";"    << endl;
    }
    cout << "error_" << method << "_" << max_iter;
    if (!do_trunc) {
         cout << "=norm(a*x-b)\n\n";
    } else {
         cout << "=eps1*round(norm(a*x-b)/eps1)\n\n";
    }
}
int main (int argc, char** argv)
{
    do_trunc = false;
    int digits10 = numeric_limits<Float>::digits10;
    // cout.setf(ios::scientific, !(ios::fixed | ios::floatfield) );

    if (argc == 3) {
        // non-regression mode:
        // avoid machine-dependent output digits in non-regression mode

        // avoid variation of output formats 
        // when double or bigfloat
        numeric_flags<Float>::output_as_double(true);

 	do_trunc = true;
        // cout.setf(ios::scientific);
        cout.setf(ios::fixed);
        digits10 = atoi(argv[2]);
	cout << "eps1=sqrt(10^(-" << digits10 << "));\n";
        eps1=pow(Float(10), Float(-digits10/2));
 
    }
    cout << setprecision(digits10);

    csr<Float> a;
    cin >> a;
    vec<Float> x(a.ncol());
    x = 1;
    vec<Float> b = a*x;

    cout << ml << "a=" << a << ";" << endl;
    cout << "b=" << b << ";" << endl;

    solve_bicg(a, b);
    solve_bicgstab(a, b);
    solve_cg(a, b);
    solve_cgs(a, b);
    solve_cheby(a, b);
    solve_gmres(a, b);
    solve_qmr(a, b);
    
    return 0;
}

