/*
    scconfig - detection of types and type sizes
    Copyright (C) 2012  Tibor Palinkas

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

		Project page: http://repo.hu/projects/scconfig
		Contact via email: scconfig [at] igor2.repo.hu
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "libs.h"
#include "log.h"
#include "db.h"
#include "dep.h"

/* assume there is no integer that is at least this wide, in bytes */
#define MAX_INT_WIDTH 9

static int try_size(int logdepth, char *cflags, char *ldflags, const char *type, int use_stdint, const char *path, const char **sizearr)
{
	char *out = NULL;
	const char *test_c_template =
		NL "#include <stdio.h>"
		NL "int main() {"
		NL "		printf(\"OK %%d\\n\", sizeof(%s));"
		NL "	return 0;"
		NL "}"
		NL;
	char test_c[512], *start;
	const char *inc = "#include <stdint.h>\n";
	int size;

	if (use_stdint) {
		strcpy(test_c, inc);
		start = test_c + strlen(inc);
	}
	else
		start = test_c;
	sprintf(start, test_c_template, type);

	report("Testing size of type %25s... ", type);

	logprintf(logdepth, "trying size with ldflags '%s'\n", ldflags == NULL ? get("cc/ldflags") : ldflags);
	if (compile_run(logdepth+1, test_c, NULL, cflags, ldflags, &out) == 0) {
		if (target_emu_fail(out)) {
			report(" FAIL (emulator)\n");
			free(out);
			return -1;
		}

		if (strncmp(out, "OK", 2) == 0) {
			size = atoi(out+3);
			if ((size > 0) && (size < MAX_INT_WIDTH)) {
				sprintf(test_c, "%d", size);
				put(path, test_c);
				sizearr[size] = type;
				report(" OK, size %d byte%s\n", size, (size > 1) ? "s" : "");
			}
			else {
				report(" FAIL, size %d bytes\n", size);
				size = -1;
			}
			free(out);
			return size;
		}
		free(out);
	}
	report(" FAIL (compile)\n");
	return -1;
}

int find_types_stdint(int logdepth, int fatal)
{
	char *test_c =
		NL "#include <stdint.h>"
		NL "int main() {"
		NL "	if (sizeof(uint8_t) == 1)"
		NL "		puts(\"OK\");"
		NL "	return 0;"
		NL "}"
		NL;

	require("cc/cc", logdepth, fatal);

	report("Checking for stdint.h... ");
	logprintf(logdepth, "find_types_stdint: trying to find stdint.h...\n");
	logdepth++;

	if (try_icl(logdepth, "libs/types/stdint", test_c, NULL, NULL, NULL))
		return 0;
	return try_fail(logdepth, "libs/types/stdint");
}

int find_types_sizes(int logdepth, int fatal)
{
	const char *stdint;
	const char *sizearr_u[MAX_INT_WIDTH];
	const char *sizearr_s[MAX_INT_WIDTH];
	int n;
	const char *path_template = "sys/types/size/%d_%c_int";
	char path[64];

	require("cc/cc", logdepth, fatal);
	require("libs/types/stdint/presents", logdepth, 0);
	stdint = get("libs/types/stdint/presents");

	for(n = 0; n < MAX_INT_WIDTH; n++) {
		sizearr_u[n] = NULL;
		sizearr_s[n] = NULL;
	}

	try_size(logdepth+1, NULL, NULL, "unsigned long long int", 0, "sys/types/size/unsigned_long_long_int", sizearr_u);
	try_size(logdepth+1, NULL, NULL, "unsigned char", 0, "sys/types/size/unsigned_char", sizearr_u);
	try_size(logdepth+1, NULL, NULL, "unsigned short int", 0, "sys/types/size/unsigned_short_int", sizearr_u);
	try_size(logdepth+1, NULL, NULL, "unsigned int", 0, "sys/types/size/unsigned_int", sizearr_u);
	try_size(logdepth+1, NULL, NULL, "unsigned long int", 0, "sys/types/size/unsigned_long_int", sizearr_u);

	try_size(logdepth+1, NULL, NULL, "signed long long int", 0, "sys/types/size/signed_long_long_int", sizearr_s);
	try_size(logdepth+1, NULL, NULL, "signed char", 0, "sys/types/size/signed_char", sizearr_s);
	try_size(logdepth+1, NULL, NULL, "signed short int", 0, "sys/types/size/signed_short_int", sizearr_s);
	try_size(logdepth+1, NULL, NULL, "signed int", 0, "sys/types/size/signed_int", sizearr_s);
	try_size(logdepth+1, NULL, NULL, "signed long int", 0, "sys/types/size/signed_long_int", sizearr_s);

	if ((stdint != NULL) && (istrue(stdint))) {
		try_size(logdepth+1, NULL, NULL, "uint8_t", 1, "sys/types/size/uint8_t", sizearr_u);
		try_size(logdepth+1, NULL, NULL, "uint16_t", 1, "sys/types/size/uint16_t", sizearr_u);
		try_size(logdepth+1, NULL, NULL, "uint32_t", 1, "sys/types/size/uint32_t", sizearr_u);
		try_size(logdepth+1, NULL, NULL, "uint64_t", 1, "sys/types/size/uint64_t", sizearr_u);
		try_size(logdepth+1, NULL, NULL, "int8_t", 1, "sys/types/size/int8_t", sizearr_s);
		try_size(logdepth+1, NULL, NULL, "int16_t", 1, "sys/types/size/int16_t", sizearr_s);
		try_size(logdepth+1, NULL, NULL, "int32_t", 1, "sys/types/size/int32_t", sizearr_s);
		try_size(logdepth+1, NULL, NULL, "int64_t", 1, "sys/types/size/int64_t", sizearr_s);
	}

	for(n = 0; n < MAX_INT_WIDTH; n++) {
		if (sizearr_u[n] != NULL) {
			report("Found best fit %d bytes wide uint: %s\n", n, sizearr_u[n]);
			sprintf(path, path_template, n, 'u');
			put(path, sizearr_u[n]);
		}
		if (sizearr_s[n] != NULL) {
			report("Found best fit %d bytes wide sint: %s\n", n, sizearr_s[n]);
			sprintf(path, path_template, n, 's');
			put(path, sizearr_s[n]);
		}
	}

	put("sys/types/size/presents", strue); /* to avoid redetection */

	return 0;
}


static int find_types_something_t(int logdepth, int fatal, const char *typ, const char *preinclude)
{
	char *out = NULL;
	int res;
	char test_c[512];
	char node[256], *nodeend;
	const char **include, *includes[] = {"#include <stdlib.h>", "#include <stddef.h>", "#include <sys/types/types.h>", "#include <unistd.h>", "#include <stdio.h>", NULL};

	char *test_c_include =
		NL "%s"
		NL "int puts(const char *s);"
		NL "%s"
		NL "int main() {"
		NL "	%s s;"
		NL "	puts(\"OK\");"
		NL "	return 0;"
		NL "}"
		NL;

	char *test_c_broken =
		NL "%s"
		NL "int puts(const char *s);"
		NL "%s"
		NL "int main() {"
		NL "	%s s;"
		NL "	void *v;"
		NL "	if (sizeof(v) != sizeof(s)) puts(\"yes\");"
		NL "	else puts(\"no\");"
		NL "	return 0;"
		NL "}"
		NL;

	require("cc/cc", logdepth, fatal);

	report("Checking for type %s... ", typ);
	logprintf(logdepth, "find_types_something_t: Checking for %s...\n", typ);
	logdepth++;

	sprintf(node, "sys/types/%s/", typ);
	nodeend = node + strlen(node);

	if (preinclude == NULL)
		preinclude = "";

	for(include = includes; *include != NULL; include++) {
		sprintf(test_c, test_c_include, preinclude, *include, typ);
		if ((compile_run(logdepth, test_c, NULL, NULL, NULL, &out) == 0) && (strncmp(out, "OK", 2) == 0)) {
			report("Found; ");
			logprintf(logdepth+1, "include %s works\n", *include);
			sprintf(nodeend, "includes");
			put(node, *include);
			break;
		}
		logprintf(logdepth+1, "include %s fails\n", *include);
		if (out != NULL)
			free(out);
	}
	if (*include == NULL) {
		report("Not found\n");
		return 1;
	}

	sprintf(nodeend, "presents");
	put(node, strue);

	/* check if typ is broken (smaller than void *) */
	sprintf(test_c, test_c_broken, preinclude, *include, typ);
	if (compile_run(logdepth, test_c, NULL, NULL, NULL, &out) == 0) {
		if ((out != NULL) && (strncmp(out, "yes", 3) == 0)) {
			report("(%s is narrower than void *)\n", typ);
			sprintf(nodeend, "broken");
			put(node, strue);
			res = 0;
		}
		else if ((out != NULL) && (strncmp(out, "no", 2) == 0)) {
			report("(%s is not narrower than void *)\n", typ);
			sprintf(nodeend, "broken");
			put(node, sfalse);
			res = 0;
		}
		else {
			report("ERROR: test failed (%s)\n", out);
			res = 1;
		}
	}
	if (out != NULL)
		free(out);
	return res;
}


int find_types_size_t(int logdepth, int fatal)
{
	return find_types_something_t(logdepth, fatal, "size_t", NULL);
}

int find_types_off_t(int logdepth, int fatal)
{
	return find_types_something_t(logdepth, fatal, "off_t", NULL);
}

int find_types_off64_t(int logdepth, int fatal)
{
	return find_types_something_t(logdepth, fatal, "off64_t", NULL) &&
	       find_types_something_t(logdepth, fatal, "off64_t", "#define _LARGEFILE64_SOURCE");
}


int find_types_ptrdiff_t(int logdepth, int fatal)
{
	return find_types_something_t(logdepth, fatal, "ptrdiff_t", NULL);
}

