/*
 Copyright 2019 New Vector Ltd

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

#import <Foundation/Foundation.h>

NS_ASSUME_NONNULL_BEGIN

/**
 Short Authentication String verification utility class.
 */
@interface OLMSAS : NSObject

/**
 Get the public key of the SAS object.
 */
- (NSString * _Nullable)publicKey;

/**
 Set the public key of other user.

 @param theirPublicKey the other user's public key.
 @return error the error if any.
 */
- (NSError* _Nullable)setTheirPublicKey:(NSString*)theirPublicKey;

/**
 Generate bytes to use for the short authentication string.

 @param info extra information to mix in when generating the bytes, as per the Matrix spec.
 @param length the size of the output buffer. For hex-based SAS as in the Matrix spec, this will be 5.
 @return generated bytes
 */
- (NSData *)generateBytes:(NSString*)info length:(NSUInteger)length;

/**
 Generate a message authentication code (MAC) based on the shared secret.

 @param input the message to produce the authentication code for.
 @param info extra information to mix in when generating the MAC, as per the Matrix spec.
 @param error the error if any.
 @return the MAC.
 */
- (NSString *)calculateMac:(NSString*)input info:(NSString*)info error:(NSError* _Nullable *)error;

/**
 Generate a message authentication code (MAC) based on the shared secret.
 For compatibility with an old version of olm.js.

 @param input the message to produce the authentication code for.
 @param info extra information to mix in when generating the MAC, as per the Matrix spec.
 @param error the error if any.
 @return the MAC.
 */
- (NSString *)calculateMacLongKdf:(NSString*)input info:(NSString*)info error:(NSError* _Nullable *)error;

@end

NS_ASSUME_NONNULL_END
