# btsutils -- Python module to interact with debbugs servers.
# Copyright (C) 2007  Gustavo R. Montesino
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

import re
import urlparse
import SOAPpy
import SOAPpy.Types

import BugExceptions
from bug import Bug

class soap:
    
    DEFAULT_URL = "http://bugs.debian.org/cgi-bin/soap.cgi"
    DEFAULT_NS = "Debbugs/SOAP/V1"

    def __init__(self, url=DEFAULT_URL, ns=DEFAULT_NS, users=[]):
        """Creates a Debbugs SOAP communication channel.

        url: Debbugs server address
        ns: Debbugs server SOAP namespace
        users: Users to use with usertags"""

        self.connect(url, ns, users)
 
    def connect(self, url=DEFAULT_URL, namespace=DEFAULT_NS, users=[]):
        """Change the debbugs server information"""

        self.url = url
        self.ns = namespace
        self.proxy = SOAPpy.SOAPProxy(url, namespace)
        self.setUsers(users)

    def setUsers(self, users):
        """Defines the users to get usertags"""
        
        self.usertags = {}
        for user in users: 
            usertags = SOAPpy.Types.simplify(self.proxy.get_usertag(user))
            if usertags == '':
                # Invalid user
                continue
            for tag, bugs in usertags.iteritems():
                for bug in bugs:
                    if self.usertags.has_key(bug):
                        self.usertags[bug].append(tag)
                    else:
                        self.usertags[bug] = [tag]

    def parse(self, soapbug):
        """Parse a bug returned from soap to a Bug instance

        This function receives a SOAPpy.Types.StructType from
        the result of a debbugs soap call and returns a 
        btsutils.bug.Bug instance of that bug."""

        bug = Bug()
        bug.setBug(soapbug['id'])
        bug.setPackage(soapbug['package'])
        bug.setSummary(unicode(soapbug['subject'], 'utf-8'))
        bug.setSeverity(soapbug['severity'])
        bug.setSubmitter(unicode(soapbug['originator'], 'utf-8'))
        bug.setTags(soapbug['tags'].split())
        bug.setStatus(soapbug['pending'])
        bug.setForwarded(soapbug['forwarded'])

        if self.usertags.has_key(soapbug['id']):
            bug.setUserTags(self.usertags[soapbug['id']])
        
        baseurl = urlparse.urlsplit(self.url)[1]
        bug.setURL("http://%s/%s" % (baseurl, bug.getBug()))

        return bug

    def get(self, bugnumber):
        """Return a single bug through SOAP
    
        Query the SOAP server and return an Bug instance
        with the bug data.
        """

        raw = self.proxy.get_status(bugnumber)
        if raw == '':
            raise BugExceptions.InvalidBugIdError("Invalid bug id: %s" % bugnumber)
        else:
            return self.parse(raw['item']['value'])

    def getList(self, *buglist):
        """Return a list of bugs by their numbers

        Query the SOAP interface and return a list of Bug instances.
        It's similar to the get function, but return more than
        one bug using only one SOAP call"""

        bugs = []

        soapbugs = self.proxy.get_status(*buglist)['item']

        for soapbug in soapbugs:
            bugs.append(self.parse(soapbug['value']))

        return bugs

    def queryToArgs(self, querystr):
        """Transforms a query string into a list of arguments to debbugs' SOAP"""

        # Source package
        if querystr.startswith("src:"):
            args = [ "src", querystr[4:].strip() ]

        # Binary package
        elif querystr.startswith("pkg:"):
            args = [ "package", querystr[4:].strip() ]

        # Maintainer
        elif querystr.startswith("maint:"):
            args = [ "maint", querystr[6:].strip() ]

        # Submitter
        elif querystr.startswith("from:"):
            args = [ "submitter", querystr[5:].strip() ]

        else:
            raise BugExceptions.InvalidQueryError("Invalid query: %s" % querystr)

        return args

    def query(self, querystr):
        """Returns a list of bugs which matches the query string"""

        queries = re.findall("([&|+-])?\s*\((.*?)\)", querystr)

        if queries == []:
            queries = [('', querystr)]

        result = set([])
        for query in queries:

            arguments = query[1].split("&")

            filter = []
            args = []
            for argument in arguments:
                current = []

                if argument.startswith("bug:"):
                    # Specifing a bug number with other stuff makes no sense,
                    # just return the specified bug and ignore everything else
                    current = [ argument[4:].strip() ]
                    break
 
                elif argument.startswith("usertag:"):
                    # If usertags is specified more than one time, return bugs
                    # which match any of them (instead of all of them)
                    filter.extend(self.query_usertags_id(*argument[8:].split()))

                else:
                    for arg in self.queryToArgs(argument):
                        args.append(arg)

            if args != []:
                current = self.proxy.get_bugs(*args)
                if filter != []:
                    current = set(current) & set(filter)
            elif filter != []:
                current = filter
            elif current == []:
                raise BugExceptions.InvalidQueryError("Invalid query: %s" % querystr)

            if query[0] == '':
                result = set(current)
            elif query[0] == '&':
                result = result & set(current)
            elif query[0] == '|' or query[0] == '+':
                result = result | set(current)
            elif query[0] == '-':
                result = result - set(current)

        result = list(result)

        if len(result) == 0:
            return []
        elif len(result) == 1:
            return [ self.get(result[0]) ]
        else:
            return self.getList(*result)

    def query_usertags_id(self, *tags):
        """Returns a list of bug id by usertags

        Returns a list with the id of all the bugs containing any of the
        (user)tags specified."""

        bugnumlist = []
        for bug, taglist in self.usertags.iteritems():
            if set(taglist) & set(tags) != set([]):
                bugnumlist.append(bug)
        
        return bugnumlist

    def query_usertags(self, *tags):
        """Returns a list of bugs by usertag

        Returns all bugs which contains any of the usertags specified"""

        bugnumlist = self.query_usertags_id(*tags)

        if len(bugnumlist) != 0:
            return self.getList(*bugnumlist)
        else:
            return []

# vim: tabstop=4 expandtab shiftwidth=4
