/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sstream>
#include <netdb.h>
#include <arpa/inet.h> /* for inet_ntoa */

#include "wfservice.h"
#include "wfport.h"
#include "defs.h"


wf_service::wf_service(const wf_protocol& protocol, uint16_t port,
		       const wf_ipaddr& ipaddr) :
  _protocol(protocol),
  _port(port),
  _ipaddr(ipaddr)
{}


void
wf_service::protocol_set(const wf_protocol& protocol) {
  _protocol = protocol;
}

int16_t
wf_service::protocol_get() const {
  return _protocol.get();
}


bool
wf_service::port_set(uint16_t port) {
  if (wf_port_check(port) == false)
    return false;

  _port = port;
  return true;
}

uint16_t
wf_service::port_get() const {
  return _port;
}


void
wf_service::ipaddr_set(const wf_ipaddr& ipaddr) {
  _ipaddr = ipaddr;
}

const wf_ipaddr&
wf_service::ipaddr_get() const {
  return _ipaddr;
}


string
wf_service::tostr() const {
  ostringstream os;
  struct servent *servent;

  os << _ipaddr << ':' << _port;

  servent = getservbyport(htons(_port), _protocol.tostr().c_str());
  if (servent != NULL)
    os << '(' << servent->s_name << ')';
  
  os << '/' << _protocol;

  return os.str();
}

ostream&
wf_service::print(ostream& os) const {
  return os << *this;
}

ostream&
wf_service::debugprint(ostream& os) const {
  return os << _("Service:\t") << *this << endl;
}

bool
operator==(const wf_service& service1, const wf_service& service2) {
  return (service1._protocol == service2._protocol &&
	  service1._port == service2._port &&
	  service1._ipaddr == service2._ipaddr);
}

ostream&
operator<<(ostream& os, const wf_service& service) {
  return os << service.tostr();
}
