/*
 *  Copyright (C) 2000 by Marco G"otze.
 *
 *  This code is part of the ThoughtTracker source package, which is
 *  distributed under the terms of the GNU GPL2.
 */

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <string>
#include <strstream>

#include <gtk--/alignment.h>
#include <gtk--/box.h>
#include <gtk--/buttonbox.h>
#include <gtk--/frame.h>
#include <gtk--/main.h>
#include <gtk--/tooltips.h>
#include <gtk--/window.h>

#include "thoughttracker.h"
#include "app.h"
#include "entry.h"
#include "fontdlg.h"
#include "optionsdlg.h"
#include "querydlg.h"

#ifdef ENABLE_DEBUG
#undef DMSG 
#define DMSG cerr << "TTOptionsDialog::" << __FUNCTION__ << "(): "
#endif  /* ENABLE_DEBUG */

using SigC::bind;
using SigC::slot;

using namespace Gtk;

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
TTOptionsDialog::TTOptionsDialog(TTApplication *application)
  : app(application), db_changes(false), done(false)
{
  // configure self
  set_title(_("Options..."));
  set_wmclass(WM_CLASS_NAME, "options_dialog");
  set_transient_for(*app);
  realize();
  GdkWindowHints hints = GDK_HINT_BASE_SIZE;
  GdkGeometry geometry;
  geometry.min_width = 150;
  geometry.min_height = 100;
  set_geometry_hints(*this, &geometry, hints);
  set_position(GTK_WIN_POS_MOUSE);
  set_policy(false, false, true);
  set_border_width(0);

  // configure main vbox
  Gtk::Box *mainbox = manage(new Gtk::VBox);
  add(*mainbox);

  // configure frame with check buttons
  Gtk::Frame *frame = manage(new Gtk::Frame(_("Options")));
  frame->set_shadow_type(GTK_SHADOW_ETCHED_IN);
  frame->set_border_width(5);
  mainbox->pack_start(*frame);
  Alignment *align = manage(new Alignment(1.0, 1.0, 0.5, 0.5));
  frame->add(*align);
  Gtk::Box *box = manage(new Gtk::VBox);
  box->set_border_width(5);
  align->add(*box);

  Gtk::Box *hbox = manage(new Gtk::HBox);
  hbox->set_border_width(0);
  box->pack_start(*hbox, false, false, false);
  w.o.fixeddefault = manage(new CheckButton(_("fixed font by default"), 0,
    0.5));
  w.o.fixeddefault->toggled.connect(bind(slot(this,
    &TTOptionsDialog::toggle), w.o.fixeddefault, &app->opts.fixeddefault));
  manage(new Tooltips)->set_tip(*w.o.fixeddefault,
    _("use fixed font for details fields by default"));
  hbox->pack_start(*w.o.fixeddefault, false, false, false);
  Gtk::Button *btn = manage(new Gtk::Button(_("Fixed font...")));
  manage(new Tooltips)->set_tip(*btn,
    _("choose which font to use as fixed-width font"));
  btn->clicked.connect(slot(this, &TTOptionsDialog::btn_set_font));
  hbox->pack_start(*btn, false, false, false);

  w.o.autolink = manage(new CheckButton(_("auto-link"), 0, 0.5));
  w.o.autolink->toggled.connect(bind(slot(this,
    &TTOptionsDialog::toggle), w.o.autolink, &app->opts.autolink));
  manage(new Tooltips)->set_tip(*w.o.autolink,
    _("suggest linking new and previously viewed entries"));
  box->pack_start(*w.o.autolink, false, false, false);
  w.o.quicksearch = manage(new CheckButton(_("quick search"), 0, 0.5));
  w.o.quicksearch->toggled.connect(bind(slot(this,
    &TTOptionsDialog::toggle), w.o.quicksearch, &app->opts.quicksearch));
  manage(new Tooltips)->set_tip(*w.o.quicksearch,
    _("immediately show single search results"));
  box->pack_start(*w.o.quicksearch, false, false, false);
  w.o.nolistlimit = manage(new CheckButton(_("no list limit"), 0, 0.5));
  w.o.nolistlimit->toggled.connect(bind(slot(this,
    &TTOptionsDialog::toggle), w.o.nolistlimit, &app->opts.nolistlimit));
  manage(new Tooltips)->set_tip(*w.o.nolistlimit,
    _("don't truncate the list of search results"));
  box->pack_start(*w.o.nolistlimit, false, false, false);
  w.o.savesize = manage(new CheckButton(_("save size"), 0, 0.5));
  w.o.savesize->toggled.connect(bind(slot(this,
    &TTOptionsDialog::toggle), w.o.savesize, &app->opts.savesize));
  manage(new Tooltips)->set_tip(*w.o.savesize,
    _("remember the main window's size"));
  box->pack_start(*w.o.savesize, false, false, false);

  // configure buttons for actions
  frame = manage(new Gtk::Frame(_("Actions")));
  frame->set_shadow_type(GTK_SHADOW_ETCHED_IN);
  frame->set_border_width(5);
  mainbox->pack_start(*frame);
  Gtk::ButtonBox *bbox = manage(new Gtk::VButtonBox);
  bbox->set_border_width(10);
  frame->add(*bbox);
  btn = manage(new Gtk::Button(_("Check database consistency")));
  manage(new Tooltips)->set_tip(*btn,
    _("check database consistency, repair if necessary"));
  btn->set_flags(GTK_CAN_DEFAULT);
  btn->clicked.connect(slot(this, &TTOptionsDialog::btn_check_db));
  bbox->pack_start(*btn, false, false, false);
  btn = manage(new Gtk::Button(_("Clear bookmarks")));
  manage(new Tooltips)->set_tip(*btn, _("delete all bookmarks"));
  btn->set_flags(GTK_CAN_DEFAULT);
  btn->clicked.connect(slot(this, &TTOptionsDialog::btn_clear_bm));
  bbox->pack_start(*btn, false, false, false);

  // configure buttons for terminating the dialog
  bbox = manage(new Gtk::HButtonBox);
  bbox->set_border_width(10);
  mainbox->pack_start(*bbox, false, false, false);
  w.done = manage(new Gtk::Button(_("Done")));
  w.done->set_flags(GTK_CAN_DEFAULT);
  w.done->clicked.connect(slot(this, &TTOptionsDialog::be_done));
  bbox->pack_start(*w.done, false, false, false);
  w.done->grab_default();

  // configure status bar
  w.sbar = manage(new Gtk::Statusbar);
  mainbox->pack_start(*w.sbar, false, false, false);

  // set current options' values
  w.o.fixeddefault->set_active(app->opts.fixeddefault);
  w.o.autolink->set_active(app->opts.autolink);
  w.o.quicksearch->set_active(app->opts.quicksearch);
  w.o.nolistlimit->set_active(app->opts.nolistlimit);
  w.o.savesize->set_active(app->opts.savesize);
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
void
TTOptionsDialog::run()
{
  // save entry possibly being edited first, since one or more options of the
  // options dialog may result in changes of what's currently being displayed
  if (!app->w.entry->save_record(false, true)) return;

  // run the dialog modally
  show_all();
  set_modal(true);
  sbar_msg(_("Choose an option..."));
  while (!done) Main::iteration();
  app->w.entry->set_fixed_font(app->opts.fixedfont);
  app->save_config();

  // update UI information the hard way if errors in the DB have been corrected
  if (db_changes) {
    app->w.entry->btn_clear();
    app->w.search->btn_clear();
    app->w.mbar->update_bookmark(-1);
  }
}

//.............................................................................
void
TTOptionsDialog::btn_set_font()
{
  TTFontDialog dlg(app, _("Fixed-width font..."));
  const gchar *dummy[] = { 0 }, *spacings[] = { "c", "m", 0 };
  dlg.set_filter(GTK_FONT_FILTER_BASE, GTK_FONT_ALL, dummy, dummy, dummy,
    dummy, spacings, dummy);
  string s = dlg.run();
  if (s.length()) app->opts.fixedfont = s;
}

//.............................................................................
void
TTOptionsDialog::btn_check_db()
{
  int status = DB_SUCCESS;
  bool errors = false;
  ostrstream res;

  sbar_msg(_("Checking administrative information..."));
  status = app->db.check_administrative_information(true);
  if (status > 0) {
    errors = true;
    res << "\n  " << _("- invalid administrative information");
  }

  // continue only if previous tests didn't fail critically
  if (status != DB_FAILURE) {
    sbar_msg(_("Checking data structures..."));
    status = app->db.check_data_structures(true);
    if (status > 0) {
      errors = true;
      res << "\n  " << _("- bad data structures") << " (" << status << ' ' <<
        (status == 1 ? _("entry") : _("entries")) << ')';
    }
  }

  if (status != DB_FAILURE) {
    sbar_msg(_("Checking links..."));
    status = app->db.check_stray_linklists(true);
    if (status > 0) {
      errors = true;
      res << "\n  " << _("- stray link lists") << " (" << status << ' ' <<
        (status == 1 ? _("entry") : _("entries")) << ')';
    }
  }

  if (status != DB_FAILURE) {
    status = app->db.check_linklist_structures(true);
    if (status > 0) {
      errors = true;
      res << "\n  " << _("- bad link list structures") << " (" <<
        status << ' ' <<
        (status == 1 ? _("entry") : _("entries")) << ')';
    }
  }

  if (status != DB_FAILURE) {
    status = app->db.check_links_consistency(true);
    if (status > 0) {
      errors = true;
      res << "\n  " << _("- consistency of links") << " (" <<
        status << ' ' <<
        (status == 1 ? _("entry") : _("entries")) << ')';
    }
  }

  if (status != DB_FAILURE) {
    sbar_msg(_("Optimizing database..."));
    if (!app->db.optimize()) {
      errors = true;
      res << "\n  " << _("- optimizing the database failed");
    }
  }

  // done: present user with summary
  if (status == DB_FAILURE)
    error_popup(app, _("There was a critical error performing the check."));

  // display this only now so the user has time to read what check ran last
  sbar_msg(_("Done checking the database."));

  if (errors) {  // checking just pcount might cause a crash if pristine
    string s = res.str();
    s.resize(res.pcount());
    error_popup(app,
      string(_("Database consistency has been checked.  "
               "The following errors have been corrected:")) + s);
  } else {
    message_popup(app, _("Results..."),
      _("Database consistency has been verified.  No errors found."));
  }

  if (status == DB_FAILURE || errors) db_changes = true;
}

//.............................................................................
void
TTOptionsDialog::btn_clear_bm()
{
  if (query_popup(app, _("Clear bookmarks..."),
    _("Are you sure you want to delete all bookmarks?"),
    TTQD_BTN_YES | TTQD_BTN_NO, TTQD_BTN_NO) != TTQD_BTN_YES)
  {
    return;
  }
  app->w.mbar->clear_bookmarks();
  sbar_msg(_("All bookmarks have been deleted."));
}

