/*
 * (C) Copyright 2002, Schlund+Partner AG
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef SP_GXML_XMLTREE_HPP
#define SP_GXML_XMLTREE_HPP

/**
 * @file XMLTree.hpp
 * @version @$Id: XMLTree.hpp,v 1.9 2003/02/27 14:06:49 geye Exp $
 * @author Schlund + Partner AG
 * @brief *ps*
 *
 * (C) Copyright by Schlund+Partner AG
 *
 * Synopsis: @#include <sp-gxmlcpp/XMLTree.hpp>
 *
 */

// STDC++
#include <string>
#include <memory>
#include <fstream>

// Local
#include <sp-gxmlcpp/XMLNodeSet.hpp>
#include <sp-gxmlcpp/XMLException.hpp>


/* NOTES
   o xmlChar is one byte.
   o "xmlChar" is compatible to "char"; xmlChar * Strings are always 0-terminated; (xmlChar *) is compatible to (char *)/string.c_str()
   o Simply using char *, string makes us still compatible with: UTF-8, ASCII, ISO-8859-*, ... (basically, all "up-to-8-Bits-Encodings").
   o UCS-2/UCS-4 encodings might be a problem.

   UNICODE  ISO 10646    All characters of the planet listed and labeled....

   UCS-2: 16-Bit Encoding of UNICODE.
   UCS-4: 32-Bit Encoding of UNICODE.
   UTF-8: 8-Bit Encoding of UNICODE. All todays ISO-8859-1 texts qualify as UTF-8. This is what UNIXses will do.
*/

namespace SP
{
namespace GXML
{

//
// XMLTree
//
class XMLTree
{
	/* enum Exceptions
		 {
		 PARSE_ERR=1,           // XML could not be parsed
		 CONTEXT_ERR,           // Could not create (XPath) context for Tree
		 NO_NODE_FROM_PATH_ERR, // Could not find node from specified path
		 NODE_CREATION_ERR,     // Could not create a node
		 NODE_ADDING_ERR,       // Could not add a creted node
		 TREE_INVALID_ERR       // Could not add a creted node
		 }; */
public:
	// Creators for char-Buffer, C++ string, ifstream
	XMLTree(const xmlDocPtr doc);
	XMLTree(const char * xmlBuffer, int size=-1);  // if size < 0, xmlBuffer must be a valid C-String!
	XMLTree(const std::string & xmlString);
	XMLTree(std::ifstream & xmlStream);
	XMLTree(std::istream & xmlStream);

	~XMLTree();

	std::string getString( const std::string& xpath );
	bool getBool( const std::string& xpath );
	double getFloat( const std::string& xpath );

	// REMARK : the returned XMLNodeSet has to be freed by the calling method
	std::auto_ptr<XMLNodeSet> getNodeSet( const std::string& xpath );

	////////////////////////////////////////////////////////////////////////
	// Get internal libxml2 Pointers
	//  - this will allow us to call libxml2-Functions on the tree directly.
	//    (i.e., helling the abstraction for speed)
	xmlDocPtr getDocPtr() const;
	xmlXPathContextPtr getXPathContextPtr() const;

	////////////////////////////////////////////////////////////////////////
	// "Get Methods" based on "XPath"; paths are XPath Expressions;
	// all paths can be given an optional position
	//  - Methods delivering (xmlChar *) point into the tree
	//  - Methods delivering (std::string, int) obviously copy from tree
	//  - getValue-Methods will deliver the contents of the first child only
	//    e.g.: getValue("/req/X", 0) on
	// "<req><X>a b c<subtag>def</subtag></X></req>" returns: "a b c" only.

	// "libxml2"-Functions. Do not use if you want to be abstract.
	xmlChar * getXmlCharValue(const xmlChar * path) const;
	xmlChar * getXmlCharValue(const std::string & path) const;

	// "C++"-functions
	char * getAddrValue(const std::string & path) const;
	std::string getValue(const std::string & path) const;

	char * getAddrName(const std::string & path) const;
	std::string getName(const std::string & path) const;

	void setValue(const std::string & path, char * value);
	void setValue(const std::string & path, const std::string & value);

	/* Obsoleted by XMLDump
	// getTreeAsString=recursive: This gets the complete contents of the tag
	//  e.g.: getTreeAsString("/req/X", 0) on "<req><X>a b c<subtag>def</subtag>
	// </X></req>" returns: "<X>a b c<subtag>def</subtag></X>".
	std::string getTreeAsString(const std::string & path) const;
	*/

	int getCount(const xmlChar * path) const;
	int getCount(const char * path) const;
	int getCount(const std::string & path) const;

	////////////////////////////////////////////////////////////////////////
	// "Modifying Methods" based on "XPath"; paths are XPath Expressions
	//  - Methods delivering (xmlChar *) point into the tree
	//  - Methods delivering (std::string, int) copy from tree
	void delTag(const std::string & path);
	void addTag(const std::string & path, const std::string & name, const std::string & content);

	/**
	 * Add a new node at the same level(tree depth) the given xpath points to
	 *
	 * @param path to sibling node
	 * @param name the name of the new node
	 * @param content the content of the new node
	 */
	void addSiblingTag(const std::string & path, const std::string & name, const std::string & content);

	/**
	 * return sibling nodes beneath the given xpath expression
	 *
	 * @param path to parent node
	 *
	 * @return all nodes beneath the given xpath
	 */
	std::string getSiblingXML( const std::string& path = "/*" );

	// addTree: Adds the whole tree as new child for node
	// This will COPY. No other method available from libxml2. We cannot mix trees.
	void addTree(const std::string & path, XMLTree * xmlTree);
	void addXML(const std::string & path, const std::string & xml);

	std::string getXML(const std::string & path="/") const;

	std::auto_ptr<XMLTree> getTree(const std::string &path="/");

	////////////////////////////////////////////////////////////////////////
	// Utility functions
	//
	// Node from Path
	static xmlNodePtr nodeFromPath(xmlXPathContextPtr context, const xmlChar * path);
	// Node from Path
	static xmlNodePtr nodeFromPath(xmlXPathContextPtr context, const std::string & path);

private:
	xmlDocPtr tree_;
	xmlXPathContextPtr context_;

	// Constructor helper function
	void genTree(const char * xmlBuffer, int size=-1);

	xmlXPathObjectPtr createXPathObject( const std::string& xpath );
	void destroyXPathObject( xmlXPathObjectPtr obj );
};

}}
#endif
