/*
 * rtp-qm.cc --
 *
 *      Receives audio packets from the Audio Agent and passes to the
 *      slave (usually rtp-qm-indexer)
 *
 * Copyright (c) 2000-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include "tclcl.h"
#ifndef WIN32
#include <sys/time.h>
#else
#include "time.h"
#endif
#include "qm-files.h"
#include "rtp-qm.h"

static class RTPQMRcvrClass : public TclClass {
public:
	RTPQMRcvrClass() : TclClass("Module/RTPQMRcvr") {}
	TclObject* create(int argc, const char*const* argv) {
		return(new RTPQMRcvr(argc, argv));
	}
} rtp_qm_rcvr_class;

/**********************************************************/

RTPQMRcvr::RTPQMRcvr(int argc, const char* const* argv)
{
//  printf("RTPQMRcvr::RTPQMRcvr: called; this=0x%x\n", (unsigned int)this);
  struct QMHeader hdr;

  numPkts = 0;
  outfile = NULL;

  if(argc > 4)
  {
    if(argv[4][0] != '\0')
    {
      outfile = fopen(argv[4], "w");
      if(outfile == NULL)
      {
	fprintf(stderr, "unable to open timestamps file %s\n", argv[4]);
      }
      else
      {
	fwrite(&hdr, sizeof(struct QMHeader), 1, outfile);
      }
    }
  }

  logT = NULL;
  if(argc > 5)
  {
    if(argv[5][0] != '\0')
    {
      Tcl& tcl = Tcl::instance();
      logT = (LogTransducer*)tcl.lookup(argv[5]);
      if(logT == NULL)
      {
	fprintf(stderr, "major problem, logT is NULL!\n");
      }
    }
    //printf("logT is 0x%x\n", (unsigned int)logT);
  }

  // attach a slave recvr if desired
  slaveRcvr = NULL;
  if(argc > 6)
  {
    if(argv[6][0] != '\0')
    {

      Tcl& tcl = Tcl::instance();
      slaveRcvr = (PacketModule*)tcl.lookup(argv[6]);
      if(slaveRcvr == NULL)
      {
	fprintf(stderr, "major problem, slaveRcvr is NULL!\n");
      }
    }
    //printf("slaveRcvr is 0x%x\n", (unsigned int)slaveRcvr);
  }
}


RTPQMRcvr::~RTPQMRcvr()
{
  struct QMHeader hdr;

  if(outfile != NULL)
  {
    hdr.version = QM_TIMESTAMPS_FILE_VERSION;
    hdr.numPackets = numPkts;
    hdr.postProcessed = 0;
    fseek(outfile, 0, SEEK_SET);
    fwrite(&hdr, sizeof(struct QMHeader), 1, outfile);
    fflush(outfile);
    fclose(outfile);
  }
}

void RTPQMRcvr::recv(pktbuf* pb)
{
  struct QMPacketInfo info;

  //  printf("RTPQMRcvr::recv: got packet\n");
  numPkts++;
  rtphdr* rh = (rtphdr*)pb->data;

  info.flags = ntohs(rh->rh_flags);
  info.seqno = ntohs(rh->rh_seqno);
  info.ts = ntohl(rh->rh_ts);
  info.ssrc = ntohl(rh->rh_ssrc);
  info.index = -1;
  gettimeofday(&(info.sysTime), NULL);

  // printf("len=%d, layer=%d\n", pb->len, pb->layer);
  // printf("flags=0x%x\n", ntohs(rh->rh_flags));
//  printf("\tts=0x%lx\n", ntohl(rh->rh_ts));
//  printf("seqno=%u\n", info.seqno);
//  printf("\tsysTime=%lu %lu\n", info.sysTime.tv_sec, info.sysTime.tv_usec);
  if(logT != NULL)
  {
    info.index = logT->getIndex();
//    printf("\tindex=%d\n", info.index);
  }

  fwrite(&info, sizeof(struct QMPacketInfo), 1, outfile);

  if(slaveRcvr != NULL)
  {
    slaveRcvr->recv(pb);
    // assume that the slave will release the packet buffer
  }
  else
  {
    // important!!  RTP data packets must be released
    pb->release();
  }
}

