# layer.tcl --
#
#       A layered web cache object.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#
# A layered web cache object. It does cooperative multi-resolution media
# caching. In the current implementation, the only layer-able object is
# jpeg image, since we have extended the progressive jpeg library. But
# the code should be organized enough so it is easy to extend this to
# video and audio (or other data types) in the future.
#

Class WebCacheApplication/Layer -superclass WebCacheApplication -configuration {
	layersDir /tmp/
}

#
# The constructor.
#
WebCacheApplication/Layer public init { argv } {
	$self next $argv

	# directory to store temporary layers or blocks
	$self instvar layers_dir_
	$self create_dir [$self get_option layersDir]
	#set layers_dir_  [glob [$self get_option layersDir]]
	set layers_dir_  [glob [$self get_option cacheDir]]

	# progressive jpeg
	# we only use 5 recoding levels: 10 (full), 8, 6, 3, 1
	# this table given an existing level tells you what level
	# the image should be reduced to.
	$self instvar pjpeg_recoding_table_
	set pjpeg_recoding_table(10) 8
	set pjpeg_recoding_table(9) 8
	set pjpeg_recoding_table(8) 6
	set pjpeg_recoding_table(7) 6
	set pjpeg_recoding_table(6) 3
	set pjpeg_recoding_table(5) 3
	set pjpeg_recoding_table(4) 3
	set pjpeg_recoding_table(3) 1
	set pjpeg_recoding_table(2) 1
}

WebCacheApplication/Layer private init_argv { argv } {
	set o [$self options]

	$o register_option -cacheDir cacheDir
	$o register_option -cacheSize cacheSize
	$o register_option -layersDir layersDir

	return [$o parse_args $argv]
}


#
# Called by srm if another cache needs to recover the specified data.
# The difference here is we give back a layer of layer-able object
# instead of the whole object.
#
WebCacheApplication/Layer public read_data { source cid seqno } {
	$self instvar index_ layers_dir_

	set url [$self cid_2_name $source $cid]
	if { $url == "" || ![info exists index_($url)] } {
		return ""
	}

	# check whether url is a layer-able object
	# pass to parent method if not
	set data_type [$self is_layerable $url]
	if { "$data_type" == "" } {
		$self next $source $cid $seqno
		return
	}

	# progressive jpeg
	if { "$data_type" == "jpeg" } {
		set num_layers [jpeg_is_progressive $index_($url)]

		if { $num_layers > $seqno } {
			set layer [expr $seqno + 1]
			set fn [file join $layers_dir_ tmp[clock clicks]]

			jpeg_getrange_progressive $index_($url) $fn $layer $layer

			return $fn
		} else {
			return ""
		}
	}
}


#
# Called by the cache control after winning the response timer war. The
# difference here is if the url is a layer-able object, we send out the
# object layer by layer. This means we allocate a new adu to each layer,
# but within the same container.
#
WebCacheApplication/Layer public send_data { url } {
	$self instvar sockets_ source_ index_ cid_names_ proxy_ layers_dir_

	puts "layer: send_data $url"

	# check whether url is a layer-able object
	# pass to parent method if not
	set data_type [$self is_layerable $url]
	if { "$data_type" == "" } {
		$self next $url
		return
	}

	# check if stored on disk at all
	if [info exists index_($url)] {
		# allocate a cid for this url if one does not exist
		set p [split [$self name_2_cid $url] ,]
		set source [lindex $p 0]
		set cid [lindex $p 1]
		if { $cid == "" || $source != $source_} {
			set cid [srm_calloc $source_ 0 $url]
			set cid_names_($source_,$cid) $url
		}

		# send the data to the session, while allocating
		# increasing adu sequecne number to each layer.
		# for now, the only data type is progressive jpeg
		if { "$data_type" == "jpeg" } {
			puts "   filename: $index_($url)"

			set num_layers [jpeg_is_progressive $index_($url)]

			# change from baseline jpeg to progressive jpeg
			# this means we have 10 layers of enhancements
			if { $num_layers == 0 } {
				set fn [file join $layers_dir_ tmp[clock clicks]]
				jpeg_tran_progressive $index_($url) $fn
				file rename -force $fn $index_($url)

				set num_layers 10
			}

			for { set i 0 } { $i < $num_layers } { incr i } {
				# create a temp file for this layer
				set fn [file join $layers_dir_ tmp[clock clicks]]

				# get the layer
				jpeg_getrange_progressive $index_($url) $fn $i $i

				# read it from the file
				set f [open $fn]
				fconfigure $f -translation binary
				set blk ""
				while { ![eof $f] } {
					append blk [read $f 4096]
				}
				close $f

				# remove this temp file
				file delete $fn

				# send the layer using srm
				srm_send $source_ $cid $i $blk
			}
		}

		# callback to proxy to hand data to browser
		if [info exists sockets_($url)] {
			$proxy_ done_fetch $url $sockets_($url) $index_($url)
			unset sockets_($url)

			pust 9
		}
	} else {
		# need to fetch data from origin server
		$self fetch $url
	}
}


#
# Called when received data from the cache session. The difference here
# is the cache can choose not to store the whole object if it is
# layer-able. Also, the data is not broken up into aduss or layers if it
# is layer-able. Assume that a layer-able object is already broken into
# layers and transported layer-by-layer with each adu in a container
# representing a layer.
#
WebCacheApplication/Layer public recv_data { source cid seqno fn } {
	$self instvar proxy_ control_ sockets_ index_ passed_layers_ buffered_layers_ buffered_fns_ layers_dir_

	set url [$self cid_2_name $source $cid]

	puts "layer: recv_data $url $seqno"

	# check whether url is layer-able
	# pass to parent method if not
	set data_type [$self is_layerable $url]
	if { "$data_type" == "" } {
		$self next $source $cid $seqno
		return
	}

	# progressive jpeg
	if { "$data_type" == "jpeg" } {
		# pass to client if the request originiated from here
		# also, check whether to store the layer in cache, if so
		# put it there.

		# an enhancement layer
		if [info exists passed_layers_($url)] {
			set expected_seqno [expr $passed_layers_($url) + 1]
		} else {
			set expected_seqno 0
		}

		if { $seqno == $expected_seqno } {
			if [info exists sockets_($url)] {
				$proxy_ done_partial_fetch $url $sockets_($url) $fn
			}
			if [$self should_store jpeg $url $seqno $fn] {
				$self put $url $fn $seqno
			}
			set passed_layers_($url) $seqno

			set n [expr $seqno + 1]
			while { $n <= 9 } {
				if { [info exists buffered_layers_($url)] && \
				     [set idx [lsearch $buffered_layers_($url) $n]] != -1 } {

					if [$self should_store jpeg $url $n $fn] {
						$self put $url $buffered_fns_($url,$n) $n
					}

					set passed_layers_($url) $n
					set buffered_layers_($url) \
						[lreplace $buffered_layers_($url) $idx $idx]
					unset buffered_fns_($url,$n)

					incr n

					if { $n == 9 } {
						# last layer can close and forget socket
						if [info exists sockets_($url)] {
							$proxy_ done_fetch $url $sockets_($url) $fn
							unset sockets_($url)
						}
						if [info exists passed_layers_($url)] {
							unset passed_layers_($url)
						}
						if [info exists buffered_layers_($url)] {
							unset buffered_layers_($url)
						}

						# tell cache control to cancel timers related for this url
						$control_ cancel_all_timers $url
					} else {
						# still more layers to come, so don't close down socket yet
						if [info exists sockets_($url)] {
							$proxy_ done_partial_fetch $url $sockets_($url) \
								$buffered_fns_($url,$n)
						}
					}
				} else {
					break
				}
			}
		} elseif { $seqno > $expected_seqno } {
			# first check whether this is original data or repair
			# look whether some part of image is already stored on disk

			# <FILL IN>

			puts "layer: buffering data $url $seqno"

			# remember whic layer are buffered
			if ![info exists buffered_layers_($url)] {
				set buffered_layers_($url) ""
			}
			set buffered_layers_($url) [lappend $buffered_layers_($url) $seqno]

			# move the layer into another temp file
			set buffered_fns_($url,$seqno) [file join $layers_dir_ buf[clock clicks]]
			file rename $fn $buffered_fns_($url,$seqno)
		}
	}
}

#
# Algorithm to decide whether to store this layer. Should only be called
# from recv_data. This assumes the url is a layer-able object. The heuristics
# we use are: if the request originated from here, always store everything.
#
WebCacheApplication/Layer private should_store { data_type url seqno fn } {
	$self instvar sockets_ used_ total_

	# check whether request originated from us
	if [info exists sockets_($url)] {
		return 1
	}

	if { "$data_type" == "jpeg" } {
		# only store the first 3 layers of the image
		if { $seqno > 2 } {
			return 0
		} else {
			return 1
		}
	}
}


#
# Uses the function should_store to decide whether to recover. That is,
# if the cache will not store the object, then there is no point to
# recover the losses anyway.
#
WebCacheApplication/Layer public should_recover { source cid sseq eseq } {

	set url [$self cid_2_name $source $cid]
	set data_type [$self is_layerable $url]

	if { $data_type == "" } {
		$self next $source $cid $sseq $eseq
	} else {
		# use the start seqno as hints
		# FIXME foo
		return [$self should_store $data_type $url $sseq foo]
	}
}


#
# Make room in cache by evicting or reducing (or both) objects selected
# by the LRU algorithm. <i>needed</i> denotes the amount of space needed
# in B.
#
WebCacheApplication/Layer public make_room { needed } {
	$self instvar index_ lru_ layers_dir_ used_ pjpeg_recoding_table_

	set cleared 0

	while { $cleared < needed } {

		# find the lru object in cache
		set url [lindex $lru_ 0]
		set fn $index_($url)
		set fs [file size $fn]

		set data_type [$self is_layerable $fn]
		if { $data_type == "" } {
			set cleared [expr $cleared + $fs]

			$self flush_url $url
		} elseif { "$data_type" == "jpeg" } {
			set num_layers [jpeg_is_progressive $fn]

			if { $num_layers == 1 } {
				$self flush_url $url

				set cleared [expr $cleared + $fs]
			} else {
				set reduced_layers $pjpeg_recoding_table_($num_layers)

				set dstfn [file join $layers_dir_ tmp[clock clicks]]
				jpeg_reduce_progressive $fn $dstfn $reduced_layers
				file rename $dstfn $fn

				set cleared [expr $cleared + [expr $fs - [file size $fs]]]
			}
		}
	}
}


#
# Called to initiate the loop to access the cache. The difference is the
# cache needs to determine whether the url is a layer-able object. If
# only the base layers of the object is available, the cache do a repair
# request to get the additional layers.
#
WebCacheApplication/Layer public get { url { socket "" } } {
	$self instvar index_ sockets_ proxy_ wcc_ passed_layers_ buffered_layers_

	puts "layer: get $url"

	# check if stored on disk at all or url is a layer-able object
	# pass to parent method if not
	set data_type [$self is_layerable $url]

	if { "$data_type" == "" || ![info exists index_($url)] } {
		$self next $url $socket
		return
	}

	# do the lru business
	$self push_lru $url

	if { $socket != "" } {
		set sockets_($url) $socket
	}

	# progressive jpeg
	if { "$data_type" == "jpeg" } {
		# check whether whole or partial object is stored
		set num_layers [jpeg_is_progressive $index_($url)]

		if { $num_layers == 0 || $num_layers == 10 } {
			# pass to client if whole
			$proxy_ done_fetch $url $sockets_($url) $index_($url)
			unset sockets_($url)
		} else {
			# pass the existing layers to clients first, and
			# remember what has been passed to keep order
			$proxy_ done_partial_fetch $url $sockets_($url) \
				$index_($url)
			set passed_layers_($url) [expr $num_layers - 1]
			set buffered_layers_($url) ""

			# do downcall for a repair request to get rest
			# of the layers from other caches
			set m [$self name_2_cid $url]
			set source [lindex $m 0]
			set cid [lindex $m 1]

			srm_recover $source $cid $num_layers 9
		}
	}
	# no other data types yet

}


#
# Put contents of url into the cache. If url is a layer-able object,
# we need to enhance the existing layers instead of overwriting them.
# Assume for a layer-able object, it is already layered, and the layers
# are given to this method in-order.
#
WebCacheApplication/Layer public put { url fn { seqno 0 } } {
	$self instvar index_ layers_dir_ used_ total_ wcc_

	puts "layer: put $url"

	# check whether url is a layer-able object
	# pass to parent method if not
	set data_type [$self is_layerable $url]
	if { "$data_type" == "" } {
		$self next $url $fn
		return
	}

	# first check whether there is enough space in cache
	# if not need to make room
	set fs [file size $fn]
	if { [expr $fs + $used_] > $total_ } {
		# need to make room in cache
		$self make_room [expr $fs - ($total_ - $used_)]
	}

	# progressive jpeg
	if { "$data_type" == "jpeg" } {
		# if this is the first layer or a baseline jpeg file,
		# need to create file etc,  so pass to parent method
		if { $seqno == 0 } {
			$self next $url $fn
			return
		}

		# include this enhancement layer into the jpeg image
		set dstfn [file join $layers_dir_ tmp[clock clicks]]
		jpeg_enhance_progressive $index_($url) $dstfn $fn 1
		file rename -force $dstfn $index_($url)
	}
}


WebCacheApplication/Layer private is_layerable { url } {

	# for now, we can only do layers on jpeg images
	set ext [string trimleft [file extension $url] .]
	if [regexp -nocase {jpeg|jpg} $ext] {
		return jpeg
	} else {
		return ""
	}
}
