# al-as.tcl --
#
#       Base class for Announce/Listen Manager implementation in
#       Active Service Framework.    
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/as/al-as.tcl,v 1.10 2002/02/03 04:25:20 lim Exp $


import AnnounceListenManager Timer/Adaptive/ConstBW

#----------------------------------------------------------------------
# Class:
#   AnnounceListenManager/AS
# Description:
#   Base class for AnnounceListenManager in Active Service Framework.
# Members:
#   atype_ -- 
#       Type of this agent
#   agentbytype_ --
#       An array of list of agents, indexed by type (srv, client, hm).
#   aliveid_ --
#       ID to the "after" callback check_alive{}.
#   lastann_ --
#       An array indexed by agent spec, that contains time the last 
#       announcement was made to the corresponding agent.  Time are
#       stored as absolute time (gettimeofday) and as ascii time (
#       clock format).
#   avgdelta_ --
#       An array indexed by agent spec, that contains average time 
#       between announcement from the corresponding agent.
#----------------------------------------------------------------------
Class AnnounceListenManager/AS -superclass AnnounceListenManager


#----------------------------------------------------------------------
# Method:
#   AnnounceListenManager/AS init
# Description:
#   Initialize various members and timer.
#----------------------------------------------------------------------
AnnounceListenManager/AS instproc init { netspec bw atype } {
	#
	# FIXME Seed the random number generator (0 means use
	# a heurstic to derive a seed from system parameters)
	# This belongs elsewhere.
	#
	random 0
	$self next $netspec 1024
	$self instvar atype_
	set atype_ $atype

	$self instvar agentbytype_
	set agentbytype_(srv) ""
	set agentbytype_(client) ""
	set agentbytype_(hm) ""

	# FIXME
	set t [new Timer/Adaptive/ConstBW $bw 3000]
	$t randomize
        $self timer $t

	set o [$self options]
	$o add_default startupWait 60
	$self instvar aliveid_
	set aliveid_ [after [expr [$self get_option startupWait]*1000] "$self check_alive 1"]
}


#----------------------------------------------------------------------
# Method:
#   AnnounceListenManager/AS version
# Description:
#   Return the version string of the current Active Service Framework.
#----------------------------------------------------------------------
AnnounceListenManager/AS proc version {} {
	return 2.0
}


#----------------------------------------------------------------------
# Method:
#   AnnounceListenManager/AS send_announcement
# Description:
#   Construct an announcement, and send it.  Since this method is called
#   periodically, we take this chance to check for liveness in other 
#   agent by calling "check_alive 0" before exiting.
#----------------------------------------------------------------------
AnnounceListenManager/AS public send_announcement {} {
	$self instvar atype_

	set o "ASCP v[$class version]"
	set n $atype_
	set o $o\n$n
	set n [$self agent_instance]
	set o $o\n$n
	set n [$self service_name]
	set o $o\n$n
	set n [$self service_location]
	set o $o\n$n
	set n [$self service_instance]
	set o $o\n$n
	set n [$self ssg_port]
	set o $o\n$n

	set n [$self agent_data]
	set o $o\n$n
	$self announce $o

	$self check_alive 0
}


#----------------------------------------------------------------------
# Method:
#   AnnounceListenManager/AS announce_death
# Description:
#   Construct a BYE announcement and announce it.
#----------------------------------------------------------------------
AnnounceListenManager/AS instproc announce_death {} {
	$self instvar id1_ id2_ atype_

	set o "ASCP v[AnnounceListenManager/AS version]"
	set n $atype_
	set o $o\n$n
	set n [$self agent_instance]
	set o $o\n$n
	set n bye
	set o $o\n$n
	set n -
	set o $o\n$n
	set n -
	set o $o\n$n
	set n -
	set o $o\n$n

	$self announce $o
}


#----------------------------------------------------------------------
# Method:
#   AnnounceListenManager/AS agent_instance
# Description:
#   Return a string that represent an instance of the current agent.
#----------------------------------------------------------------------
AnnounceListenManager/AS public agent_instance {} {
	return "[pid]@[lookup_host_name [localaddr]]"
}


#----------------------------------------------------------------------
# Method:
#   AnnounceListenManager/AS agent_data
# Description:
#   Subclass should overwrite this to return agent specific data.
#----------------------------------------------------------------------
AnnounceListenManager/AS public agent_data {} {
	return ""
}


#----------------------------------------------------------------------
# Method:
#   AnnounceListenManager/AS ssg_port
# Description:
#   Subclass should overwrite this to return port number to a soft state
#   gateway.
#----------------------------------------------------------------------
AnnounceListenManager/AS public ssg_port {} {
	return "-"
}

#AnnounceListenManager/AS instproc announce_bw { bw } {
#	$self instvar timer_
#	$timer_ set bw_ $bw
#}


#----------------------------------------------------------------------
# Method:
#   AnnounceListenManager/AS service_location
# Description:
#   Subclass should overwrite this to return the service location.
#----------------------------------------------------------------------
AnnounceListenManager/AS instproc service_location {} {
	return "-"
}


#----------------------------------------------------------------------
# Method:
#   AnnounceListenManager/AS destroy
# Description:
#   Cleanup "after" callback.
#----------------------------------------------------------------------
AnnounceListenManager/AS instproc destroy {} {
	$self instvar aliveid_
	after cancel $aliveid_

	$self next
}


#----------------------------------------------------------------------
# Method:
#   AnnounceListenManager/AS recv_announcement
# Description:
#   Called when an announcement is received.  The announcement is parsed
#   and special cases are handled (DEATH, BYE, wrong version).  The method 
#   "recv_msg {}" is then called to process other general messages.
# Arguments:
#   addr -- The address where this announcement came from.
#   port -- The port number where this announcement came from.
#   data -- The data contained in this announcement.
#   size -- The length of data in bytes.
#----------------------------------------------------------------------
AnnounceListenManager/AS instproc recv_announcement { addr port data size } {
	$self instvar lastann_ sdp_ agentbytype_ agenttab_ atype_
        set t [$self get_timer]
	$t sample_size $size
#puts ""
#puts "$self ($class): recv {$data}"
#puts ""
	set o [split $data \n]
	if { [lindex $o 0] != "ASCP v[$class version]" } {
		# FIXME
		set msg "$self ($class): received non-ASCP v[$class version] announcement from $addr."
		if { $atype_ == "hm" } {
			$self instvar agent_
			$agent_ log $msg
		} else {
			puts stderr $msg
		}

 		return
	}
	set atype [lindex $o 1]
	set aspec [lindex $o 2]
	set srv_name [lindex $o 3]
	set srv_loc [lindex $o 4]
	set srv_inst [lindex $o 5]
	set ssg_port [lindex $o 6]
	set ad [join [lrange $o 7 end] \n]

	# Special case death packet.
	if { $srv_name == "DEATH" } {
		set msg "Received death packet from $aspec at $addr - exiting."
		if { $srv_loc == $atype_ } {
			if { $atype_ == "hm" } {
				$self instvar agent_
				$agent_ log $msg
			} else {
				puts stderr $msg
			}
			$self announce_death
			exit 0
		}
		$self recv_msg $atype $aspec $addr DEATH $srv_loc \
			$srv_inst $ssg_port "$ad"
		return
	}
	# Synchronous bye
	if { $srv_name == "bye" } {
		$self delete_agent $aspec
		return
	}

	if ![info exists agenttab_($aspec)] {
		# new agent
		$self instvar avgdelta_
		$self register $atype $aspec $addr $srv_name $srv_inst "$ad"
	        $t incr_nsrcs
		set timeout [$self get_option startupWait]
		set avgdelta_($aspec) [expr $timeout / 8]
		lappend agentbytype_($atype) $aspec
	} else {
		set now [gettimeofday]
		set delta [expr $now - $lastann_($aspec,abs)]
		$self instvar avgdelta_
		set avgdelta_($aspec) \
				[expr 0.875*$avgdelta_($aspec)+0.125*$delta]

	}
	set agenttab_($aspec) "$addr {$ad} $atype $srv_name $srv_inst"
	set lastann_($aspec,abs) [gettimeofday]
	set lastann_($aspec,ascii) [gettimeofday ascii]
	$self recv_msg $atype $aspec $addr $srv_name $srv_loc $srv_inst \
			$ssg_port "$ad"
}

AnnounceListenManager/AS instproc advance_timers { delta } {
	$self instvar lastann_ agenttab_ avgdelta_
	set aspecs [array names agenttab_]
	foreach aspec $aspecs {
		set lastann_($aspec,abs) [expr $lastann_($aspec,abs)+$delta]
	}
}


#----------------------------------------------------------------------
# Method:
#   AnnounceListenManager/AS check_alive
# Description:
#   Go through all agents and delete agents that hasn't been announcing
#   for a while.  
# Arguments:
#   timer -- 
#       Specify if this is a one time call (timer == 0), or we are gonna
#       doing this periodically (timer == 1)
#----------------------------------------------------------------------
AnnounceListenManager/AS instproc check_alive { timer } {
	$self instvar lastann_ agenttab_ avgdelta_

	set now [gettimeofday]
	set aspecs [array names agenttab_]
	foreach aspec $aspecs {
		set lastann $lastann_($aspec,abs)
		set avgdelta $avgdelta_($aspec)
#puts "		set avgdelta $avgdelta_($aspec)"
#puts "		set delta [expr $now - $lastann]"
		set delta [expr $now - $lastann]
		if { $delta > 8 * $avgdelta } {
			$self delete_agent $aspec
		}
	}
	$self instvar aliveid_
	if { $timer } {
		set t [expr [$self get_option startupWait]*1000]
		set aliveid_ [after $t "$self check_alive 1"]
	}
}


#----------------------------------------------------------------------
# Method:
#   AnnounceListenManager/AS delete_agent
# Description:
#   Assume that the agent specified by $aspec is dead.  Clean up all
#   states corresponding to that agent.
# Arguments:
#   aspec -- 
#       Specification to the agent to be deleted.
#----------------------------------------------------------------------
AnnounceListenManager/AS instproc delete_agent { aspec } {
 	$self instvar agentbytype_ agenttab_ lastann_ avgdelta_
	if ![info exists agenttab_($aspec)] {
		# puts "hm: agent $aspec doesn't exist!"
		return
	}
	set a $agenttab_($aspec)

	set addr [lindex $a 0]
	set ad [lindex $a 1]
	set atype [lindex $a 2]
	set srv_name [lindex $a 3]
	set srv_inst [lindex $a 4]

	unset agenttab_($aspec)
	unset lastann_($aspec,abs)
	unset lastann_($aspec,ascii)
	unset avgdelta_($aspec)

	set t $agentbytype_($atype)
	set i [lsearch -exact $t $aspec]
	set agentbytype_($atype) [lreplace $t $i $i]

	[$self get_timer] incr_nsrcs -1

	$self unregister $atype $aspec $addr $srv_name $srv_inst "$ad"
}


#----------------------------------------------------------------------
# Method:
#   AnnounceListenManager/AS agenttab
# Description:
#   Return the agent matching specification $aspec, or return "" if
#   no such agent can be found.
# Arguments:
#   aspec -- 
#       Specification to the agent to be retrieve.
#----------------------------------------------------------------------
AnnounceListenManager/AS instproc agenttab aspec {
	$self instvar agenttab_
	if [info exists agenttab_($aspec)] {
		return $agenttab_($aspec)
	}
	return ""
}

#AnnounceListenManager/AS instproc send_announcement {} {
#	$self next
#	$self check_alive 0
#}
