/*
 * serial.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] = "$Header: /usr/mash/src/repository/mash/mash-1/misc/serial.cc,v 1.14 2002/02/03 04:11:42 lim Exp $";
#endif

#include "tclcl.h"
#include "iohandler.h"
#include <stdlib.h>
#include <ctype.h>
#include <sys/fcntl.h>
#include <sys/termios.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <sys/time.h>


#define PARITY_NONE 0
#define PARITY_EVEN 1
#define PARITY_ODD  2

// FIXME - ErikM - I don't think this file is used anymore.  There is now
//   mash/tcl/serial/serial.tcl which implements this I believe.
//   this file shouldn't show up in the Makefile

/*
 * <otcl> Class SerialChannel
 *
 * This module implements serial communication for controlling devices.
 * It implements a mapping allowing text-only encodings of otherwise
 * binary (but non-NULL) data to be sent from tcl to the line and
 * from the line back to tcl.
 *
 * Note that the input encoding and out put encoding are different.
 */
class SerialChannel : public IOHandler, public TclObject {
  public:
    SerialChannel(u_int maxReplyLen = 1024);
    virtual ~SerialChannel();
	int device(int argc, const char * const *argv);
	int baud(int argc, const char * const *argv);
	int stopBits(int argc, const char * const *argv);
	int byteBits(int argc, const char * const *argv);
	int parity(int argc, const char * const *argv);
	int getReply(int argc, const char * const *argv);
	int clearReply(int argc, const char * const *argv);
	int open(int argc, const char * const *argv);
	int close(int argc, const char * const *argv);
    int send(int argc, const char * const *argv);
  private:
	virtual void dispatch(int mask);
	int open();
	int close();
	int init(int argc, const char*const* argv);
  protected:
    virtual int send(const char * formattedMsg);
	virtual int sendBinary(const u_char * binMsg, const int in);
    virtual int receive();
	int myfd_;  /* avoid conflict with fd_ in IOHandler (?) */
	int is_open_;
    char device_[512];  /* assume it isn't a really long path! */
    int baud_;
    int stopBits_;
    int byteBits_;
    int parity_;
	char *reply_;   /* last read result */
	u_int maxReplyLen_;  /* references to maxReplyLen_*5 are due to padding */
};

DEFINE_OTCL_CLASS(SerialChannel, "SerialChannel") {
	INSTPROC_PUBLIC(device);
	INSTPROC_PUBLIC(baud);
	INSTPROC_PUBLIC(stopBits);
	INSTPROC_PUBLIC(byteBits);
	INSTPROC_PUBLIC(parity);
	INSTPROC_PUBLIC(open);
	INSTPROC_PUBLIC(close);
	INSTPROC_PRIVATE(send);
	/*
	 * <otcl> SerialChannel private getReply {}
	 * Get the reply buffer
	 */
	INSTPROC_PRIVATE(getReply);
	/*
	 * <otcl> SerialChannel private clearReply {}
	 * Clear the reply buffer
	 */
	INSTPROC_PRIVATE(clearReply);
}


/*
 * create SerialChannel and set *most* the attributes to defaults.
 *   FIXME -- need to allow for setting of maxReplyLen_
 */
SerialChannel::SerialChannel(u_int maxReplyLen = 1024) :
	 myfd_(-1), is_open_(0), baud_(B9600), stopBits_(1), byteBits_(8),
	 parity_(PARITY_NONE), maxReplyLen_(maxReplyLen)
{
    strcpy(device_, "/dev/cuaa0");
	reply_ = new char[maxReplyLen_ * 5];
	/* (x5) to leave room for maximum *encoded* version in result */
	bzero(reply_, maxReplyLen_ * 5);

	/* bind(c++ inst var, "otcl instvar");   */
	/* ...    rather than all those commands */
}

SerialChannel::~SerialChannel()
{
	close();
	delete[] reply_;
}

/*
 * set the rest of the defaults and hadle any args
 */
int SerialChannel::init(int argc, const char*const* argv) {

	BEGIN_PARSE_ARGS(argc, argv);
	ARG_DEFAULT(maxReplyLen_, 1024);
	END_PARSE_ARGS;

	return(TCL_OK);
}

int
SerialChannel::device(int argc, const char*const* argv) {
	const char *d;

	if (argc == 3) {
		BEGIN_PARSE_ARGS(argc, argv);
		ARG(d);
		END_PARSE_ARGS;
		strcpy(device_, d);  /* Should check if port already in use */
		return(TCL_OK);
	} else {
		BEGIN_PARSE_ARGS(argc, argv);
		END_PARSE_ARGS;
	    Tcl::instance().resultf("%s", device_);
	    return(TCL_OK);
	}
}

int
SerialChannel::baud(int argc, const char*const* argv) {
	int baud;

	if (argc == 3) {
		BEGIN_PARSE_ARGS(argc, argv);
		ARG(baud);
		END_PARSE_ARGS;
		switch (baud) {
		  case 0:       	baud_ = B0;      	 break;
		  case 50:      	baud_ = B50;     	 break;
		  case 75:      	baud_ = B75;     	 break;
		  case 110:     	baud_ = B110;        break;
		  case 134:     	baud_ = B134;        break;
		  case 150:     	baud_ = B150;        break;
		  case 200:     	baud_ = B200;        break;
		  case 300:     	baud_ = B300;        break;
		  case 600:     	baud_ = B600;        break;
		  case 1200:        baud_ = B1200;       break;
		  case 1800:        baud_ = B1800;       break;
		  case 2400:        baud_ = B2400;       break;
		  case 4800:        baud_ = B4800;       break;
		  case 9600:        baud_ = B9600;       break;
		  case 19200:       baud_ = B19200;      break;
		  case 38400:       baud_ = B38400;      break;
		  default:
		  Tcl::instance().resultf("bad baud rate: %d", baud);
			return(TCL_ERROR);
		}
		return(TCL_OK);
	} else {
		BEGIN_PARSE_ARGS(argc, argv);
		END_PARSE_ARGS;
	    Tcl::instance().resultf("%d", baud_);
	    return(TCL_OK);
	}
}

int
SerialChannel::stopBits(int argc, const char*const* argv) {
	int stopBits;

	if (argc == 3) {
		BEGIN_PARSE_ARGS(argc, argv);
		ARG(stopBits);
		END_PARSE_ARGS;
		if (stopBits < 0 || stopBits > 2) {
		  Tcl::instance().resultf("bad stopBits: %d", stopBits);
			return(TCL_ERROR);
		}
		stopBits_ = stopBits;
		return(TCL_OK);
	} else {
		BEGIN_PARSE_ARGS(argc, argv);
		END_PARSE_ARGS;
	    Tcl::instance().resultf("%d", stopBits_);
	    return(TCL_OK);
	}
}

int
SerialChannel::byteBits(int argc, const char*const* argv) {
	int byteBits;

	if (argc == 3) {
		BEGIN_PARSE_ARGS(argc, argv);
		ARG(byteBits);
		END_PARSE_ARGS;
		if (byteBits < 5 || byteBits > 8) {
		  Tcl::instance().resultf("bad byteBits: %d", byteBits);
			return(TCL_ERROR);
		}
		byteBits_ = byteBits;
		return(TCL_OK);
	} else {
		BEGIN_PARSE_ARGS(argc, argv);
		END_PARSE_ARGS;
	    Tcl::instance().resultf("%d", byteBits_);
	    return(TCL_OK);
	}
}

int
SerialChannel::parity(int argc, const char*const* argv) {
	const char *p;

	if (argc == 3) {
		BEGIN_PARSE_ARGS(argc, argv);
		ARG(p);
		END_PARSE_ARGS;
		if (strcmp(p, "none") == 0) {
			parity_ = PARITY_NONE;
		} else if (strcmp(p, "even") == 0) {
			parity_ = PARITY_EVEN;
		} else if (strcmp(p, "odd") == 0) {
			parity_ = PARITY_ODD;
		} else {
		  Tcl::instance().resultf("bad parity: %s", p);
		  return TCL_ERROR;
      }
	} else {
		BEGIN_PARSE_ARGS(argc, argv);
		END_PARSE_ARGS;
		if (parity_ == PARITY_NONE) {
		  Tcl::instance().resultf("none");
		} else if (parity_ == PARITY_EVEN) {
		  Tcl::instance().resultf("even");
		} else if (parity_ == PARITY_ODD) {
		  Tcl::instance().resultf("odd");
		}
	}
	return TCL_OK;
}

int
SerialChannel::getReply(int argc, const char*const* argv) {

	BEGIN_PARSE_ARGS(argc, argv);
	END_PARSE_ARGS;
    Tcl::instance().resultf("%s", reply_);
	return TCL_OK;

}

int
SerialChannel::clearReply(int argc, const char*const* argv) {
	BEGIN_PARSE_ARGS(argc, argv);
	END_PARSE_ARGS;
	bzero(reply_, maxReplyLen_ * 5);
    return TCL_OK;
}

int
SerialChannel::close(int argc, const char * const *argv)
{
    BEGIN_PARSE_ARGS(argc, argv);
    END_PARSE_ARGS;

    close();
    return TCL_OK;
}

int
SerialChannel::close()
{
	if (is_open_ == 1) {
		is_open_ = 0;
		unlink();
		int s = ::close(myfd_);
		myfd_ = -1;
		return (s);
	}
	return -1;
}

int
SerialChannel::open(int argc, const char*const* argv) {
	BEGIN_PARSE_ARGS(argc, argv);
	END_PARSE_ARGS;
	open();
    return TCL_OK;
}


/*
 * Open the device
 */
int
SerialChannel::open()
{
	if (is_open_ == 1)
	  return TCL_OK;

	Tcl& tcl = Tcl::instance();
    struct termios terminfo;

    myfd_ = ::open(device_, O_RDWR | O_NOCTTY | O_NONBLOCK, 0);
    if (myfd_ < 0) {
		sprintf(tcl.buffer(),"unable to open device \"%s\"", device_);
        tcl.result(tcl.buffer());
        return TCL_ERROR;
	}

    errno = 0;
    tcgetattr(myfd_, &terminfo);
    if (errno != 0) {
		sprintf(tcl.buffer(), "unable to get attributes for device \"%s\"",
				device_);
		tcl.result(tcl.buffer());
        return TCL_ERROR;
    }

    cfsetospeed(&terminfo, baud_);
    if (errno != 0) {
		sprintf(tcl.buffer(),
				"unable to set output baud rate for device \"%s\"",
				device_);
		tcl.result(tcl.buffer());
        return TCL_ERROR;
    }

    cfsetispeed(&terminfo, baud_);
    if (errno != 0) {
		sprintf(tcl.buffer(),
				"unable to set input baud rate for device \"%s\"",
				device_);
		tcl.result(tcl.buffer());
        return TCL_ERROR;
    }

	/*     terminfo.c_iflag &= ~(INLCR | IGNCR | ICRNL | IUCLC | */
	/*                        IXON | IXANY | IXOFF); */
    terminfo.c_iflag &= ~(INLCR | IGNCR | ICRNL | IXON | IXANY | IXOFF);
    terminfo.c_oflag &= ~OPOST;

    if (stopBits_ == 2) {
        terminfo.c_cflag |= CSTOPB;
    } else {
        terminfo.c_cflag &= ~(CSTOPB);
    }
    terminfo.c_cflag &= ~(CSIZE);

    /* terminfo.c_cflag |= CRTSCTS; */

    if (parity_ == PARITY_NONE) {
        terminfo.c_cflag &= ~(PARENB);
    } else {
        terminfo.c_cflag |= PARENB;
        if (parity_ == PARITY_ODD) {
            terminfo.c_cflag |= PARODD;
        } else {
            terminfo.c_cflag &= ~(PARODD);
        }
    }

    switch (byteBits_) {
      case 5:   terminfo.c_cflag |= CS5;    break;
      case 6:   terminfo.c_cflag |= CS6;    break;
      case 7:   terminfo.c_cflag |= CS7;    break;
      case 8:   terminfo.c_cflag |= CS8;    break;
    }

    terminfo.c_cflag |= CREAD | CLOCAL;
    terminfo.c_lflag &= ~(ISIG | ICANON |
						  ECHO | ECHOE | ECHOK | ECHONL |
						  ECHOCTL | ECHOPRT | ECHOKE | IEXTEN);


    tcsetattr(myfd_, TCSANOW, &terminfo);
    if (errno != 0) {
		sprintf(tcl.buffer(),
				"unable to set attributes for device \"%s\"", device_);
        tcl.result(tcl.buffer());
		return TCL_ERROR;
    }

	/* sprintf(tcl.buffer(), "opened fd = %d", myfd_);
	 * tcl.result(tcl.buffer());
	 */

	link(myfd_, TCL_READABLE);
	is_open_ = 1;
	return TCL_OK;
}

void
SerialChannel::dispatch(int mask)
{
	// printf("SerialChannel::dispatch()  called\n");
	/* ignore mask since it is assumed that only reads are caught */
	if (mask != TCL_READABLE) {
		printf("expected TCL_READABLE, got %d\n", mask);
	} else
	  receive();
}


int
SerialChannel::send(int argc, const char*const* argv) {
	const char *msg;

    BEGIN_PARSE_ARGS(argc, argv);
    ARG(msg);
	END_PARSE_ARGS;

	send(msg);
    return TCL_OK;
}


/* Send message, encoded in ASCII form for use from tcl.
 * Format is space-separated bytes, where '0xZZ'
 * will be converted to the equivalent binary data and other data
 * will passed as is.  "  " (three spaces) is a single space character,
 * 0x00 is NULL, and "0 x Z Z" will become the literals "0xZZ", so no
 * explicit escape character is needed. e.g.
 *  "M A S H   0 x A f 0x0A 0x0d" --> "MASH 0xAf\n\r"
 */
int
SerialChannel::send(const char * formattedMsg)
{
	int lenF = strlen(formattedMsg);
	int lenB = 0;               /* lenB = binaryMsg len */
	int i;	                    /* index into formattedMsg */
	u_char *binMsg = new u_char[(lenF/2)+1];
	int binaryNum;
	char binaryNumString[5];
	char *f = new char[lenF+1]; /* copy of formattedMsg -w- extra char
								   at end to simplify parse */
	strcpy(f, formattedMsg);
	f[lenF] = ' ';


	for (i=0; i < lenF; /*NULL*/) {
		if (f[i] != '0') {
			/* ascii data, dump directly */
			binMsg[lenB] = f[i];
			lenB++;
			i+=2;
		} else {
			/* check for 'x' vs. ' ' in next char */
			if	(f[i+1] == ' ') {
				/* next is space char, dump as literal '0' */
				binMsg[lenB] = f[i];
				lenB++;
				i+=2;
			} else {
				/* better be "0x" ... */
				if (f[i+1] != 'x') {
					printf("Impropery formatted msg data\n");
					return -1;
				}
				strncpy(binaryNumString, &(f[i]), 4);
				binaryNumString[4] = 0;
				/* printf("binStr = %s\n", binaryNumString); */
				binaryNum = strtol(binaryNumString, (char **) NULL, 16);
				binMsg[lenB] = binaryNum;
				lenB++;
				i+=5;
			}
		}
	}
	i = 0;
	if (lenB > 0)
	 i = sendBinary(binMsg, lenB);

	usleep(100000);

	receive();

	delete[] binMsg;
	delete[] f;

	return i;
}

int
SerialChannel::sendBinary(const u_char * binaryMsg, int len)
{
#if 0
	int i;
	printf("write: ");
	for (i = 0; i < len; i++) {
		printf("0x%.2x ", (int) binaryMsg[i]);
		if (write(myfd_, &binaryMsg[i], 1) != 1) {
			printf("Error writing to serial channel\n");
		}
	}
	printf("\n");
	return 0;
#else
	return write(myfd_, binaryMsg, len);
#endif
}

/*
 * Receive from the channel, and cast into an encoded form for
 * passing to tcl: integers indicating the byte value
 */
int
SerialChannel::receive()
{
    fd_set readfds;
    struct timeval timeout;
    int len = 0, i;
    u_char *tmpBuf = new u_char[maxReplyLen_];
	char *tmpNumHolder  = new char[6];

    bzero(tmpBuf, maxReplyLen_);
    // bzero(reply_, maxReplyLen_ * 5);

    FD_ZERO(&readfds);
    FD_SET(myfd_, &readfds);
    timeout.tv_sec = 3; /* FIXME */
    timeout.tv_usec = 0;
    select(myfd_ + 1, &readfds, NULL, NULL, &timeout);
    if (FD_ISSET(myfd_, &readfds)) {
		/* avoid buffer overflow by blowing away "old" (hopefully) data */
		if (strlen(reply_) >= (9*(maxReplyLen_ * 5)/10))
		  bzero(reply_, maxReplyLen_ * 5);
        do {
            len = read(myfd_, tmpBuf, sizeof(tmpBuf));

			/* convert binary data to integers */
            for (i = 0; i < len; i++) {
				bzero(tmpNumHolder, 6);
#if 0
                if (isascii(tmpBuf[i])) {
                    sprintf(tmpNumHolder, "%c ", tmpBuf[i]);
                } else {
                    sprintf(tmpNumHolder, "0x%.2x ", tmpBuf[i]);
                }
#else
				sprintf(tmpNumHolder, "%d ", tmpBuf[i]);
#endif
                strcat(reply_, tmpNumHolder);
				printf("%c", tmpBuf[i]);
            }
#if 0
            for (i = 0; i < len; i++) {
                if (isascii(reply_[i])) {
                    printf("%c ", reply_[i]);
                } else {
                    printf("0x%.2x ", reply_[i]);
                }
            }
            printf("\n");
#endif
        } while (len > 0);
    } else {
        /* printf("No response\n"); */
    }

    // response_ = waitingForResponse_ = 0;

	delete[] tmpBuf;
	delete[] tmpNumHolder;
    return len;
}

