#include "../config.h"
#include <sys/types.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <usb.h>
#include "base.h"
#include "libnjb.h"
#include "njbusb.h"
#include "ioutil.h"
#include "usb_io.h"
#include "njb_error.h"

extern int __sub_depth;

ssize_t usb_pipe_write (njb_t *njb, void *buf, size_t nbytes)
{
	ssize_t bwritten = -1;

	/* Used for all libnjb enabled platforms. 
	 * This section might be the source of timeout problems so
	 * it is currently being tested. Also see pipe_read below. */

	int usb_timeout = 10 * nbytes; /* that's 5ms / byte */
	int retransmit = 10; /* Try ten times! (That should do it...) */
	usb_dev_handle **dev = &njb->bout;
	int usb_mode;

	/* This will probably go for NJB ZEN USB 2.0 and NJV ZEN NX too */
	if (USB20_DEVICE(njb->device_type)) {
	  usb_mode = 0x01|UT_WRITE; /* EP marker for USB 2.0 */
	} else {
	  usb_mode = 0x02|UT_WRITE; /* EP marker for USB 1.1 */
	}

	/* Set a timeout floor */
	if (usb_timeout < USBTIMEOUT)
	  usb_timeout = USBTIMEOUT;

	while (retransmit > 0) {
	  bwritten= usb_bulk_write(*dev, usb_mode, buf, nbytes, 
				   usb_timeout);
	  if ( bwritten < 0 )
	    retransmit--;
	  else
	    break;
	}
	if (retransmit == 0) {
	  NJB_RERROR("usb_bulk_write", -1);
	  return -1;
	}

	if ( njb_debug(DD_USBBLK|DD_USBBLKLIM) ) {
		size_t bytes= ( njb_debug(DD_USBBLK) ) ? nbytes : 16;

		fprintf(stderr, "Bulk >>\n");
		data_dump_ascii(stderr, buf, bytes, 0);
		fprintf(stderr, "\n");
	}

	return bwritten;
}

ssize_t usb_pipe_read (njb_t *njb, void *buf, size_t nbytes)
{
	ssize_t bread = 0;

	/* Used for all libnjb enabled platforms. 
	 * This section might be the source of timeout problems so
	 * it is currently being tested. Also see pipe_write above. */
	int usb_timeout = 10 * nbytes; /* that's 10ms / byte */
	int retransmit = 10; /* Try ten times! (That should do it...) */
	/* Needs to be 0x82 for all boxes, which is what it will be */
	usb_dev_handle **dev = &njb->bin;
	int usb_mode = NJB_BULK_EP|UT_READ;
	

	/* Set a timeout floor */
	if (usb_timeout < USBTIMEOUT)
	  usb_timeout = USBTIMEOUT;
	
	while (retransmit > 0) {
	  bread= usb_bulk_read(*dev, usb_mode, buf, nbytes, 
			       usb_timeout);
	  /* This should be changed to (bread < nbytes) asap, but needs
	   * an NJB3 to test it, it cancels out short reads if I set
	   * it to that, so these must first be avoided in all NJB3
	   * code by properly checking length etc. */
	  if ( bread < 0 )
	    retransmit--;
	  else
	    break;
	}
	if ( retransmit == 0 ) {
	  NJB_RERROR("usb_bulk_read", -1);
	  return -1;
	}

	if ( njb_debug(DD_USBBLK|DD_USBBLKLIM) ) {
		size_t bytes= ( njb_debug(DD_USBBLK) ) ? bread : 16;

		fprintf(stderr, "Bulk <<\n");
		data_dump_ascii(stderr, buf, bytes, 0);
		fprintf(stderr, "\n");
	}

	/* return nbytes; ???
	 * Why not returning the number of bytes that were actually
	 * transfered? */
	return bread;
}

int usb_setup (usb_dev_handle **dev, int type, int request, int value,
	int index, int length, void *data)
{
	char setup[8];

	if ( njb_debug(DD_USBCTL) ) {
		memset(setup, 0, 8);
		setup[0]= type;
		setup[1]= request;
		if ( value ) {
			setup[2] = value & 255;
			setup[3] = (value >> 8) & 255;
		}
		if ( index ) {
			setup[4] = index & 255;
			setup[5] = (index >> 8) & 255;
		}
		if ( length ) {
			setup[6] = length & 255;
			setup[7] = (length >> 8) & 255;
		}
	}

	if ( njb_debug(DD_USBCTL) ) {
		fprintf(stderr, "%*sSetup: ", 3*__sub_depth, " ");
		data_dump(stderr, setup, 8);
	}

	if ( usb_control_msg(*dev, type, request, value, index, data, length,
		USBTIMEOUT) < 0 ) {
		NJB_RERROR("ioctl", -1);
		return -1;
	}

	if ( njb_debug(DD_USBCTL) ) {
		if ( length ) {
			fprintf(stderr, "%s", ( (type & UT_READ) == UT_READ ) ?
				"<<" : ">>");
			data_dump_ascii(stderr, data, length, 0);
			fprintf(stderr, "\n");
		}
	}

	return 0;
}

