// =============================================================================
//
//   Copyright (C) 1999-2000 Szymon Stefanek
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviStyleWood"

#define _KVI_STYLE_WOOD_CPP_

#include <qdrawutil.h>
#include <qimage.h>
#include <qpainter.h>

#include "kvi_pushbutton.h"
#include "kvi_style_wood.h"
#include "kvi_style_images_wood.h"

extern QPixmap *wood_back_pix;

/**
 * Wood style: as provided with the Qt library example "themes"
 */
KviWoodStyle::KviWoodStyle()
	: QMotifStyle(true)
{
	// Nothing here
}

KviWoodStyle::~KviWoodStyle()
{
	// Nothing here
}

void KviWoodStyle::polish(QPalette &pal)
{
	// We simply create a nice QColorGroup with a couple of fancy wood
	// pixmaps here and apply to it all widgets.

	QImage img(wood_button_xpm);
	QImage orig = img;
	orig.detach();
	QPixmap button;
	button.convertFromImage(img);

	QPixmap background(wood_polish_xpm);

	int i;
	for( i = 0; i < img.numColors(); i++ ) {
		QRgb rgb = img.color(i);
		QColor c(rgb);
		rgb = c.dark().rgb();
		img.setColor(i, rgb);
	}
	QPixmap mid;
	mid.convertFromImage(img);

	img = orig;
	for(i = 0; i < img.numColors(); i++ ) {
		QRgb rgb = img.color(i);
		QColor c(rgb);
		rgb = c.light().rgb();
		img.setColor(i, rgb);
	}
	QPixmap light;
	light.convertFromImage(img);

	img = orig;
	for( i = 0; i < img.numColors(); i++ ) {
		QRgb rgb = img.color(i);
		QColor c(rgb);
		rgb = c.dark().rgb();
		img.setColor(i, rgb);
	}
	QPixmap dark;
	dark.convertFromImage(img);

	QPalette op(QColor(212, 140, 95));
	QColorGroup nor(
		op.inactive().foreground(),
		QBrush(op.inactive().button(), button),
		QBrush(op.inactive().light(),  light),
		QBrush(op.inactive().dark(),   dark),
		QBrush(op.inactive().mid(),    mid),
		op.inactive().text(),
		Qt::white,
		QColor(236, 182, 120),
		QBrush(op.inactive().background(), background)
	);
	QColorGroup disabled(
		op.disabled().foreground(),
		QBrush(op.disabled().button(), button),
		QBrush(op.disabled().light(),  light),
		op.disabled().dark(),
		QBrush(op.disabled().mid(),    mid),
		op.disabled().text(),
		Qt::white,
		QColor(236, 182, 120),
		QBrush(op.disabled().background(), background)
	);
	QColorGroup active(
		op.active().foreground(),
		QBrush(op.active().button(), button),
		QBrush(op.active().light(),  light),
		op.active().dark(),
		QBrush(op.active().mid(),     mid),
		op.active().text(),
		Qt::white,
		QColor(236, 182, 120),
		QBrush(op.active().background(), background)
	);

	pal.setInactive(nor);
	pal.setDisabled(disabled);
	pal.setActive(active);
}

/**
 * Reimplementation from QStyle.
 * \sa QStyle
 */
void KviWoodStyle::polish(QWidget *w)
{
	// The polish function will set some widgets to transparent mode
	// to get the full benefit from the nice pixmaps in the color
	// group.
	if( w->inherits("QTipLabel")  ) return;
	if( w->inherits("KviMenuBar") ) w->setBackgroundPixmap(*wood_back_pix);
	if( !w->isTopLevel() )
	{
		if( w->autoMask() ) return;
		if( w->inherits("KviChanLabel") ) return;
		if( (w->inherits("KviLabel")    ||
			 w->inherits("QButton")     ||
			 w->inherits("KviComboBox") ||
			 w->inherits("QGroupBox")   ||
			 w->inherits("QSlider")     ||
			 w->inherits("QTabWidget")  ||
			 w->inherits("QTabBar")
			) && !w->backgroundPixmap() )
		{
			w->setAutoMask(true); // This will not work for the labels that show variable text
		}
	}
}

void KviWoodStyle::unPolish(QWidget *w)
{
	// The polish function will set some widgets to transparent mode
	// to get the full benefit from the nice pixmaps in the color
	// group.

	if( w->inherits("QTipLabel")  ) return;
	if( w->inherits("QLCDNumber") ) return;

	if( w->inherits("KviMenuBar") ) {
		w->setBackgroundPixmap(QPixmap());
		w->setBackgroundMode(PaletteButton);
	}

	if( !w->isTopLevel() ) {
		if( !w->autoMask() ) return;
		if( w->inherits("KviChanLabel") ) return;
		if( (w->inherits("KviLabel")    ||
			 w->inherits("QButton")     ||
			 w->inherits("KviComboBox") ||
			 w->inherits("QGroupBox")   ||
			 w->inherits("QSlider")     ||
			 w->inherits("QTabWidget")  ||
			 w->inherits("QTabBar")
			) && !w->backgroundPixmap() )
		{
			w->setAutoMask(false);
		}
	}
}

static void drawroundrect(QPainter *p, QCOORD x, QCOORD y, QCOORD w, QCOORD h, QCOORD d)
{
	int rx = (200 * d) / w;
	int ry = (200 * d) / h;
	p->drawRoundRect(x, y, w, h, rx, ry);
}

void KviWoodStyle::drawButton(
	QPainter *p, int x, int y, int w, int h,
	const QColorGroup &g, bool sunken, const QBrush *fill
)
{
	qDrawShadePanel(p, x, y, w, h, g, sunken, 5);

	QBrush oldBrush = p->brush();
	QPen oldPen = p->pen();
	p->setPen(NoPen);
	p->setBrush(fill ? *fill : (sunken ? g.brush(QColorGroup::Mid) : g.brush(QColorGroup::Button)));
	drawroundrect(p, x + 3, y + 3, w - 6, h - 6, 5);
	p->setBrush(oldBrush);
	p->setPen(g.foreground());
	drawroundrect(p, x, y, w, h, 8);
	p->setPen(oldPen);
}

void KviWoodStyle::drawBevelButton(
	QPainter *p, int x, int y, int w, int h,
	const QColorGroup &g, bool sunken, const QBrush *fill
)
{
	const QRect r(x, y, w, h);
	QMotifStyle::drawPrimitive(QStyle::PE_ButtonBevel, p, r, g);
}

void KviWoodStyle::drawPushButton(KviPushButton *btn, QPainter *p)
{
	QColorGroup g = btn->colorGroup();
	int x1, y1, x2, y2;

	btn->rect().coords(&x1, &y1, &x2, &y2); // Get coordinates

	p->setPen(g.foreground());
	p->setBrush(QBrush(g.button(), NoBrush));

	QBrush fill;
	if( btn->isDown() )
		fill = g.brush(QColorGroup::Mid);
	else if( btn->isOn() )
		fill = QBrush(g.mid(), Dense4Pattern);
	else
		fill = g.brush(QColorGroup::Button);

	if( btn->isDefault() ) {
		QPointArray a;
		a.setPoints(9,
			x1,     y1,     x2,     y1,     x2,     y2,     x1,     y2,
			x1,     y1 + 1, x2 - 1, y1 + 1, x2 - 1, y2 - 1, x1 + 1, y2 - 1,
			x1 + 1, y1 + 1
		);
		p->setPen(Qt::black);
		p->drawPolyline(a);
		x1 += 2;
		y1 += 2;
		x2 -= 2;
		y2 -= 2;
	}

	drawButton(p, x1, y1, x2 - x1 + 1, y2 - y1 + 1, g, btn->isOn() || btn->isDown(), &fill);

	if( btn->isMenuButton() ) {
		int dx = (y1 -y2 - 4) / 3;
		const QRect r(x2 - dx, dx, y1, y2 - y1);
		QMotifStyle::drawPrimitive(QStyle::PE_ArrowDown, p, r, g);
	}

	if( p->brush().style() != NoBrush )
		p->setBrush(NoBrush);
}

void KviWoodStyle::drawPushButtonLabel(KviPushButton *btn, QPainter *p)
{
	QRect r = btn->rect();
	int x, y, w, h;
	r.rect(&x, &y, &w, &h);

	int x1, y1, x2, y2;
	btn->rect().coords(&x1, &y1, &x2, &y2); // Get coordinates
	int dx = 0;
	int dy = 0;
	if( btn->isMenuButton() )
		dx = (y2 - y1) / 3;
	if( dx || dy )
		p->translate(dx, dy);

	x += 2;
	y += 2;
	w -= 4;
	h -= 4;
	QColorGroup g = btn->colorGroup();
	QRect r2(x, y, w, h);
	drawItem(
		p, r2, AlignCenter | ShowPrefix, g, btn->isEnabled(), btn->pixmap(), btn->text(), -1,
		(btn->isDown() || btn->isOn()) ? &btn->colorGroup().brightText() : &btn->colorGroup().buttonText()
	);
	if( dx || dy )
		p->translate(-dx, -dy);
}

QRect KviWoodStyle::buttonRect(int x, int y, int w, int h)
{
	return QRect(x + 3, y + 2, w - 6, h - 4);
}

void KviWoodStyle::drawButtonMask(QPainter *p, int x, int y, int w, int h)
{
	drawroundrect(p, x, y, w, h, 8);
}

#include "m_kvi_style_wood.moc"
