// =============================================================================
//
//      --- kvi_stat.cpp ---
//
//   This is a plugin for KVIrc 2. It allows you to see
//   statistics of your IRC sessions - total words said etc.
//   More features are still to come.
//
//   (c) Krzysztof Godlewski <kristoff@poczta.wprost.pl>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviStat"

#define _KVI_STAT_CPP_

#include <qobjectlist.h>

#define __KVIRC_PLUGIN__

#include "kvi_channel.h"
#include "kvi_input.h"
#include "kvi_locale.h"
#include "kvi_mirccntrl.h"
#include "kvi_stat.h"
#include "kvi_stat_chan.h"
#include "kvi_stat_controller.h"
#include "kvi_stat_options.h"
#include "kvi_stat_systray.h"
#include "kvirc_plugin.h"

KviStatOptions    *g_pStatPluginOptionsWindow = 0;
KviStatController *g_pStatPluginController    = 0;

/**
 * Usage text.
 */
const char *usageText = _i18n_(
	"\nUsage:\nSTATS [option] - see and manipulate your statistics.\n"
	"If used without options it shows number of words for current window type, "
	"and if it is typed in a chan window it shows the number of words said on "
	"this channel.\n"
	"If used with an option it shows:\n"
	"        -h      this help message.\n"
	"        -t      number of words spoken since last stats reset\n"
	"        -c      number of words spoken on channels\n"
	"        -q      number of words spoken on queries\n"
	"        -d      number of words spoken on DCCs\n"
	"        -k      number of kicks given\n"
	"        -b      number of bans set\n"
	"        -w      a window with complete statistics\n"
	"        -j      number of times you have joined the channel\n"
	"        -m      a message to a channel telling how many words you have got :)\n"\
	"        -M      same as -m, but shows number of words on channel you are "
					"currently on (not total)\n"
	"        -o      number of topic changes\n"
	"        -D      detailed stats of current channel\n"
	"        -S      current session stats\n"
	"        -r      session record (words)\n"
	"Other options:\n"
	"        -x str  Delete stats for channel 'str'. 'Str' can be in '#chan' or "
					"'chan' format\n"
	"        -s      Save stats file NOW\n"
	"        -C      Show configuration dialog\n"
	"        -R      Reset stats (after confirmation)\n"
	"See plugin help page for more details\n"
	"This plugin was written by Krzysztof \"Kristoff\" Godlewski "
	"<kristoff@poczta.wprost.pl, krzygod@kki.net.pl>\n"
	"See you on IRCNet!\n"
);

/**
 * Global stuff.
 */
void stat_plugin_processJoinStats(KviStatChan *c, KviWindow *window)
{
	if( window->type() != KVI_WND_TYPE_CHANNEL ) return;

	switch( g_pStatPluginController->joinStatType() ) {
		case ShowFullStatsOnJoin: {
			if( c->joins() == 1 )
				window->output(KVI_OUT_INFO, (_i18n_("You have joined %s once.")), window->caption());
			else
				window->output(KVI_OUT_INFO, (_i18n_("You have joined %s for %u times.")), window->caption(), c->joins());
			window->output(KVI_OUT_INFO, (_i18n_("Full stats for %s:")), window->caption());
			window->output(KVI_OUT_INFO, _CHAR_2_QSTRING(_i18n_("    Words: %d\n    Kicks: %u\n    Bans: %u\n    Topic changes: %u")), c->words(), c->kicks(), c->bans(), c->topics());
		} break;

		case ShowCustomStatsOnJoin: {
			if( g_pStatPluginController->showJoins() ) {
				if( c->joins() == 1 )
					window->output(KVI_OUT_INFO, (_i18n_("You have joined %s once.")), window->caption());
				else
					window->output(KVI_OUT_INFO, (_i18n_("You have joined %s for %u times.")), window->caption(), c->joins());
			}
			if( g_pStatPluginController->showKicks() ||
				g_pStatPluginController->showWords() ||
				g_pStatPluginController->showBans()  ||
				g_pStatPluginController->showTopics() )
			{
				window->output(KVI_OUT_INFO, (_i18n_("Stats for %s:")), window->caption());
			}
			if( g_pStatPluginController->showWords() )
				window->output(KVI_OUT_INFO, (_i18n_("    Words spoken: %d.")), c->words());
			if( g_pStatPluginController->showKicks() )
				window->output(KVI_OUT_INFO, (_i18n_("    Kicks given: %d.")), c->kicks());
			if( g_pStatPluginController->showBans() )
				window->output(KVI_OUT_INFO, (_i18n_("    Bans set: %d.")), c->bans());
			if( g_pStatPluginController->showTopics() )
				window->output(KVI_OUT_INFO, (_i18n_("    Topic changes: %d.")), c->topics());
		} break;
	}
}

bool stat_plugin_hook_addstat(KviPluginCommandStruct *cmd)
{
	int len = 0;
	int numWords = 0;

	if( cmd->params->count() >= 2 )
	{
		QPtrListIterator<KviStr> it(*cmd->params);
		it += 2;    // Omit event and window name
		for( ; it.current(); ++it )
			len += it.current()->len();
		numWords = cmd->params->count() - 2;
	}

	// If the channel is not in stats add it

	KviStatChan *c = g_pStatPluginController->findStatChan(cmd->window->caption());

	switch( cmd->window->type() ) { // Determine the window type
		case KVI_WND_TYPE_CHANNEL: { // Channel window
			g_pStatPluginController->addChansWords(numWords);
			g_pStatPluginController->addChansLetters(len);
			if( !c ) {
				g_pStatPluginController->addTotalJoin();

				KviStatChan *stch = new KviStatChan(cmd->window->caption(), 1, numWords);
				g_pStatPluginController->addChan(stch);

				cmd->window->output(KVI_OUT_INFO, _i18n_("Added %s to stats."), cmd->window->caption());
				break;
			}
			c->addWords(numWords);
		} break;

		case KVI_WND_TYPE_QUERY: { // Query window
			g_pStatPluginController->addQueriesWords(numWords);
			g_pStatPluginController->addQueriesLetters(len);
		} break;

		case KVI_WND_TYPE_CHAT: { // DCC Chat window
			g_pStatPluginController->addDccsWords(numWords);
			g_pStatPluginController->addDccsLetters(len);
		} break;
	}

	g_pStatPluginController->addTotalWords(numWords);
	g_pStatPluginController->addSessionWords(numWords);
	g_pStatPluginController->addTotalLetters(len);

	return false;
}

bool stat_plugin_command_stats(KviPluginCommandStruct *cmd)
{
	QString s;
	KviStatChan *chan = g_pStatPluginController->findStatChan(cmd->window->caption());

	if( cmd->params->count() > 1 ) {
		switch( cmd->params->at(1)->ptr()[1] ) { // Check argument
			case 'C': { // Show config dialog
				g_pStatPluginController->slotShowConfig();
			} break;

			case 'D': { // Detailed stats of current chan
				if( cmd->window->type() == 1 ) { // This is a chan win
					if( chan ) { // Got the chan in stats
						s.sprintf(_i18n_("Stats for %s:\n"), cmd->window->caption());
						cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
						s.sprintf(_i18n_("Joins: %u\nWords: %d\nKicks: %u\n"), chan->joins(), chan->words(), chan->kicks());
						cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
						s.sprintf(_i18n_("Bans: %u\nTopics: %u\n"), chan->bans(), chan->topics());
						cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
					} else { // Do not have it
						s.sprintf(_i18n_("No stats for %s.\n"), cmd->window->caption());
						cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
					}
				} else { // Current window is not a chan window
					if( cmd->window->type() == 0 ) { // @console
						s.sprintf(_i18n_("Total number of:\n* IRC sessions: %d.\n"), g_pStatPluginController->numOnIrc());
						cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
						s.sprintf(_i18n_("* joins to channels: %d.\n"), g_pStatPluginController->numJoins());
						cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
						s.sprintf(_i18n_("* kicks given: %d.\n"), g_pStatPluginController->numKicks());
						cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
						s.sprintf(_i18n_("* bans set: %d.\n* topic changes: %d.\n"), g_pStatPluginController->numBans(), g_pStatPluginController->numTopics());
						cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
						break;
					}
					if( cmd->window->type() == 2 ) { // Query
						s.sprintf(_i18n_("Total queries stats:\n* number of words: %d.\n"), g_pStatPluginController->queriesWords());
						cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
						s.sprintf(_i18n_("* number of letters: %d.\n"), g_pStatPluginController->queriesLetters());
						cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
						break;
					}
					if( cmd->window->type() == 3 ) { // DCC
						s.sprintf(_i18n_("Total DCCs stats:\n* number of words: %d.\n"), g_pStatPluginController->dccsWords());
						cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
						s.sprintf(_i18n_("* number of letters: %d.\n"), g_pStatPluginController->dccsLetters());
						cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
						break;
					}
				}
			} break;

			case 'o': { // Show total number of topics
				s.sprintf(_i18n_("You have changed topic %d times.\n"), g_pStatPluginController->numTopics());
				cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
			} break;

			case 'm': { // Show channel message
				s.sprintf(_i18n_("/me has spoken %d words since %s!"),
					g_pStatPluginController->totalWords(),
					g_pStatPluginController->startDate()
				);
				kvirc_plugin_simulate_typing(cmd->window, s);
			} break;

			case 'M': { // Show channel message, but with words from this chan
				if( !chan ) {
					s.sprintf(_i18n_("No stats for %s."), cmd->window->caption());
					cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
				} else {
					s.sprintf(_i18n_("/me has spoken %d words on %s since %s!"),
						chan->words(), cmd->window->caption(),
						g_pStatPluginController->startDate()
					);
					kvirc_plugin_simulate_typing(cmd->window, s);
				}
			} break;

			case 'h': { // Print help message
				s.sprintf(usageText);
				cmd->window->outputNoFmt(KVI_OUT_HELP, s.ascii());
			} break;

			case 't': { // Just echo total words
				s.sprintf(_i18n_("Total number of words %d.\n"), g_pStatPluginController->totalWords());
				cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
			} break;

			case 'c': { // Channels words
				s.sprintf(_i18n_("Words spoken on channels: %d.\n"), g_pStatPluginController->chansWords());
				cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
			} break;

			case 'q': { // Queries words
				s.sprintf(_i18n_("Words spoken on queries: %d.\n"), g_pStatPluginController->queriesWords());
				cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
			} break;

			case 'd': { // DCCs words
				s.sprintf(_i18n_("Words spoken on DCCs: %d.\n"), g_pStatPluginController->dccsWords());
				cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
			} break;

			case 'k': { // Kicks given
				s.sprintf(_i18n_("Total number of kicks given: %d.\n"), g_pStatPluginController->numKicks());
				cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
			} break;

			case 'b': { // Bans set
				s.sprintf(_i18n_("Total number of bans set: %d.\n"), g_pStatPluginController->numBans());
				cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
			} break;

			case 'j': { // Joins to the channel
				if( !chan )
					s.sprintf(_i18n_("No stats for %s.\n"), cmd->window->caption());
				else
					s.sprintf(_i18n_("Number of times you have joined %s: %u"), cmd->window->caption(), chan->joins());
				cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
			} break;

			case 'w': { // Show complete stats window
				g_pStatPluginController->slotShowStats();
			} break;

			case 's': { // Save stats to file right now
				g_pStatPluginController->slotSaveStats();
				s.sprintf(_i18n_("Stats file saved.\n"));
				cmd->window->outputNoFmt(KVI_OUT_PLUGIN, s.ascii());
			} break;

			case 'R': { // Reset stats
				if( g_pStatPluginController->doReset()) {
					s.sprintf(_i18n_("All stats cleared.\n"));
					cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
				}
			} break;

			case 'x': { // Delete stats for given channel
				if( cmd->params->count() < 3 ) {
					s.sprintf(_i18n_("Missing argument to /STATS -x !\nUsage: /STATS -x #channel."));
					cmd->window->outputNoFmt(KVI_OUT_ERROR, s.ascii());
					break;
				} else {
					QString qs(cmd->params->at(2)->ptr());
					if( !qs.contains("#") ) qs.insert(0, '#');
					KviStatChan *sc = g_pStatPluginController->findStatChan((const char *) qs.utf8());
					if( !sc ) {
						s.sprintf(_i18n_("Channel %s is not in stats!"), (const char *) qs.utf8());
						cmd->window->outputNoFmt(KVI_OUT_ERROR, s.ascii());
					} else {
						g_pStatPluginController->removeChan(sc);
						s.sprintf(_i18n_("Channel %s removed from stats."), (const char *) qs.utf8());
						cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
					}
				}
			} break;

			case 'S': { // This session stats
				s.sprintf(_i18n_("Words spoken in this session: %u."), g_pStatPluginController->sessionWords());
				cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
			} break;

			case 'r': { // Session record
				s.sprintf(_i18n_("Biggest number of words spoken in one session: %u."), g_pStatPluginController->wordsRecord());
				cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
			} break;

			default: {
				s.sprintf(_i18n_("STATS: unknown option!\n"));
				cmd->window->outputNoFmt(KVI_OUT_ERROR, s.ascii());
			}
		}
	} else { // Show stats for current window type
		switch( cmd->window->type() ) {
			case 0: { // @console
				s.sprintf(_i18n_("Total number of IRC sessions: %d.\n"), g_pStatPluginController->numOnIrc());
				cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
			} break;

			case 1: { // Chan window; show stats for this chan
				if( !chan ) {
					s.sprintf(_i18n_("No stats for %s.\n"), cmd->window->caption());
					cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
				} else {
					s.sprintf(_i18n_("Words spoken on %s: %d.\n"), cmd->window->caption(), chan->words() );
					cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
				}
			} break;

			case 2: { // Query window
				s.sprintf(_i18n_("Words spoken on queries: %d.\n"), g_pStatPluginController->queriesWords());
				cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
			} break;

			case 3: { // DCC window
				s.sprintf(_i18n_("Words spoken on DCCs: %d.\n"), g_pStatPluginController->dccsWords());
				cmd->window->outputNoFmt(KVI_OUT_INFO, s.ascii());
			} break;
		}
	}

	return true;
}

bool stat_plugin_hook_on_topic(KviPluginCommandStruct *cmd)
{
	KviStr myNick = cmd->frame->m_global.szCurrentNick.ptr();
	KviStr sourceNick = cmd->params->at(2)->ptr();
	if( myNick == sourceNick ) {
		g_pStatPluginController->addTotalTopic();

		KviStatChan *c = g_pStatPluginController->findStatChan(cmd->window->caption());
		if( c )
			c->addTopics();
		else {
			KviStatChan *kvsc = new KviStatChan(cmd->window->caption());
			kvsc->addTopics();
			g_pStatPluginController->addChan(kvsc);
			g_pStatPluginController->addTotalJoin();

			KviStr s;
			s.sprintf(_i18n_("Added %s to stats."), cmd->window->caption());
			cmd->window->outputNoFmt(KVI_OUT_INFO, s.ptr());

			g_pStatPluginController->setCurrentChan(kvsc);
		}
	}

	return false;
}

bool stat_plugin_hook_on_irc(KviPluginCommandStruct *cmd)
{
	g_pStatPluginController->addOnIrc();
	g_pStatPluginController->resetSessionWords();

	return false;
}

bool stat_plugin_hook_on_kick(KviPluginCommandStruct *cmd)
{
	KviStr myNick = cmd->frame->m_global.szCurrentNick.ptr();
	KviStr kickerNick = cmd->params->at(3)->ptr();
	if( myNick == kickerNick ) {
		g_pStatPluginController->addTotalKick();
		KviStatChan *c = g_pStatPluginController->findStatChan(cmd->window->caption());
		if( c )
			c->addKicks();
		else {
			KviStatChan *kvsc = new KviStatChan(cmd->window->caption());
			kvsc->addKicks();
			g_pStatPluginController->addChan(kvsc);
			g_pStatPluginController->addTotalJoin();

			KviStr s;
			s.sprintf(_i18n_("Added %s to stats."), cmd->window->caption());
			cmd->window->outputNoFmt(KVI_OUT_INFO, s.ptr());

			g_pStatPluginController->setCurrentChan(kvsc);
		}
	}

	return false;
}

bool stat_plugin_hook_on_ban(KviPluginCommandStruct *cmd)
{
	KviStr myNick = cmd->frame->m_global.szCurrentNick.ptr();
	KviStr banningNick = cmd->params->at(2)->ptr();
	if( myNick == banningNick ) {
		g_pStatPluginController->addTotalBan();

		KviStatChan *c = g_pStatPluginController->findStatChan(cmd->window->caption());
		if( c )
			c->addBans();
		else
		{
			KviStatChan *kvsc = new KviStatChan(cmd->window->caption());
			kvsc->addBans();
			g_pStatPluginController->addChan(kvsc);
			g_pStatPluginController->addTotalJoin();

			KviStr s;
			s.sprintf(_i18n_("Added %s to stats."), cmd->window->caption());
			cmd->window->outputNoFmt(KVI_OUT_INFO, s.ptr());

			g_pStatPluginController->setCurrentChan(kvsc);
		}
	}

	return false;
}

bool stat_plugin_hook_on_me_join(KviPluginCommandStruct *cmd)
{
	// Does it make sense? :)))
	if( cmd->window->type() != 1 ) // Not a chan
		return false;

	g_pStatPluginController->addTotalJoin();
	KviStr myNick = cmd->frame->m_global.szCurrentNick.ptr();
	KviStr joinChan = *cmd->params->at(1);

	KviStatChan *c = g_pStatPluginController->findStatChan(joinChan.ptr());
	if( c ) {
		c->addJoins();
		stat_plugin_processJoinStats(c, cmd->window);
	} else {
		KviStatChan *tmpchan = new KviStatChan(joinChan.ptr());
		g_pStatPluginController->addChan(tmpchan);
		g_pStatPluginController->addTotalJoin();

		KviStr tmps;
		tmps.sprintf(_i18n_("Added %s to stats.\n"), tmpchan->name());
		cmd->window->outputNoFmt(KVI_OUT_INFO, tmps.ptr());
	}

	// Systray widget update stuff
	KviChannel *kvic = cmd->frame->findChannel(cmd->params->at(1)->ptr());
	if( kvic ) {
		QObjectList *list = kvic->queryList("KviInput", 0, false, true);
		QObjectListIt it(*list);
		KviInput *i = (KviInput *) it.current();
		if( i )
			i->installEventFilter(g_pStatPluginController);
		delete list;
	}

	return false;
}

bool stat_plugin_hook_on_startup(KviPluginCommandStruct *cmd)
{
	QString s;
	s.sprintf(
		_i18n_("/echo -i=$icon(kvirc) Running $b\\KviStat$o\\ plugin version $b\\%s$o\\. Have fun!!"), KVISTAT_VERSION
	);
	kvirc_plugin_simulate_typing(cmd->window, s);

	if( g_pStatPluginController->sysTrayOnStartup() ) {
		// How about this little ugly hack :) ?
		s.sprintf("/stattray dock");
		kvirc_plugin_simulate_typing(cmd->window, s);
	}

	KviConsole *con = cmd->console;
	if( !con ) return false;
	QObjectList *li = con->queryList("KviInput", 0, false, true);
	QObjectListIt it_(*li);
	KviInput *i = (KviInput *) it_.current();
	if( i )
		i->installEventFilter(g_pStatPluginController);
	delete li;

	return false;
}

bool stat_plugin_hook_on_disconnect(KviPluginCommandStruct *cmd)
{
	if( g_pStatPluginController->sessionWords() > g_pStatPluginController->wordsRecord() ) {
		g_pStatPluginController->setWordsRecord(g_pStatPluginController->sessionWords());
		cmd->window->outputNoFmt(KVI_OUT_INFO, _i18n_("[stats]: you have just beaten your session record!"));
		cmd->window->output(KVI_OUT_INFO,
			_i18n_("[stats]: now it is %c%d%c words!"),
			KVI_TEXT_BOLD, g_pStatPluginController->wordsRecord(), KVI_TEXT_BOLD
		);
	}
	return false;
}

bool stat_plugin_hook_on_shutdown(KviPluginCommandStruct *cmd)
{
	if( g_pStatPluginController->sessionWords() > g_pStatPluginController->wordsRecord() )
		g_pStatPluginController->setWordsRecord(g_pStatPluginController->sessionWords());
	return false;
}

/**
 * Syntax:
 * /stattray [on]             - Dock the stattray into current frame
 * /stattray <off> <undock>   - Undock it
 */
bool stat_plugin_command_stattray(KviPluginCommandStruct *cmd)
{
	KviStr param = kvirc_plugin_param(cmd, 1);

	if( kvi_strEqualCI(param.ptr(), "undock") || kvi_strEqualCI(param.ptr(), "off") ) {
		KviSysTrayWidget *w = kvirc_plugin_lookup_systray_widget(cmd->frame, "KviStatSysTray");
		if( !w ) {
			cmd->error = KVI_ERROR_InvalidParameter;
			cmd->errorstr = _i18n_("No StatTray to undock");
			return false;
		}

		kvirc_plugin_remove_systray_widget(cmd->frame, w, true);
		return true;
	} else { // Dock
		KviSysTrayWidget *w = kvirc_plugin_lookup_systray_widget(cmd->frame, "KviStatSysTray");
		if( w ) {
			cmd->error = KVI_ERROR_InvalidParameter;
			cmd->errorstr = _i18n_("StatTray already docked in this frame");
			return false;
		}
	}

	KviStatSysTray *sw = new KviStatSysTray(kvirc_plugin_get_systray(cmd->frame), cmd->frame);
	QObject::connect(sw, SIGNAL(wantOptions()), g_pStatPluginController, SLOT(setSysTrayOptions()));

	kvirc_plugin_add_systray_widget(cmd->handle, cmd->frame, sw, true);

	g_pStatPluginController->setSysTrayOptions();

	return true;
}

bool stat_plugin_hook_on_action(KviPluginCommandStruct *cmd)
{
	return true;
}

bool stat_plugin_init(KviPluginCommandStruct *cmd)
{
	g_pStatPluginController = new KviStatController();

	kvirc_plugin_add_hook(cmd->handle, KviEvent_OnAction,       stat_plugin_hook_on_action);
	kvirc_plugin_add_hook(cmd->handle, KviEvent_OnChannelInput, stat_plugin_hook_addstat);
	kvirc_plugin_add_hook(cmd->handle, KviEvent_OnQueryInput,   stat_plugin_hook_addstat);
	kvirc_plugin_add_hook(cmd->handle, KviEvent_OnDCCInput,     stat_plugin_hook_addstat);
	kvirc_plugin_add_hook(cmd->handle, KviEvent_OnIrc,          stat_plugin_hook_on_irc);
	kvirc_plugin_add_hook(cmd->handle, KviEvent_OnTopic,        stat_plugin_hook_on_topic);
	kvirc_plugin_add_hook(cmd->handle, KviEvent_OnKick,         stat_plugin_hook_on_kick);
	kvirc_plugin_add_hook(cmd->handle, KviEvent_OnBan,          stat_plugin_hook_on_ban);
	kvirc_plugin_add_hook(cmd->handle, KviEvent_OnMeJoin,       stat_plugin_hook_on_me_join);

	kvirc_plugin_add_hook(cmd->handle, KviEvent_OnStartup, stat_plugin_hook_on_startup);

	kvirc_plugin_add_hook(cmd->handle, KviEvent_OnDisconnect, stat_plugin_hook_on_disconnect);
	kvirc_plugin_add_hook(cmd->handle, KviEvent_OnShutdown,   stat_plugin_hook_on_shutdown);
	kvirc_plugin_register_command(cmd->handle, "STATS",    stat_plugin_command_stats);
	kvirc_plugin_register_command(cmd->handle, "STATTRAY", stat_plugin_command_stattray);

	if( !(cmd->frame) ) return true;
	if( !(cmd->frame->m_pWinList) ) return true;

	for(KviWindow *wnd = cmd->frame->m_pWinList->first(); wnd; wnd = cmd->frame->m_pWinList->next() ) {
		if( wnd->type() == KVI_WND_TYPE_CHANNEL ) {
			KviStatChan *c = g_pStatPluginController->findStatChan(wnd->caption());
			if( !c ) {
				c = new KviStatChan(wnd->caption());
				g_pStatPluginController->addChan(c);
				wnd->output(KVI_OUT_INFO, _i18n_("Added %s to stats.\n"), c->name());
			}

			if( wnd->m_pInput )
				wnd->m_pInput->installEventFilter(g_pStatPluginController);

			c->addJoins(1);
			g_pStatPluginController->addTotalJoin();

			g_pStatPluginController->setCurrentChan(c);

			stat_plugin_processJoinStats(c, wnd);
		}
	}

	if( cmd->console ) {
		if( cmd->console->m_pInput )
			cmd->console->m_pInput->installEventFilter(g_pStatPluginController);
	}

	return true;
}

void stat_plugin_cleanup()
{
	if( g_pStatPluginOptionsWindow ) {
		delete g_pStatPluginOptionsWindow;
		g_pStatPluginOptionsWindow = 0;
	}
	delete g_pStatPluginController;  g_pStatPluginController  = 0;
}

/**
	@document: doc_plugin_stat.kvihelp
	@title: The KVIrc Stat plugin ver. 0.2.0
		This is a plugin which keeps statistics of your IRC sessions,
		which means you can check at any time how many words you have said,
		how many times you have joined specific channel etc.<br>
		&nbsp;&nbsp;&nbsp;The plugin exports commands: <b>/STATS</b> and
		<b>/STATTRAY</b>.
		Command <b>/STATS</b> can be invoked with quite a large number of option.
		Only one option at a time is allowed though. Every option except the
		first one is ignored.
		<br>
		Here are all the options with descriptions:<br>
		<br>
		<ul><b>Options that show something:
			<li><i>-h</i> - a help message with all the options available.</li>
			<li><i>-t</i> - total number of words spoken</li>
			<li><i>-c</i> - total number of words spoken on all channels</li>
			<li><i>-q</i> - total number of words spoken on all queries</li>
			<li><i>-d</i> - total number of words spoken on all DCC Chats</li>
			<li><i>-k</i> - total number of kicks given (on all channels)</li>
			<li><i>-b</i> - total number of kicks given (all channels also)</li>
			<li><i>-o</i> - total number of topic changes</li>
			<li><i>-j</i> - number of times you have joined the channel you are
			currently on.</li>
			<li><i>-m</i> - a message to current channel telling how many words
			you have said since last reset. It looks just like you'd written:
			/me has spoken <i>(total_words)</i> words since <i>(reset_date)</i>
			</li>
			<li><i>-M</i> - similar to <i>-m</i>, but shows how many words you
			have spoken on the channel you are currently on. Example:<br>
			<i>Kristoff has spoken 143 words on #linuxpl since Wed Mar
			29 02:10:21 2000!</i></li>
			<li><i>-D</i> - detailed statistics for current <i>window type</i>.
			For example if you type <b>/STATS -D</b> in <i>@CONSOLE</i> you will
			get "totals" list - total words, IRC sessions, kicks etc. If you type
			the same command in a <i>channel window</i> you will get this
			channel's detailed statistics</li>
			<li><i>-S</i> - current session stats
			<li><i>-r</i> - your "high score" of words spoken in one session
			<li><i>-w</i> - a window with <b>complete</b> stats. Everything is
			there - "totals", channel list, and actions such as kicks etc.
			In the <i>Channels</i> tab you can remove specific channels from
			stats, or remove all the channels (without
			resetting total words etc.). There is also a <i>Reset</i> button
			which allows
			you to reset <b>all</b> the statistics. You will have to confirm
			resetting statistics as well as removing a chan.</li>
		</ul>
		<br>
		<ul><b>Other options:</b>
			<li><i>-R</i> - reset statistics, after confirmation of course.</li>
			<li><i>-s</i> - save statistics to file <b>now</b>.</li>
			<li><i>-x &lt;channel&gt;</i> - remove specified channel from your
			stats. You are not asked if you are sure you want to do this, so be
			careful ;-). It is not necessary to enter a <b>#</b> in front of
			the channel's name.</li>
			<li><i>-C</i> - configure the plugin. Here you can configure stats
			shown on <b>OnMeJoin</b> Event (read more about events
			<a href="events.kvihelp">here</a>). You can choose to see either
			no, full or customized stats. This is quite simple really, so
			there is not much to explain. Another thing you can configure is the
			<b>StatTray</b>.</li>
		</ul>
		<br>
		&nbsp;&nbsp;&nbsp;If <b>/STATS</b> is invoked without any options it
		shows information depending on <i>window type</i>. In <i>channel windows</i>
		it shows number of words on the channel, on <i>query</i>
		and <i>DCC Chat windows</i> you will see number of words you have said
		there. In <i>@CONSOLE</i> it'll show number of IRC sessions.<br><br>
		&nbsp;&nbsp;&nbsp;Option <b>/STATTRAY</b> can be invoked with <i>ON</i>
		or <i>OFF</i> option. They do exactly what you think - turn the StatTray
		widget on or off. This widget's properties can be set in the plugin's
		configuration dialog.<br><br>

		<ul><b>Scrolling Options</b>
			<li><i>When is it enabled</i> - You have 3 possibilities: <i>Never</i>,
			<i>Always</i> and <i>When Needed</i>. <i>When Needed</i> means
			that scrolling will be enabled only if text will not fit into the
			widget.
			<li><i>Direction</i> - Left or Right - plenty to choose from ;-)
			<li><i>Delay</i> - time between each text movement
			<li><i>Text Movement</i> - ammount of pixels the text moves every
			time &lt;- set by the <i>Delay</i> slider
		</ul><br>
		It is generally better to have the <i>Text Movement</i> slider set more
		to the right... Although scrolling does not look as nice as if it was
		moved max to the left, but at least it is not so lagging...<br>
		<br>
		<hr>
		<b>KviStat</b> was written by Krzysztof <b>"Kristoff"</b> Godlewski
		&lt;<i>kristoff@poczta.wprost.pl</i>&gt;<br>
		<br>
		If you notice any bugs, or have suggestions / ideas how to make this
		plugin better please email me.<br>
		<br>
		<u><b>BIG</b></u> Thanks to <b>Szymon Stefanek</b> for his support.
*/
void stat_plugin_help()
{
	kvirc_plugin_show_help_page("doc_plugin_stat.kvihelp");
}

void stat_plugin_config()
{
	g_pStatPluginController->slotShowConfig();
}

KviPlugin kvirc_plugin =
{
	"Stat",
	_i18n_("Stats plugin"),
	KVISTAT_VERSION,
	"Krzysztof Godlewski\n<kristoff@poczta.wprost.pl>", // Author field
	_i18n_("<html>"                                       // Description field
		"This plugin creates and saves your session statistics.<br><br>"
		"<b>/STATS</b>: check your statistics.<br>"
		"<b>/STATTRAY</b>: control the StatTray widget.<br><br>"
		"See help page or <b>/STATS -h</b> for more information."
		"</html>"),
	stat_plugin_init,      // Init function
	stat_plugin_cleanup,   // Cleanup function
	stat_plugin_config,    // Config function
	stat_plugin_help       // Help function
};
