// =============================================================================
//
//      --- kvi_options.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Sxoftware Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviOptions"

#include <qdir.h>

#define _KVI_OPTIONS_CPP_

#include "kvi_alias.h"
#include "kvi_app.h"
#include "kvi_config.h"
#include "kvi_event.h"
#include "kvi_irc_proxy.h"
#include "kvi_irc_server.h"
#include "kvi_locale.h"
#include "kvi_mimemanager.h"
#include "kvi_mirccntrl.h"
#include "kvi_options.h"
#include "kvi_rawevent.h"
#include "kvi_regusersdb.h"
#include "kvi_settings.h"
#include "kvi_strsub.h"
#include "kvi_userpopupmenu.h"
#include "kvi_usertoolbartemplate.h"
#include "kvi_winproperties.h"
#ifdef COMPILE_PLUGIN_SUPPORT
	#include "kvi_plugin.h"
	extern KviPluginManager *g_pPluginManager;
#endif
#ifdef COMPILE_NEED_CHARSET_TRANSLATION
	#include "kvi_charset_translator.h"
	extern KviCharsetTranslator *g_pCharsetTranslator;
#endif

KviOptions *g_pOptions;

QRgb defaultMircRgb[16] =
{
	qRgb(255, 255, 255), qRgb(  0,   0,   0), qRgb(  0,   0, 140), qRgb(  0, 100,   0),
	qRgb(230,   0,   0), qRgb(150,   0,   0), qRgb( 80,   0,  80), qRgb(255,  90,   0),
	qRgb(255, 255,   0), qRgb(  0, 255,   0), qRgb(  0, 150, 180), qRgb(170, 170, 255),
	qRgb( 15,  15, 255), qRgb(200,   0, 200), qRgb( 80,  80,  80), qRgb(170, 170, 170)
};

QRgb defaultScriptEditorRgb[KVI_SCRIPTEDITOR_NUM_COLORS] =
{
	qRgb(  0, 255,   0), // TEXT
	qRgb(  0, 235,   0), // COMMAND
	qRgb(  0, 170,   0), // COMMENT
	qRgb(135, 135, 255), // STRING
	qRgb(255,   0, 255), // ESCAPE
	qRgb(255,   0,   0), // BRACE
	qRgb(194, 194, 194), // LOCAL VARIABLE
	qRgb(255, 255, 255), // GLOBAL VARIABLE
	qRgb(180, 180,  20), // OPERATOR
	qRgb(255, 190,   0), // THIS
	qRgb(255, 255,   0), // IDENTIFIER
	qRgb(180, 180, 180), // DISABLED BACK
	qRgb(  0,   0,   0)  // ENABLED BACK
};

// Declared in kvi_app.cpp
extern void g_setSplashScreenProgress(int progress);
extern void g_setSplashScreenText(const char *text);

// Created in KviApp constructor, loaded and saved here
extern KviAliasManager    *g_pAliasManager;
extern KviEventManager    *g_pEventManager;
extern KviRawEventManager *g_pRawEventManager;

/**
 * The constructor should set the splash screen progress from 14% to 83%
 */
KviOptions::KviOptions()
{
	g_setSplashScreenProgress(14);
	g_setSplashScreenText(_i18n_("Loading options..."));

	for( int i = 0; i < 16; i++ )
		m_pMircColor[i] = new QColor(defaultMircRgb[i]);
	for( int i = 0; i < KVI_SCRIPTEDITOR_NUM_COLORS; i++ )
		m_pScriptEditorClr[i] = new QColor(defaultScriptEditorRgb[i]);
	m_pixViewBack        = new QPixmap();
	m_pixInputBack       = new QPixmap();
	m_pixMdiBack         = new QPixmap();
	m_pixListBoxBack     = new QPixmap();
	m_pixLabelsBack      = new QPixmap();
	m_pServerManager     = new KviIrcServerManager();
	m_pProxyManager      = new KviIrcProxyManager();
	m_pWinPropertiesList = new KviWindowPropertiesList();
	m_pMimeManager       = new KviMimeManager();
	m_pRegUsersDb        = new KviRegisteredUsersDb();
	m_pHighlightWords    = new QPtrList<KviStr>;
	m_pHighlightWords->setAutoDelete(true);
	m_pDirBookmarks      = new QPtrList<KviStr>;
	m_pDirBookmarks->setAutoDelete(true);
	m_pStrSub            = new KviStrSub();
	g_setSplashScreenProgress(16);
	load();
}

/**
 * ============ ~KviOptions ============
 */
KviOptions::~KviOptions()
{
	save();
	for( int i = 0; i < 16; i++ ) {
		delete m_pMircColor[i];
		m_pMircColor[i] = 0;
	}
	for( int i = 0; i < KVI_SCRIPTEDITOR_NUM_COLORS; i++ ) {
		delete m_pScriptEditorClr[i];
		m_pScriptEditorClr[i] = 0;
	}
	delete m_pixViewBack;        m_pixViewBack        = 0;
	delete m_pixInputBack;       m_pixInputBack       = 0;
	delete m_pixMdiBack;         m_pixMdiBack         = 0;
	delete m_pixListBoxBack;     m_pixListBoxBack     = 0;
	delete m_pixLabelsBack;      m_pixLabelsBack      = 0;
	delete m_pServerManager;     m_pServerManager     = 0;
	delete m_pProxyManager;      m_pProxyManager      = 0;
	delete m_pWinPropertiesList; m_pWinPropertiesList = 0;
	delete m_pMimeManager;       m_pMimeManager       = 0;
	delete m_pRegUsersDb;        m_pRegUsersDb        = 0;
	delete m_pHighlightWords;    m_pHighlightWords    = 0;
	delete m_pDirBookmarks;      m_pDirBookmarks      = 0;
	delete m_pStrSub;            m_pStrSub            = 0;
}

/**
 * ============== load ===============
 */
void KviOptions::load()
{
	g_setSplashScreenText(_i18n_("Loading GUI options..."));
	loadMircColors();
	g_setSplashScreenProgress(18);
	loadViewOptions();
	g_setSplashScreenProgress(22);
	loadListBoxOptions();
	g_setSplashScreenProgress(26);
	loadInputOptions();
	g_setSplashScreenProgress(30);
	loadStrSub();
	g_setSplashScreenProgress(32);
	loadLabelsOptions();
	g_setSplashScreenProgress(34);
	loadMdiOptions();
	g_setSplashScreenProgress(38);
	loadHelpOptions();
	g_setSplashScreenProgress(42);
	loadFrameOptions();
	g_setSplashScreenProgress(46);
	loadWinProperties();
	g_setSplashScreenProgress(50);
	loadDirBrowserOptions();
	g_setSplashScreenProgress(52);
	loadScriptCenterOptions();
	g_setSplashScreenProgress(54);
	g_setSplashScreenText(_i18n_("Loading core options..."));
	loadSocketOptions();
	g_setSplashScreenProgress(56);
	loadServerOptions();
	g_setSplashScreenProgress(59);
	loadUserOptions();
	g_setSplashScreenProgress(61);
	loadIrcOptions();
	g_setSplashScreenProgress(63);
	loadDccOptions();
	g_setSplashScreenProgress(65);
	loadMimeTypes();
	g_setSplashScreenProgress(67);
	loadRegUsers();
	g_setSplashScreenProgress(69);
	loadLoggingOptions();
	g_setSplashScreenProgress(71);
	g_setSplashScreenText(_i18n_("Loading aliases, popups, event handlers..."));
	loadAliases();
	g_setSplashScreenProgress(73);
	loadPopups();
	g_setSplashScreenProgress(75);
	loadEvents();
	g_setSplashScreenProgress(78);
	loadToolBar();
	g_setSplashScreenProgress(81);
	loadRawEvents();
	g_setSplashScreenProgress(83);
#ifdef COMPILE_PLUGIN_SUPPORT
	loadPlugins();
#endif
	g_setSplashScreenProgress(84);
}

/**
 * ============== save ===============
 */
void KviOptions::save()
{
	saveMircColors();
	saveViewOptions();
	saveListBoxOptions();
	saveHelpOptions();
	saveLabelsOptions();
	saveInputOptions();
	saveIrcOptions();
	saveFrameOptions();
	saveWinProperties();
	saveDirBrowserOptions();
	saveLoggingOptions();
	saveSocketOptions();
	saveServerOptions();
	saveRegUsers();
	saveDccOptions();
	saveUserOptions();
	saveMdiOptions();
	saveMimeTypes();
	saveAliases();
	savePopups();
	saveEvents();
	saveToolBar();
	saveRawEvents();
#ifdef COMPILE_PLUGIN_SUPPORT
	savePlugins();
#endif
	saveStrSub();
	saveScriptCenterOptions();
}

/**
 * ============== loadMircColors ==============
 */
void KviOptions::loadMircColors()
{
	KviStr szConf;
	g_pApp->getReadOnlyConfigPath(szConf, "kvi.mircclr.conf");
	KviConfig cfg(szConf.ptr());
	cfg.setGroup("MircColors");
	m_bUseKsircControlCodes = cfg.readBoolEntry("UseKsircControlCodes", false);
	for( int i = 0; i < 16; i++ ) {
		KviStr szEntry(KviStr::Format, "Color%d", i);
		QColor defClr(defaultMircRgb[i]);
		(*(m_pMircColor[i])) = cfg.readColorEntry(szEntry.ptr(), defClr);
	}
}

/**
 * ============= saveMircColors ===========
 */
void KviOptions::saveMircColors()
{
	KviStr szConf;
	g_pApp->getLocalKVIrcDirectory(szConf, KviApp::Config, "kvi.mircclr.conf");
	KviConfig cfg(szConf.ptr());
	cfg.setGroup("MircColors");
	cfg.writeEntry("UseKsircControlCodes", m_bUseKsircControlCodes);
	for( int i = 0; i < 16; i++ ) {
		KviStr szEntry(KviStr::Format, "Color%d", i);
		cfg.writeEntry(szEntry.ptr(), (*(m_pMircColor[i])));
	}
}

/**
 * ============ loadLabelsOptions ==============
 */
void KviOptions::loadLabelsOptions()
{
	KviStr szTemp;
	g_pApp->getReadOnlyConfigPath(szTemp, "kvi.labels.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("LabelsOptions");

	m_szLabelsBack = cfg.readEntry("BackgroundImage", "");
	loadLabelsBackgroundImage();
	// Font
	QFont fnt(KVI_DEFAULT_FONT, KVI_DEFAULT_FONTSIZE);
	m_fntLabels = cfg.readFontEntry("Font", fnt);
	QColor wh(255, 255, 255);
	QColor gr(180, 180, 180);
	QColor bk(  0,   0,   0);
	m_clrLabelsBack         = cfg.readColorEntry("BackgroundColor",         wh);
	m_clrLabelsActiveFore   = cfg.readColorEntry("ActiveForegroundColor",   bk);
	m_clrLabelsInactiveFore = cfg.readColorEntry("InactiveForegroundColor", gr);
}

void KviOptions::loadLabelsBackgroundImage(const QPixmap *cached)
{
	loadBackgroundImage(m_szLabelsBack, m_pixLabelsBack, cached);
}

/**
 * ============ saveLabelsOptions ================
 */
void KviOptions::saveLabelsOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.labels.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("LabelsOptions");

	cfg.writeEntry("BackgroundImage",         m_szLabelsBack.ptr());
	cfg.writeEntry("Font",                    m_fntLabels);
	cfg.writeEntry("BackgroundColor",         m_clrLabelsBack);
	cfg.writeEntry("ActiveForegroundColor",   m_clrLabelsActiveFore);
	cfg.writeEntry("InactiveForegroundColor", m_clrLabelsInactiveFore);
}

void KviOptions::saveDirBrowserOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.dirbrowser.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("DirBrowserOptions");

	cfg.writeEntry("ShowHiddenFiles",       m_bDirBrowserShowHiddenFiles);
	cfg.writeEntry("GroupFilesByExtension", m_bDirBrowserGroupFilesByExtension);
	cfg.writeEntry("SortBySize",            m_bDirBrowserSortBySize);
	cfg.writeEntry("SortCaseInsensitive",   m_bDirBrowserSortCaseInsensitive);
	cfg.writeEntry("OpenWithEditorCmd",     m_szDirBrowserOpenWithEditorCmd.ptr());
	cfg.writeEntry("OpenAsMdiWindow",       m_bDirBrowserOpenAsMdiWindow);
	cfg.writeEntry("OpenAlwaysUndocked",    m_bDirBrowserOpenAlwaysUndocked);
}

void KviOptions::loadDirBrowserOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.dirbrowser.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("DirBrowserOptions");

	m_bDirBrowserShowHiddenFiles       = cfg.readBoolEntry("ShowHiddenFiles", false);
	m_bDirBrowserGroupFilesByExtension = cfg.readBoolEntry("GroupFilesByExtension", true);
	m_bDirBrowserSortBySize            = cfg.readBoolEntry("SortBySize", false);
	m_bDirBrowserSortCaseInsensitive   = cfg.readBoolEntry("SortCaseInsensitive", false);
	m_szDirBrowserOpenWithEditorCmd    = cfg.readEntry("OpenWithEditorCmd", "run xedit $ExecFileName");
	m_bDirBrowserOpenAsMdiWindow       = cfg.readBoolEntry("OpenAsMdiWindow", false);
	m_bDirBrowserOpenAlwaysUndocked    = cfg.readBoolEntry("OpenAlwaysUndocked", false);
}

void KviOptions::loadScriptCenterOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.scriptcenter.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("ScriptCenter");
	m_bScriptCenterShowAsDialog         = cfg.readBoolEntry("ShowScriptCenterAsDialog", true);
	m_bScriptCenterAddHelpBrowser       = cfg.readBoolEntry("AddHelpBrowserTab", false);
	m_bScriptCenterAutoSaveWhenAccepted = cfg.readBoolEntry("AutoSave", true);
}

void KviOptions::saveScriptCenterOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.scriptcenter.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("ScriptCenter");
	cfg.writeEntry("ShowScriptCenterAsDialog", m_bScriptCenterShowAsDialog);
	cfg.writeEntry("AddHelpBrowserTab",        m_bScriptCenterAddHelpBrowser);
	cfg.writeEntry("AutoSave",                 m_bScriptCenterAutoSaveWhenAccepted);
}

void KviOptions::loadFrameOptions()
{
	KviStr szTemp;
	g_pApp->getReadOnlyConfigPath(szTemp, "kvi.frame.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("FrameOptions");

	m_bUseBigToolbarPixmaps            = cfg.readBoolEntry("UseBigToolbarPixmaps", true);
	m_bUseBigUserToolbarPixmaps        = cfg.readBoolEntry("UseBigUserToolbarPixmaps", true);
	m_bUseTextToolbarComment           = cfg.readBoolEntry("UseTextToolbarComment", false);
	m_bUseThemeMenubarBackground       = cfg.readBoolEntry("UseThemeMenubarBackground", false);

	QFont fnt(KVI_DEFAULT_FONT, KVI_DEFAULT_FONTSIZE);
	QColor bk(  0,   0,   0);
	QColor wh(255, 255, 255);
	m_szSysTrayMode                    = cfg.readEntry("SysTrayMode", "KviSysTrayIoLed");
	m_fntSysTray                       = cfg.readFontEntry("SysTrayFont", fnt);
	m_clrSysTrayFore                   = cfg.readColorEntry("SysTrayForegroundColor", bk);
	m_clrSysTrayBack                   = cfg.readColorEntry("SysTrayBackgroundColor", wh);

	QRect r(0, 0, 780, 580);
	m_rectFrameGeometry                = cfg.readRectEntry("MainFrameGeometry", r);
	m_rectScriptDialog                 = cfg.readRectEntry("ScriptDialogGeometry", r);
	m_rectGeneralOptionsDialog         = cfg.readRectEntry("GeneralOptionsDialogGeometry", r);

	m_bUseHackedFrameGeometry          = cfg.readBoolEntry("UseHackedFrameGeometry", true);
	m_bDialogShowHackForBrainDamagedWM = cfg.readBoolEntry("DialogShowHackForBrainDamagedWM", false);

	KviStr tmp = QDir::homeDirPath();
	QFont fnt2(KVI_DEFAULT_FONT, KVI_DEFAULT_FONTSIZE);
	m_szLastBrowsedDirectory           = cfg.readEntry("LastBrowsedDirectory", tmp.ptr());
	m_fntApplication                   = cfg.readFontEntry("ApplicationFont", fnt2);

	m_szLastScriptPath                 = cfg.readEntry("LastScriptPath", "");

	m_bSetGlobalPaletteAtStartup       = cfg.readBoolEntry("SetGlobalPaletteAtStartup", false);
	m_bMdiOpaqueMove                   = cfg.readBoolEntry("MdiOpaqueMove", false);
	m_bMdiOpaqueResize                 = cfg.readBoolEntry("MdiOpaqueResize", false);

	QColor   clrFore(245, 215,   0);
	QColor   clrText(140, 155, 255);
	QColor   clrDark( 50,  50,  75);
	QColor clrButton( 80,  88, 140);
	QColor  clrLight(160, 200, 205);
	QColor    clrMid(115, 130, 190);
	QColor   clrBase(105,  70, 120);
	m_clrNormalGlobalForeground         = cfg.readColorEntry("NormalGlobalForeground",   clrFore);
	m_clrNormalGlobalButton             = cfg.readColorEntry("NormalGlobalButton",       clrButton);
	m_clrNormalGlobalLight              = cfg.readColorEntry("NormalGlobalLight",        clrLight);
	m_clrNormalGlobalDark               = cfg.readColorEntry("NormalGlobalDark",         clrDark);
	m_clrNormalGlobalMid                = cfg.readColorEntry("NormalGlobalMid",          clrMid);
	m_clrNormalGlobalText               = cfg.readColorEntry("NormalGlobalText",         clrText);
	m_clrNormalGlobalBase               = cfg.readColorEntry("NormalGlobalBase",         clrBase);
	m_clrDisabledGlobalForeground       = cfg.readColorEntry("DisabledGlobalForeground", Qt::darkGray);
	m_clrDisabledGlobalText             = cfg.readColorEntry("DisabledGlobalText",       Qt::darkGray);
	m_clrDisabledGlobalBase             = cfg.readColorEntry("DisabledGlobalBase",       clrBase);
	m_szApplicationGUIStyle             = cfg.readEntry("ApplicationGUIName", "default");

	tmp = cfg.readEntry("DirectoryBookmarks", "");
	tmp.stripWhiteSpace();
	while( m_pDirBookmarks->first() )
		m_pDirBookmarks->removeFirst();
	KviStr url;
	while( tmp.hasData() ) {
		tmp.getToken(url, ' ');
		if( url.hasData() && (m_pDirBookmarks->count() <= KVI_DIRBROWSER_MAX_BOOKMARKS) ) {
			m_pDirBookmarks->append(new KviStr(url));
		}
		tmp.stripWhiteSpace();
	}
	m_bEnableDockWidget                = cfg.readBoolEntry("EnableDockWidget", false);
	m_bShowTipOfTheDayAtStartup        = cfg.readBoolEntry("ShowTipOfTheDayAtStartup", true);
	m_iNextTipOfTheDayIdx              = cfg.readIntEntry("NextTipOfTheDayIndex", 0);
	m_szRecentServers                  = cfg.readEntry("RecentServers", "");
	m_szRecentChannels                 = cfg.readEntry("RecentChannels", "#kvirc");
	m_szRecentNicknames                = cfg.readEntry("RecentNicknames", "");

	m_bStatusBarVisible                = cfg.readBoolEntry("StatusBarVisible", true);
#ifdef COMPILE_NEED_CHARSET_TRANSLATION
	m_bUseCharsetTranslation           = cfg.readBoolEntry("UseCharsetTranslation", false);
	m_szCharsetTranslationMapFile      = cfg.readEntry("CharsetTranslationMapFile", "");
	if( m_bUseCharsetTranslation ) {
		if( !g_pCharsetTranslator->setTranslationMap(m_szCharsetTranslationMapFile.ptr()) ) {
			m_bUseCharsetTranslation      = false;
			m_szCharsetTranslationMapFile = "";
		}
	}
#endif
}

void KviOptions::saveFrameOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.frame.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("FrameOptions");

	cfg.writeEntry("UseBigToolbarPixmaps",            m_bUseBigToolbarPixmaps);
	cfg.writeEntry("UseBigUserToolbarPixmaps",        m_bUseBigUserToolbarPixmaps);
	cfg.writeEntry("UseTextToolbarComment",           m_bUseTextToolbarComment);
	cfg.writeEntry("UseThemeMenubarBackground",       m_bUseThemeMenubarBackground);
	cfg.writeEntry("SysTrayMode",                     m_szSysTrayMode.ptr());
	cfg.writeEntry("SysTrayFont",                     m_fntSysTray);
	cfg.writeEntry("SysTrayForegroundColor",          m_clrSysTrayFore);
	cfg.writeEntry("SysTrayBackgroundColor",          m_clrSysTrayBack);
	cfg.writeEntry("MainFrameGeometry",               m_rectFrameGeometry);
	cfg.writeEntry("ScriptDialogGeometry",            m_rectScriptDialog);
	cfg.writeEntry("GeneralOptionsDialogGeometry",    m_rectGeneralOptionsDialog);
	cfg.writeEntry("UseHackedFrameGeometry",          m_bUseHackedFrameGeometry);
	cfg.writeEntry("LastBrowsedDirectory",            m_szLastBrowsedDirectory.ptr());
	cfg.writeEntry("LastScriptPath",                  m_szLastScriptPath.ptr());
	cfg.writeEntry("ApplicationFont",                 m_fntApplication);
	cfg.writeEntry("EnableDockWidget",                m_bEnableDockWidget);
	cfg.writeEntry("ShowTipOfTheDayAtStartup",        m_bShowTipOfTheDayAtStartup);
	cfg.writeEntry("NextTipOfTheDayIndex",            m_iNextTipOfTheDayIdx);
	cfg.writeEntry("SetGlobalPaletteAtStartup",       m_bSetGlobalPaletteAtStartup);
	cfg.writeEntry("NormalGlobalForeground",          m_clrNormalGlobalForeground);
	cfg.writeEntry("NormalGlobalButton",              m_clrNormalGlobalButton);
	cfg.writeEntry("NormalGlobalLight",               m_clrNormalGlobalLight);
	cfg.writeEntry("NormalGlobalDark",                m_clrNormalGlobalDark);
	cfg.writeEntry("NormalGlobalMid",                 m_clrNormalGlobalMid);
	cfg.writeEntry("NormalGlobalText",                m_clrNormalGlobalText);
	cfg.writeEntry("NormalGlobalBase",                m_clrNormalGlobalBase);
	cfg.writeEntry("DisabledGlobalForeground",        m_clrDisabledGlobalForeground);
	cfg.writeEntry("DisabledGlobalText",              m_clrDisabledGlobalText);
	cfg.writeEntry("DisabledGlobalBase",              m_clrDisabledGlobalBase);
	cfg.writeEntry("DialogShowHackForBrainDamagedWM", m_bDialogShowHackForBrainDamagedWM);
	cfg.writeEntry("RecentServers",                   m_szRecentServers.ptr());
	cfg.writeEntry("RecentChannels",                  m_szRecentChannels.ptr());
	cfg.writeEntry("RecentNicknames",                 m_szRecentNicknames.ptr());
	cfg.writeEntry("ApplicationGUIName",              m_szApplicationGUIStyle.ptr());
	cfg.writeEntry("MdiOpaqueMove",                   m_bMdiOpaqueMove);
	cfg.writeEntry("MdiOpaqueResize",                 m_bMdiOpaqueResize);
	cfg.writeEntry("StatusBarVisible",                m_bStatusBarVisible);
#ifdef COMPILE_NEED_CHARSET_TRANSLATION
	cfg.writeEntry("UseCharsetTranslation",           m_bUseCharsetTranslation);
	cfg.writeEntry("CharsetTranslationMapFile",       m_szCharsetTranslationMapFile.ptr());
#endif

	KviStr tmp;
	for( KviStr *s = m_pDirBookmarks->first(); s; s = m_pDirBookmarks->next() ) {
		if( tmp.hasData() )
			tmp.append(' ');
		tmp.append(*s);
	}
	cfg.writeEntry("DirectoryBookmarks", tmp.ptr());
}

/**
 * ============ loadListBoxOptions =============
 */
void KviOptions::loadListBoxOptions()
{
	KviStr szTemp;
	g_pApp->getReadOnlyConfigPath(szTemp, "kvi.listbox.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("ListBoxOptions");
	// Background Image
	m_szListBoxBack = cfg.readEntry("BackgroundImage", "");
	loadListBoxBackgroundImage();
	// Font
	QFont fnt(KVI_DEFAULT_FONT, KVI_DEFAULT_FONTSIZE);
	m_fntListBox = cfg.readFontEntry("Font", fnt);
	calculateListBoxFontVariables();
	// Show images?
	m_bListBoxShowImages = cfg.readBoolEntry("ShowImages", true);
	QColor wh(255, 255, 255);
	QColor pu(160,   0, 100);
	QColor rd(160,  40,   0);
	QColor gr(  0, 100,   0);
	QColor bl(  0,   0, 160);
	QColor ye(160, 160,   0);
	QColor bk(  0,   0,   0);
	m_clrListBoxBack     = cfg.readColorEntry("BackgroundColor",     wh);
	m_clrListBoxOwner    = cfg.readColorEntry("OwnerColor",          pu);
	m_clrListBoxOp       = cfg.readColorEntry("OpColor",             rd);
	m_clrListBoxHalfOp   = cfg.readColorEntry("HalfOpColor",         gr);
	m_clrListBoxVoice    = cfg.readColorEntry("VoiceColor",          bl);
	m_clrListBoxUserOp   = cfg.readColorEntry("UserOpColor",         ye);
	m_clrListBoxNormal   = cfg.readColorEntry("NormalColor",         bk);
	m_clrListBoxSeleBack = cfg.readColorEntry("SelectionBackground", bk);
	m_clrListBoxSeleFore = cfg.readColorEntry("SelectionForeground", wh);
}

void KviOptions::loadListBoxBackgroundImage(const QPixmap *cached)
{
	loadBackgroundImage(m_szListBoxBack, m_pixListBoxBack, cached);
}

void KviOptions::calculateListBoxFontVariables()
{
	QFontMetrics fm(m_fntListBox);
	m_iListBoxCellHeight = fm.lineSpacing();
	// The icons are 16x16!
	if( m_iListBoxCellHeight < 16 )
		m_iListBoxCellHeight = 16;
	m_iListBoxBaseLineOffset = fm.descent();
}

/**
 * ============ saveListBoxOptions =============
 */
void KviOptions::saveListBoxOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.listbox.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("ListBoxOptions");
	// Background Image
	cfg.writeEntry("BackgroundImage",     m_szListBoxBack.ptr());
	cfg.writeEntry("Font",                m_fntListBox);
	cfg.writeEntry("ShowImages",          m_bListBoxShowImages);
	cfg.writeEntry("BackgroundColor",     m_clrListBoxBack);
	cfg.writeEntry("OwnerColor",          m_clrListBoxOwner);
	cfg.writeEntry("OpColor",             m_clrListBoxOp);
	cfg.writeEntry("HalfOpColor",         m_clrListBoxHalfOp);
	cfg.writeEntry("VoiceColor",          m_clrListBoxVoice);
	cfg.writeEntry("UserOpColor",         m_clrListBoxUserOp);
	cfg.writeEntry("NormalColor",         m_clrListBoxNormal);
	cfg.writeEntry("SelectionBackground", m_clrListBoxSeleBack);
	cfg.writeEntry("SelectionForeground", m_clrListBoxSeleFore);
}

/**
 * ============ loadViewOptions =============
 */
void KviOptions::loadViewOptions()
{
	KviStr szTemp;
	g_pApp->getReadOnlyConfigPath(szTemp, "kvi.view.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("ViewOptions");

	// Background Image
	m_szViewBack = cfg.readEntry("BackgroundImage", "");
	loadOutputBackgroundImage();
	// Font
	QFont fnt(KVI_DEFAULT_FONT, KVI_DEFAULT_FONTSIZE);
	m_fntView = cfg.readFontEntry("Font", fnt);
	// Show images?
	m_bShowImages = cfg.readBoolEntry("ShowImages", true);
	m_bWrapMargin = cfg.readBoolEntry("WrapMargin", true);
	// Background color
	QColor wh(255, 255, 255);
	m_clrViewBack = cfg.readColorEntry("BackgroundColor", wh);
	// Selection color indexes
	m_cViewOutSeleBack   = cfg.readUCharEntry("SelectionBackground", KVI_BLACK);
	if( m_cViewOutSeleBack > 15 )
		m_cViewOutSeleBack = KVI_BLACK;
	m_cViewOutSeleFore   = cfg.readUCharEntry("SelectionForeground", KVI_WHITE);
	if( m_cViewOutSeleFore > 15 )
		m_cViewOutSeleFore = KVI_WHITE;
	m_cViewOutUrlFore    = cfg.readUCharEntry("UrlHighlightColor", KVI_BLUE);
	if( m_cViewOutUrlFore > 15 )
		m_cViewOutUrlFore = KVI_BLUE;
	m_bUrlHighlighting   = cfg.readBoolEntry("UrlHighlighting", true);
	m_bTimestamp         = cfg.readBoolEntry("Timestamp", true);
	m_iViewMaxBufferSize = cfg.readIntEntry("MaxLinesInBuffer", KVI_IRC_VIEW_MAX_LINES);
	// Typed output
	for( int j = 0; j < KVI_OUT_NUM_IMAGES; j++ ) {
		KviStr szEntry(KviStr::Format, "TypeBackground%d", j);
		m_cViewOutTypeBack[j] = cfg.readUCharEntry(szEntry.ptr(), KVI_TRANSPARENT);
		szEntry.sprintf("TypeForeground%d", j);
		m_cViewOutTypeFore[j] = cfg.readUCharEntry(szEntry.ptr(), KVI_BLACK);
		if( m_cViewOutTypeFore[j] > 15 )
			m_cViewOutTypeFore[j] = KVI_BLACK;
	}
}

void KviOptions::loadOutputBackgroundImage(const QPixmap *cached)
{
	loadBackgroundImage(m_szViewBack, m_pixViewBack, cached);
}

/**
 * ================ saveViewOptions ==============
 */
void KviOptions::saveViewOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.view.conf");

	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("ViewOptions");

	cfg.writeEntry("BackgroundImage",     m_szViewBack.ptr());
	cfg.writeEntry("Font",                m_fntView);
	cfg.writeEntry("ShowImages",          m_bShowImages);
	cfg.writeEntry("WrapMargin",          m_bWrapMargin);
	cfg.writeEntry("BackgroundColor",     m_clrViewBack);
	cfg.writeEntry("SelectionBackground", m_cViewOutSeleBack);
	cfg.writeEntry("SelectionForeground", m_cViewOutSeleFore);
	cfg.writeEntry("UrlHighlightColor",   m_cViewOutUrlFore);
	cfg.writeEntry("UrlHighlighting",     m_bUrlHighlighting);
	cfg.writeEntry("Timestamp",           m_bTimestamp);
	cfg.writeEntry("MaxLinesInBuffer",    m_iViewMaxBufferSize);
	for( int j = 0; j < KVI_OUT_NUM_IMAGES; j++ ) {
		KviStr szEntry(KviStr::Format, "TypeBackground%d", j);
		cfg.writeEntry(szEntry.ptr(), m_cViewOutTypeBack[j]);
		szEntry.sprintf("TypeForeground%d", j);
		cfg.writeEntry(szEntry.ptr(), m_cViewOutTypeFore[j]);
	}
}

/**
 * ================= loadInputOptions ===============
 */
void KviOptions::loadInputOptions()
{
	KviStr szTemp;
	g_pApp->getReadOnlyConfigPath(szTemp, "kvi.input.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("InputEditOptions");
	// Background image
	m_szInputBack            = cfg.readEntry("BackgroundImage", "");
	loadInputBackgroundImage();
	// Background color
	QColor wh(255, 255, 255);
	m_clrInputBack           = cfg.readColorEntry("BackgroundColor", wh);
	// Foreground color
	QColor bk(  0,   0,   0);
	m_clrInputFore           = cfg.readColorEntry("ForegroundColor",          bk);
	m_clrInputSeleBack       = cfg.readColorEntry("SelectionBackgroundColor", bk);
	m_clrInputSeleFore       = cfg.readColorEntry("SelectionForegroundColor", wh);
	QColor rd(255,   0,   0);
	m_clrInputCursor         = cfg.readColorEntry("CursorColor",              rd);
	QFont fnt(KVI_DEFAULT_FONT, KVI_DEFAULT_FONTSIZE);
	m_fntInput               = cfg.readFontEntry("Font", fnt);
	m_cPersonalCommandPrefix = cfg.readCharEntry("PersonalCommandPrefix", '/');

	m_szStringToAddAfterCompletedNick = cfg.readEntry("StringToAddAfterCompletedNick", ":&nbsp;");
	m_szStringToAddAfterCompletedNick.replaceAll("&nbsp;", " ");

	m_bApplyCompletionPostfixForFirstWordOnly = cfg.readBoolEntry("ApplyCompletionPostfixForFirstWordOnly", true);
	m_bCompletionReplaceWholeWord             = cfg.readBoolEntry("CompletionReplaceWholeWord", true);

	m_bInputHistoryCursorAtEnd                = cfg.readBoolEntry("InputHistoryCursorAtEnd", true);
	m_bUseStringSubstitution                  = cfg.readBoolEntry("UseStringSubstitution", false);

	calculateInputFontVariables();
}

void KviOptions::loadInputBackgroundImage(const QPixmap *cached)
{
	loadBackgroundImage(m_szInputBack, m_pixInputBack, cached);
}

inline QChar getSubstituteChar(unsigned short control_code)
{
	switch( control_code ) {
		case KVI_TEXT_COLOR:
			return QChar('K');
			break;
		case KVI_TEXT_BOLD:
			return QChar('B');
			break;
		case KVI_TEXT_RESET:
			return QChar('O');
			break;
		case KVI_TEXT_REVERSE:
			return QChar('R');
			break;
		case KVI_TEXT_UNDERLINE:
			return QChar('U');
			break;
		default:
			return QChar(control_code);
			break;
	}
}

void KviOptions::calculateInputFontVariables()
{
	QFontMetrics fm(m_fntInput);
	m_iInputFontLineSpacing    = fm.lineSpacing();
	m_iInputFontBaseLineOffset = fm.descent();
	m_iInputFontLineWidth      = fm.lineWidth();
	m_iInputFontHeight         = fm.height();
}

/**
 * ================ saveInputOptions ===============
 */
void KviOptions::saveInputOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.input.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("InputEditOptions");

	cfg.writeEntry("BackgroundImage",                        m_szInputBack.ptr());
	cfg.writeEntry("BackgroundColor",                        m_clrInputBack);
	cfg.writeEntry("ForegroundColor",                        m_clrInputFore);
	cfg.writeEntry("SelectionBackgroundColor",               m_clrInputSeleBack);
	cfg.writeEntry("SelectionForegroundColor",               m_clrInputSeleFore);
	cfg.writeEntry("CursorColor",                            m_clrInputCursor);
	cfg.writeEntry("Font",                                   m_fntInput);
	cfg.writeEntry("PersonalCommandPrefix",                  m_cPersonalCommandPrefix);
	// We cannot handle spaces...
	KviStr tmp = m_szStringToAddAfterCompletedNick.ptr();
	tmp.replaceAll(' ', "&nbsp;");
	cfg.writeEntry("StringToAddAfterCompletedNick",          tmp.ptr());
	cfg.writeEntry("ApplyCompletionPostfixForFirstWordOnly", m_bApplyCompletionPostfixForFirstWordOnly);
	cfg.writeEntry("CompletionReplaceWholeWord",             m_bCompletionReplaceWholeWord);
	cfg.writeEntry("InputHistoryCursorAtEnd",                m_bInputHistoryCursorAtEnd);
	cfg.writeEntry("UseStringSubstitution",                  m_bUseStringSubstitution);
}

/**
 * =============== loadSocketOptions =================
 */
void KviOptions::loadSocketOptions()
{
	KviStr szTemp;
	g_pApp->getReadOnlyConfigPath(szTemp, "kvi.socket.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("IrcSocket");

	m_iSocketTimeout                    = cfg.readIntEntry("ConnectTimeout", 60); // 60 seconds ok?
	m_bAutoReconnectOnDisconnect        = cfg.readBoolEntry("AutoReconnectOnDisconnect", false);
	m_bCloseWindowsOnDisconnect         = cfg.readBoolEntry("CloseWindowsOnDisconnect", true);
	m_bRejoinChannelsOnReconnect        = cfg.readBoolEntry("RejoinChannelsOnReconnect", false);
	m_bTryNextServerOnConnectFailed     = cfg.readBoolEntry("TryNextServerOnConnectFailed", false);
	if( m_iSocketTimeout < 0 || m_iSocketTimeout > 3600 )
		m_iSocketTimeout = 60;
	m_szLocalHostIp                     = cfg.readEntry("LocalIpAddress", "127.0.0.1");
	m_bForceSyncDns                     = cfg.readBoolEntry("ForceSyncDns", false);
	m_bBindIrcSocketToSpecificInterface = cfg.readBoolEntry("BindIrcSocketToSpecificInterface", false);
}

/**
 * ================ saveSocketOptions ================
 */
void KviOptions::saveSocketOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.socket.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("IrcSocket");

	cfg.writeEntry("ConnectTimeout",                   m_iSocketTimeout);
	cfg.writeEntry("AutoReconnectOnDisconnect",        m_bAutoReconnectOnDisconnect);
	cfg.writeEntry("TryNextServerOnConnectFailed",     m_bTryNextServerOnConnectFailed);
	cfg.writeEntry("CloseWindowsOnDisconnect",         m_bCloseWindowsOnDisconnect);
	cfg.writeEntry("RejoinChannelsOnReconnect",        m_bRejoinChannelsOnReconnect);
	cfg.writeEntry("LocalIpAddress",                   m_szLocalHostIp.ptr());
	cfg.writeEntry("ForceSyncDns",                     m_bForceSyncDns);
	cfg.writeEntry("BindIrcSocketToSpecificInterface", m_bBindIrcSocketToSpecificInterface);
}

/**
 * =============== loadServerOptions =================
 */
void KviOptions::loadServerOptions()
{
	KviStr szTemp;
	g_pApp->getReadOnlyConfigPath(szTemp, "kvi.server.conf");
	KviConfig cfg1(szTemp.ptr());
	m_pServerManager->load(&cfg1);
	g_pApp->getReadOnlyConfigPath(szTemp, "kvi.proxy.conf");
	KviConfig cfg2(szTemp.ptr());
	m_pProxyManager->load(&cfg2);
	cfg2.setGroup("MainProxyConfiguration");
	m_bUseTelnetProxy = cfg2.readBoolEntry("UseTelnetProxy", false);
	m_bUseProxy       = cfg2.readBoolEntry("UseProxy", false);
	m_bProxyProtoV5   = cfg2.readBoolEntry("UseProxyProtocolV5", true);
	m_bProxyProtoHttp = cfg2.readBoolEntry("UseProxyProtocolHttp", false);
}

/**
 * ================ saveServerOptions ================
 */
void KviOptions::saveServerOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.server.conf");
	KviConfig cfg1(szTemp.ptr());
	m_pServerManager->save(&cfg1);
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.proxy.conf");
	KviConfig cfg2(szTemp.ptr());
	m_pProxyManager->save(&cfg2);
	cfg2.setGroup("MainProxyConfiguration");
	cfg2.writeEntry("UseTelnetProxy",       m_bUseTelnetProxy);
	cfg2.writeEntry("UseProxy",             m_bUseProxy);
	cfg2.writeEntry("UseProxyProtocolV5",   m_bProxyProtoV5);
	cfg2.writeEntry("UseProxyProtocolHttp", m_bProxyProtoHttp);
}

/**
 * =============== loadUserOptions =================
 */
void KviOptions::loadUserOptions()
{
	KviStr szTemp;
	g_pApp->getReadOnlyConfigPath(szTemp, "kvi.user.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("UserOptions");

	m_szUserName      = cfg.readEntry("UserName",  KVI_DEFAULT_USERNAME);
	m_szRealName      = cfg.readEntry("RealName",  KVI_DEFAULT_REALNAME);
	m_szNickName1     = cfg.readEntry("NickName1", KVI_DEFAULT_NICKNAME1);
	m_szNickName2     = cfg.readEntry("NickName2", KVI_DEFAULT_NICKNAME2);
	m_bModeWOnConnect = cfg.readBoolEntry("ModeW", false);
	m_bModeIOnConnect = cfg.readBoolEntry("ModeI", false);
	m_bModeSOnConnect = cfg.readBoolEntry("ModeS", false);
	// Convert spaces in the username to underscores to prevent sending
	// wrong "USER ..." commands
	m_szUserName.replaceAll(' ', "_");
}

/**
 * ================ saveUserOptions ================
 */
void KviOptions::saveUserOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.user.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("UserOptions");

	cfg.writeEntry("UserName",  m_szUserName.ptr());
	cfg.writeEntry("RealName",  m_szRealName.ptr());
	cfg.writeEntry("NickName1", m_szNickName1.ptr());
	cfg.writeEntry("NickName2", m_szNickName2.ptr());
	cfg.writeEntry("ModeW",     m_bModeWOnConnect);
	cfg.writeEntry("ModeI",     m_bModeIOnConnect);
	cfg.writeEntry("ModeS",     m_bModeSOnConnect);
}

void KviOptions::loadIrcOptions()
{
	KviStr szTemp;
	g_pApp->getReadOnlyConfigPath(szTemp, "kvi.irc.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("IrcOptions");

	m_bShowPingPong                          = cfg.readBoolEntry("ShowPingPong", true);
	m_bJoinMsgsToConsole                     = cfg.readBoolEntry("JoinMsgsToConsole", false);
	m_bPartMsgsToConsole                     = cfg.readBoolEntry("PartMsgsToConsole", false);
	m_bDesyncMsgsToConsole                   = cfg.readBoolEntry("DesyncMsgsToConsole", true);
	m_bUnhandledServerErrorsToConsole        = cfg.readBoolEntry("UnhandledServerErrorsToConsole", false);
	m_bUnhandledNumericsToConsole            = cfg.readBoolEntry("UnhandledNumericsToConsole", true);
	m_bUnhandledLiteralsToConsole            = cfg.readBoolEntry("UnhandledLiteralsToConsole", true);
	m_bNicknameErrorsToConsole               = cfg.readBoolEntry("NicknameErrorsToConsole", true);
	m_bServerInfoToConsole                   = cfg.readBoolEntry("ServerInfoToConsole", true);
	m_bSkipMotd                              = cfg.readBoolEntry("SkipMotd", false);
	m_bNamesReplyToConsole                   = cfg.readBoolEntry("NamesReplyToConsole", true);
	m_bShowOwnParts                          = cfg.readBoolEntry("ShowOwnParts", true);
	m_bShowOwnNickChanges                    = cfg.readBoolEntry("ShowOwnNickChanges", true);
	m_bKickMsgsToConsole                     = cfg.readBoolEntry("KickMsgsToConsole", false);
	m_bQuitMsgsToConsole                     = cfg.readBoolEntry("QuitMsgsToConsole", false);
	m_bNickMsgsToConsole                     = cfg.readBoolEntry("NickMsgsToConsole", false);
	m_bTopicMsgsToConsole                    = cfg.readBoolEntry("TopicMsgsToConsole", false);
	m_bSplitWarningsToConsole                = cfg.readBoolEntry("SplitWarningsToConsole", true);
	m_bDnsResultsToConsole                   = cfg.readBoolEntry("DnsResultsToConsole", false);
	m_bServerNoticesToConsole                = cfg.readBoolEntry("ServerNoticesToConsole", true);
	m_bUserModeChangesToConsole              = cfg.readBoolEntry("UserModeChangesToConsole", true);
	m_bWallopsToConsole                      = cfg.readBoolEntry("WallopsToConsole", true);
	m_bChannelModeChangesToConsole           = cfg.readBoolEntry("ChannelModeChangesToConsole", false);
	m_bRequestBanExceptionListOnJoin         = cfg.readBoolEntry("RequestBanExceptionListOnJoin", true);
	m_bShowBanAndExceptionList               = cfg.readBoolEntry("ShowBanAndExceptionList", false);
	m_bBanAndExceptionListToConsole          = cfg.readBoolEntry("BanAndExceptionListToConsole", false);
	m_bRequestWhoAfterEndOfNames             = cfg.readBoolEntry("RequestWhoAfterEndOfNames", false);
	m_bWhoMsgsToConsole                      = cfg.readBoolEntry("WhoMsgsToConsole", true);
	m_bShowInternalWhoMessages               = cfg.readBoolEntry("ShowInternalWhoMessages", false);
	m_bCreateQueryOnPrivmsg                  = cfg.readBoolEntry("CreateQueryOnPrivmsg", true);
	m_bCreateQueryOnNotice                   = cfg.readBoolEntry("CreateQueryOnNotice", false);
	m_bOffChannelNoticesToConsole            = cfg.readBoolEntry("OffChannelNoticesToConsole", true);
	m_bCreateIconifiedQuery                  = cfg.readBoolEntry("CreateIconifiedQuery", false);
	m_bCtcpRequestsToConsole                 = cfg.readBoolEntry("CtcpRequestsToConsole", true);
	m_bChannelCtcpRequestsToConsole          = cfg.readBoolEntry("ChannelCtcpRequestsToConsole", false);
	m_bCtcpRepliesToConsole                  = cfg.readBoolEntry("CtcpRepliesToConsole", true);

	m_bListenToCtcpMultimedia                = cfg.readBoolEntry("ListenToCtcpMultimedia", true);
	m_bPlayDefaultFileWhenFileNotFound       = cfg.readBoolEntry("PlayDefaultFileWhenFileNotFound", false);
	m_bSendMultimediaRequestWhenFileNotFound = cfg.readBoolEntry("SendMultimediaRequestWhenFileNotFound", false);
	m_szCtcpMultimediaDefaultFileName        = cfg.readEntry("CtcpMultimediaDefaultFileName", "");
	m_bListenToMultimediaFileRequests        = cfg.readBoolEntry("ListenToMultimediaFileRequests", false);

	m_bWhoisRepliesToConsole                 = cfg.readBoolEntry("WhoisRepliesToConsole", true);
	m_bNotifyListChangesToConsole            = cfg.readBoolEntry("NotifyListChangesToConsole", false);
	m_bProcessOutputToConsole                = cfg.readBoolEntry("ProcessOutputToConsole", false);

	m_bReplyErrmsgOnUnknownCtcp              = cfg.readBoolEntry("ReplyErrmsgOnUnknownCtcp", false);
	m_bIgnoreCtcpPingRequests                = cfg.readBoolEntry("IgnoreCtcpPingRequests", false);
	m_bIgnoreCtcpVersionRequests             = cfg.readBoolEntry("IgnoreCtcpVersionRequests", false);
	m_bIgnoreCtcpTimeRequests                = cfg.readBoolEntry("IgnoreCtcpTimeRequests", false);
	m_bIgnoreCtcpUserinfoRequests            = cfg.readBoolEntry("IgnoreCtcpUserinfoRequests", false);
	m_bIgnoreCtcpClientinfoRequests          = cfg.readBoolEntry("IgnoreCtcpClientinfoRequests", false);
	m_bIgnoreCtcpSourceRequests              = cfg.readBoolEntry("IgnoreCtcpSourceRequests", false);
	m_bIgnoreCtcpFingerRequests              = cfg.readBoolEntry("IgnoreCtcpFingerRequests", false);

	m_szCtcpVersionAppendix                  = cfg.readEntry("CtcpVersionAppendix", "");
	m_szCtcpUserinfoAppendix                 = cfg.readEntry("CtcpUserinfoAppendix", "");
	m_szCtcpClientinfoAppendix               = cfg.readEntry("CtcpClientinfoAppendix", "");
	m_szCtcpSourceAppendix                   = cfg.readEntry("CtcpSourceAppendix", "");
	m_szCtcpFingerReply                      = cfg.readEntry("CtcpFingerReply", "I have not filled in this field yet...");

	m_bAppendRealUsernameToUserinfoReply     = cfg.readBoolEntry("AppendRealUsernameToUserinfoReply", true);
	m_bAppendUnameToClientinfoReply          = cfg.readBoolEntry("AppendUnameToClientinfoReply", false);

	m_bUseAntiCtcpFlood                      = cfg.readBoolEntry("UseAntiCtcpFlood", true);
	m_iMaxCtcpCount                          = cfg.readIntEntry("MaxCtcpCount", 6);
	m_iMaxCtcpTime                           = cfg.readIntEntry("MaxCtcpTime", 10);
	m_bCtcpFloodWarningsToConsole            = cfg.readBoolEntry("CtcpFloodWarningsToConsole", true);

	m_bUseSmallAntispamOnPrivmsg             = cfg.readBoolEntry("UseSmallAntispamOnPrivmsg", false);
	m_bUseAntiQueryFlood                     = cfg.readBoolEntry("UseAntiQueryFlood", true);
	m_iMaxTotalQueries                       = cfg.readIntEntry("MaxTotalQueries", 12);
	m_iMaxQueryCount                         = cfg.readIntEntry("MaxQueryCount", 3);
	m_iMaxQueryTime                          = cfg.readIntEntry("MaxQueryTime", 10);
	m_bQueryFloodWarningsToConsole           = cfg.readBoolEntry("QueryFloodWarningsToConsole", true);
	m_bBeepOnNewQuery                        = cfg.readBoolEntry("BeepOnNewQuery", false);
	m_bEnableBeeping                         = cfg.readBoolEntry("EnableBeeping", true);
	m_bAddUsersForBeepControl                = cfg.readBoolEntry("AddUsersForBeepControl", true);
	m_bDisableUpdateUserList                 = cfg.readBoolEntry("DisableUpdateUserList", false);
	m_iUserListUpdateTimeoutInSecs           = cfg.readIntEntry("UserListUpdateTimeoutInSecs", 20);
	m_bBrutalQuit                            = cfg.readBoolEntry("BrutalQuit", false);
	m_bBeVerbose                             = cfg.readBoolEntry("BeVerbose", false);
	m_bPedanticParser                        = cfg.readBoolEntry("PedanticParser", false);
	m_bAvoidParserWarnings                   = cfg.readBoolEntry("AvoidParserWarnings", false);
	m_bAutoDisableBrokenEvents               = cfg.readBoolEntry("AutoDisableBrokenEvents", true);
	m_bSendUnknownCommandsToServer           = cfg.readBoolEntry("SendUnknownCommandsToServer", true);

	m_bPrivmsgFormatNickIsALink              = cfg.readBoolEntry("PrivmsgFormatNickIsALink", true);
	m_szPrivmsgFormatNickLinkCommand         = cfg.readEntry("PrivmsgFormatNickLinkCommand", "whois $1");

	m_bUseExtendedPrivmsgView                = cfg.readBoolEntry("UseExtendedPrivmsgView", false);
	m_bEnableNickModePrefix                  = cfg.readBoolEntry("EnableNickModePrefix", false);
	m_bShowUserAndHostInPrivmsgs             = cfg.readBoolEntry("ShowUserAndHostInPrivmsgs", false);
	m_szExtPrivmsgPrefix                     = cfg.readEntry("ExtendedPrivmsgPrefix", "[");
	m_szExtPrivmsgPostfix                    = cfg.readEntry("ExtendedPrivmsgPostfix", "] :");

	m_bAutoRejoinOnKick                      = cfg.readBoolEntry("AutoRejoinOnKick", false);
	m_bDumpChannelOnKick                     = cfg.readBoolEntry("DumpChannelOnKick", false);

	m_bEnableNotifyList                      = cfg.readBoolEntry("EnableNotifyList", true);
	m_bEnableDalNetStyleNotifyMethod         = cfg.readBoolEntry("EnableDalNetStyleNotifyMethod", false);
	m_iNotifyListCheckTimeoutInSecs          = cfg.readIntEntry("NotifyListCheckTimeout", 60);

	m_bEnableIgnoreOnPrivmsg                 = cfg.readBoolEntry("EnableIgnoreOnPrivmsg", true);
	m_bEnableIgnoreOnNotice                  = cfg.readBoolEntry("EnableIgnoreOnNotice", true);
	m_bVerboseIgnore                         = cfg.readBoolEntry("VerboseIgnore", false);

	m_bSingleMessageForChannelModes          = cfg.readBoolEntry("SingleMessageForChannelModes", false);
	m_bShowChannelSyncTime                   = cfg.readBoolEntry("ShowChannelSyncTime", true);

	m_szPartMessage                          = cfg.readEntry("PartMessage",    KVI_DEFAULT_PART_MESSAGE);
	m_szKickMessage                          = cfg.readEntry("KickMessage",    KVI_DEFAULT_KICK_MESSAGE);
	m_szQuitMessage                          = cfg.readEntry("QuitMessage",    KVI_DEFAULT_QUIT_MESSAGE);
	m_szCaptionMessage                       = cfg.readEntry("CaptionMessage", KVI_DEFAULT_CAPTION_MESSAGE);

	m_bWordHighlighting                      = cfg.readBoolEntry("WordHighlighting", false);
	m_szHighlightWords                       = cfg.readEntry("HighlightWords", "");
	fillHighlightWordsList();

	m_bDoNotShowThePopupOnTextSelected       = cfg.readBoolEntry("DoNotShowThePopupOnTextSelected", true);

	m_bInviteMessagesToConsole               = cfg.readBoolEntry("InviteMessagesToConsole", false);
	m_bAutoJoinChanOnInvite                  = cfg.readBoolEntry("AutoJoinChanOnInvite", true);
	m_bShowExtendedServerInfo                = cfg.readBoolEntry("ShowExtendedServerInfo", false);

	m_szBrowserCommandline                   = cfg.readEntry("BrowserCommandline", "kfmclient exec %");
	m_iOnIdleAfter                           = cfg.readIntEntry("OnIdleAfter", 600);
	m_bCancelIdleOnAnyInput                  = cfg.readBoolEntry("CancelIdleOnAnyInput", false);
}

void KviOptions::fillHighlightWordsList()
{
	while( m_pHighlightWords->first() )
		m_pHighlightWords->removeFirst();
	m_szHighlightWords.stripWhiteSpace();
	KviStr tmp = m_szHighlightWords;
	while( tmp.hasData() ) {
		KviStr word;
		tmp.getToken(word, ' ');
		if( word.hasData() )
			m_pHighlightWords->append(new KviStr(word));
		tmp.stripWhiteSpace();
	}
	if( m_pHighlightWords->isEmpty() )
		m_bWordHighlighting = false;
}

void KviOptions::saveIrcOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.irc.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("IrcOptions");

	cfg.writeEntry("ShowPingPong",                          m_bShowPingPong);
	cfg.writeEntry("JoinMsgsToConsole",                     m_bJoinMsgsToConsole);
	cfg.writeEntry("PartMsgsToConsole",                     m_bPartMsgsToConsole);
	cfg.writeEntry("DesyncMsgsToConsole",                   m_bDesyncMsgsToConsole);
	cfg.writeEntry("UnhandledServerErrorsToConsole",        m_bUnhandledServerErrorsToConsole);
	cfg.writeEntry("UnhandledNumericsToConsole",            m_bUnhandledNumericsToConsole);
	cfg.writeEntry("UnhandledLiteralsToConsole",            m_bUnhandledLiteralsToConsole);
	cfg.writeEntry("NicknameErrorsToConsole",               m_bNicknameErrorsToConsole);
	cfg.writeEntry("ServerInfoToConsole",                   m_bServerInfoToConsole);
	cfg.writeEntry("SkipMotd",                              m_bSkipMotd);
	cfg.writeEntry("NamesReplyToConsole",                   m_bNamesReplyToConsole);
	cfg.writeEntry("ShowOwnParts",                          m_bShowOwnParts);
	cfg.writeEntry("ShowOwnNickChanges",                    m_bShowOwnNickChanges);
	cfg.writeEntry("KickMsgsToConsole",                     m_bKickMsgsToConsole);
	cfg.writeEntry("QuitMsgsToConsole",                     m_bQuitMsgsToConsole);
	cfg.writeEntry("NickMsgsToConsole",                     m_bNickMsgsToConsole);
	cfg.writeEntry("TopicMsgsToConsole",                    m_bTopicMsgsToConsole);
	cfg.writeEntry("WallopsToConsole",                      m_bWallopsToConsole);
	cfg.writeEntry("ServerNoticesToConsole",                m_bServerNoticesToConsole);
	cfg.writeEntry("SplitWarningsToConsole",                m_bSplitWarningsToConsole);
	cfg.writeEntry("DnsResultsToConsole",                   m_bDnsResultsToConsole);
	cfg.writeEntry("UserModeChangesToConsole",              m_bUserModeChangesToConsole);
	cfg.writeEntry("ChannelModeChangesToConsole",           m_bChannelModeChangesToConsole);
	cfg.writeEntry("RequestBanExceptionListOnJoin",         m_bRequestBanExceptionListOnJoin);
	cfg.writeEntry("ShowBanAndExceptionList",               m_bShowBanAndExceptionList);
	cfg.writeEntry("BanAndExceptionListToConsole",          m_bBanAndExceptionListToConsole);
	cfg.writeEntry("RequestWhoAfterEndOfNames",             m_bRequestWhoAfterEndOfNames);
	cfg.writeEntry("WhoMsgsToConsole",                      m_bWhoMsgsToConsole);
	cfg.writeEntry("NotifyListChangesToConsole",            m_bNotifyListChangesToConsole);
	cfg.writeEntry("ProcessOutputToConsole",                m_bProcessOutputToConsole);
	cfg.writeEntry("ShowInternalWhoMessages",               m_bShowInternalWhoMessages);
	cfg.writeEntry("CreateQueryOnPrivmsg",                  m_bCreateQueryOnPrivmsg);
	cfg.writeEntry("CreateQueryOnNotice",                   m_bCreateQueryOnNotice);
	cfg.writeEntry("CreateIconifiedQuery",                  m_bCreateIconifiedQuery);
	cfg.writeEntry("OffChannelNoticesToConsole",            m_bOffChannelNoticesToConsole);
	cfg.writeEntry("CtcpRequestsToConsole",                 m_bCtcpRequestsToConsole);
	cfg.writeEntry("ChannelCtcpRequestsToConsole",          m_bChannelCtcpRequestsToConsole);
	cfg.writeEntry("CtcpRepliesToConsole",                  m_bCtcpRepliesToConsole);
	cfg.writeEntry("ReplyErrmsgOnUnknownCtcp",              m_bReplyErrmsgOnUnknownCtcp);
	cfg.writeEntry("MaxCtcpCount",                          m_iMaxCtcpCount);
	cfg.writeEntry("MaxCtcpTime",                           m_iMaxCtcpTime);
	cfg.writeEntry("CtcpFloodWarningsToConsole",            m_bCtcpFloodWarningsToConsole);
	cfg.writeEntry("WhoisRepliesToConsole",                 m_bWhoisRepliesToConsole);
	cfg.writeEntry("UseAntiCtcpFlood",                      m_bUseAntiCtcpFlood);
	cfg.writeEntry("ListenToCtcpMultimedia",                m_bListenToCtcpMultimedia);
	cfg.writeEntry("PlayDefaultFileWhenFileNotFound",       m_bPlayDefaultFileWhenFileNotFound);
	cfg.writeEntry("CtcpMultimediaDefaultFileName",         m_szCtcpMultimediaDefaultFileName.ptr());
	cfg.writeEntry("SendMultimediaRequestWhenFileNotFound", m_bSendMultimediaRequestWhenFileNotFound);
	cfg.writeEntry("ListenToMultimediaFileRequests",        m_bListenToMultimediaFileRequests);

	cfg.writeEntry("IgnoreCtcpPingRequests",                m_bIgnoreCtcpPingRequests);
	cfg.writeEntry("IgnoreCtcpVersionRequests",             m_bIgnoreCtcpVersionRequests);
	cfg.writeEntry("IgnoreCtcpTimeRequests",                m_bIgnoreCtcpTimeRequests);
	cfg.writeEntry("IgnoreCtcpUserinfoRequests",            m_bIgnoreCtcpUserinfoRequests);
	cfg.writeEntry("IgnoreCtcpClientinfoRequests",          m_bIgnoreCtcpClientinfoRequests);
	cfg.writeEntry("IgnoreCtcpFingerRequests",              m_bIgnoreCtcpFingerRequests);
	cfg.writeEntry("IgnoreCtcpSourceRequests",              m_bIgnoreCtcpSourceRequests);
	cfg.writeEntry("AppendRealUsernameToUserinfoReply",     m_bAppendRealUsernameToUserinfoReply);
	cfg.writeEntry("AppendUnameToClientinfoReply",          m_bAppendUnameToClientinfoReply);

	cfg.writeEntry("CtcpVersionAppendix",                   m_szCtcpVersionAppendix.ptr());
	cfg.writeEntry("CtcpUserinfoAppendix",                  m_szCtcpUserinfoAppendix.ptr());
	cfg.writeEntry("CtcpClientinfoAppendix",                m_szCtcpClientinfoAppendix.ptr());
	cfg.writeEntry("CtcpSourceAppendix",                    m_szCtcpSourceAppendix.ptr());
	cfg.writeEntry("CtcpFingerReply",                       m_szCtcpFingerReply.ptr());

	cfg.writeEntry("UseSmallAntispamOnPrivmsg",             m_bUseSmallAntispamOnPrivmsg);
	cfg.writeEntry("UseAntiQueryFlood",                     m_bUseAntiQueryFlood);
	cfg.writeEntry("MaxTotalQueries",                       m_iMaxTotalQueries);
	cfg.writeEntry("MaxQueryCount",                         m_iMaxQueryCount);
	cfg.writeEntry("MaxQueryTime",                          m_iMaxQueryTime);
	cfg.writeEntry("QueryFloodWarningsToConsole",           m_bQueryFloodWarningsToConsole);
	cfg.writeEntry("BeepOnNewQuery",                        m_bBeepOnNewQuery);
	cfg.writeEntry("EnableBeeping",                         m_bEnableBeeping);
	cfg.writeEntry("AddUsersForBeepControl",                m_bAddUsersForBeepControl);
	cfg.writeEntry("DisableUpdateUserList",                 m_bDisableUpdateUserList);
	cfg.writeEntry("UserListUpdateTimeoutInSecs",           m_iUserListUpdateTimeoutInSecs);
	cfg.writeEntry("BrutalQuit",                            m_bBrutalQuit);
	cfg.writeEntry("BeVerbose",                             m_bBeVerbose);
	cfg.writeEntry("PedanticParser",                        m_bPedanticParser);
	cfg.writeEntry("AvoidParserWarnings",                   m_bAvoidParserWarnings);

	cfg.writeEntry("AutoDisableBrokenEvents",               m_bAutoDisableBrokenEvents);
	cfg.writeEntry("SendUnknownCommandsToServer",           m_bSendUnknownCommandsToServer);

	cfg.writeEntry("PrivmsgFormatNickIsALink",              m_bPrivmsgFormatNickIsALink);
	cfg.writeEntry("PrivmsgFormatNickLinkCommand",          m_szPrivmsgFormatNickLinkCommand.ptr());
	cfg.writeEntry("UseExtendedPrivmsgView",                m_bUseExtendedPrivmsgView);
	cfg.writeEntry("EnableNickModePrefix",                  m_bEnableNickModePrefix);
	cfg.writeEntry("ShowUserAndHostInPrivmsgs",             m_bShowUserAndHostInPrivmsgs);
	cfg.writeEntry("ExtendedPrivmsgPrefix",                 m_szExtPrivmsgPrefix.ptr());
	cfg.writeEntry("ExtendedPrivmsgPostfix",                m_szExtPrivmsgPostfix.ptr());

	cfg.writeEntry("SingleMessageForChannelModes",          m_bSingleMessageForChannelModes);
	cfg.writeEntry("ShowChannelSyncTime",                   m_bShowChannelSyncTime);

	cfg.writeEntry("AutoRejoinOnKick",                      m_bAutoRejoinOnKick);
	cfg.writeEntry("DumpChannelOnKick",                     m_bDumpChannelOnKick);

	cfg.writeEntry("PartMessage",                           m_szPartMessage.ptr());
	cfg.writeEntry("KickMessage",                           m_szKickMessage.ptr());
	cfg.writeEntry("QuitMessage",                           m_szQuitMessage.ptr());
	cfg.writeEntry("CaptionMessage",                        m_szCaptionMessage.ptr());

	cfg.writeEntry("EnableNotifyList",                      m_bEnableNotifyList);
	cfg.writeEntry("EnableDalNetStyleNotifyMethod",         m_bEnableDalNetStyleNotifyMethod);
	cfg.writeEntry("NotifyListCheckTimeout",                m_iNotifyListCheckTimeoutInSecs);

	cfg.writeEntry("EnableIgnoreOnPrivmsg",                 m_bEnableIgnoreOnPrivmsg);
	cfg.writeEntry("EnableIgnoreOnNotice",                  m_bEnableIgnoreOnNotice);
	cfg.writeEntry("VerboseIgnore",                         m_bVerboseIgnore);

	cfg.writeEntry("DoNotShowThePopupOnTextSelected",       m_bDoNotShowThePopupOnTextSelected);

	cfg.writeEntry("HighlightWords",                        m_szHighlightWords.ptr());
	cfg.writeEntry("WordHighlighting",                      m_bWordHighlighting);
	cfg.writeEntry("InviteMessagesToConsole",               m_bInviteMessagesToConsole);
	cfg.writeEntry("AutoJoinChanOnInvite",                  m_bAutoJoinChanOnInvite);
	cfg.writeEntry("ShowExtendedServerInfo",                m_bShowExtendedServerInfo);

	cfg.writeEntry("BrowserCommandline",                    m_szBrowserCommandline.ptr());

	cfg.writeEntry("OnIdleAfter",                           m_iOnIdleAfter);
	cfg.writeEntry("CancelIdleOnAnyInput",                  m_bCancelIdleOnAnyInput);
}

void KviOptions::loadLoggingOptions()
{
	KviStr szTemp;
	g_pApp->getReadOnlyConfigPath(szTemp, "kvi.log.conf");
	KviConfig cfg(szTemp.ptr());

	cfg.setGroup("LoggingOptions");

	m_bAutoLogConsole            = cfg.readBoolEntry("AutoLogConsole", false);
	m_bAutoLogChannels           = cfg.readBoolEntry("AutoLogChannels", false);
	m_bAutoLogQueries            = cfg.readBoolEntry("AutoLogQueries", false);
	m_bAutoLogChats              = cfg.readBoolEntry("AutoLogChats", false);
	m_bAutoLogSends              = cfg.readBoolEntry("AutoLogSends", false);
	m_bAppendServerNameToLogName = cfg.readBoolEntry("AppendServerNameToLogName", true);
	m_bAppendDateToLogName       = cfg.readBoolEntry("AppendDateToLogName", true);
	for( int i = 0; i < KVI_OUT_NUM_IMAGES; i++ ) {
		KviStr tmp(KviStr::Format, "LogMsgType%d", i);
		m_bLogMsgType[i] = cfg.readBoolEntry(tmp.ptr(), true);
	}
}

void KviOptions::saveLoggingOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.log.conf");
	KviConfig cfg(szTemp.ptr());

	cfg.setGroup("LoggingOptions");

	cfg.writeEntry("AutoLogConsole",            m_bAutoLogConsole);
	cfg.writeEntry("AutoLogChannels",           m_bAutoLogChannels);
	cfg.writeEntry("AutoLogQueries",            m_bAutoLogQueries);
	cfg.writeEntry("AutoLogChats",              m_bAutoLogChats);
	cfg.writeEntry("AutoLogSends",              m_bAutoLogSends);
	cfg.writeEntry("AppendServerNameToLogName", m_bAppendServerNameToLogName);
	cfg.writeEntry("AppendDateToLogName",       m_bAppendDateToLogName);
	for( int i = 0; i < KVI_OUT_NUM_IMAGES; i++ ) {
		KviStr tmp(KviStr::Format, "LogMsgType%d", i);
		cfg.writeEntry(tmp.ptr(), m_bLogMsgType[i]);
	}
}

void KviOptions::loadDccOptions()
{
	KviStr szTemp;
	g_pApp->getReadOnlyConfigPath(szTemp, "kvi.dcc.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("DCCOptions");

	m_bCreateIconifiedDccChat            = cfg.readBoolEntry("CreateIconifiedDccChat", false);
	m_bCreateIconifiedDccSend            = cfg.readBoolEntry("CreateIconifiedDccSend", false);
	m_bCreateIconifiedDccVoice           = cfg.readBoolEntry("CreateIconifiedDccVoice", false);
	m_bIgnoreDccRequests                 = cfg.readBoolEntry("IgnoreDccRequests", false);
	m_bAutoAcceptDccChat                 = cfg.readBoolEntry("AutoAcceptDccChat", false);
	m_bEnableResumeOnAutoAccept          = cfg.readBoolEntry("EnableResumeOnAutoAccept", false);
	m_bAutoAcceptDccVoice                = cfg.readBoolEntry("AutoAcceptDccVoice", false);
	m_bAutoAcceptDccSend                 = cfg.readBoolEntry("AutoAcceptDccSend", false);
	m_bAutoCloseDccSendOnSuccess         = cfg.readBoolEntry("AutoCloseDccSendOnSuccess", true);
	m_bReplaceSpacesInDccSendFileNames   = cfg.readBoolEntry("ReplaceSpacesInDccSendFileNames", true);
	m_bUseFastDccSend                    = cfg.readBoolEntry("UseFastDccSend", true);
	m_iDccSendBlockSize                  = cfg.readIntEntry("DccSendBlockSize", 512);
	if( m_iDccSendBlockSize < 0 )
		m_iDccSendBlockSize = 512;
	m_bIgnorePortNumberInResumeRequests  = cfg.readBoolEntry("IgnorePortNumberInResumeRequests", false);
	m_bNotifyDccSendCompletionInConsole  = cfg.readBoolEntry("NotifyDccSendCompletionInConsole", true);
	m_bFullDuplexSoundDevice             = cfg.readBoolEntry("FullDuplexSoundDevice", false);
	m_b090CompatibleVoiceCompression     = cfg.readBoolEntry("090CompatibleVoiceCompression", false);
	m_bNotifyAllDccRequestsInConsole     = cfg.readBoolEntry("NotifyAllDccRequestsInConsole", true);
	m_szLocalIncomingDirectory           = cfg.readEntry("LocalIncomingDirectory", "incoming");
	m_szSoundDevice                      = cfg.readEntry("SoundDevice", "/dev/dsp");
	m_szDccLocalIpAddress                = cfg.readEntry("DccLocalIpAddress", "127.0.0.1");
	m_bUseUserDefinedIpForDccRequests    = cfg.readBoolEntry("UseUserDefinedIpForDccRequests", false);
	m_bDccGetSendZeroAck                 = cfg.readBoolEntry("DccGetSendZeroAck", false);
	m_bDccListenOnPortsInRange           = cfg.readBoolEntry("DccListenOnPortsInRange", false);
	m_uMinDccListenPort                  = cfg.readUShortEntry("MinDccListenPort", 1025);
	m_uMaxDccListenPort                  = cfg.readUShortEntry("MaxDccListenPort", 65000);
}

void KviOptions::saveDccOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.dcc.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("DCCOptions");

	cfg.writeEntry("CreateIconifiedDccChat",           m_bCreateIconifiedDccChat);
	cfg.writeEntry("CreateIconifiedDccSend",           m_bCreateIconifiedDccSend);
	cfg.writeEntry("CreateIconifiedDccVoice",          m_bCreateIconifiedDccVoice);
	cfg.writeEntry("IgnoreDccRequests",                m_bIgnoreDccRequests);
	cfg.writeEntry("AutoAcceptDccChat",                m_bAutoAcceptDccChat);
	cfg.writeEntry("AutoAcceptDccVoice",               m_bAutoAcceptDccVoice);
	cfg.writeEntry("AutoAcceptDccSend",                m_bAutoAcceptDccSend);
	cfg.writeEntry("EnableResumeOnAutoAccept",         m_bEnableResumeOnAutoAccept);
	cfg.writeEntry("AutoCloseDccSendOnSuccess",        m_bAutoCloseDccSendOnSuccess);
	cfg.writeEntry("ReplaceSpacesInDccSendFileNames",  m_bReplaceSpacesInDccSendFileNames);
	cfg.writeEntry("UseFastDccSend",                   m_bUseFastDccSend);
	cfg.writeEntry("DccSendBlockSize",                 m_iDccSendBlockSize);
	cfg.writeEntry("IgnorePortNumberInResumeRequests", m_bIgnorePortNumberInResumeRequests);
	cfg.writeEntry("NotifyDccSendCompletionInConsole", m_bNotifyDccSendCompletionInConsole);
	cfg.writeEntry("FullDuplexSoundDevice",            m_bFullDuplexSoundDevice);
	cfg.writeEntry("090CompatibleVoiceCompression",    m_b090CompatibleVoiceCompression);
	cfg.writeEntry("NotifyAllDccRequestsInConsole",    m_bNotifyAllDccRequestsInConsole);
	cfg.writeEntry("LocalIncomingDirectory",           m_szLocalIncomingDirectory.ptr());
	cfg.writeEntry("SoundDevice",                      m_szSoundDevice.ptr());
	cfg.writeEntry("DccLocalIpAddress",                m_szDccLocalIpAddress.ptr());
	cfg.writeEntry("UseUserDefinedIpForDccRequests",   m_bUseUserDefinedIpForDccRequests);
	cfg.writeEntry("DccGetSendZeroAck",                m_bDccGetSendZeroAck);
	cfg.writeEntry("DccListenOnPortsInRange",          m_bDccListenOnPortsInRange);
	cfg.writeEntry("MinDccListenPort",                 m_uMinDccListenPort);
	cfg.writeEntry("MaxDccListenPort",                 m_uMaxDccListenPort);
}

void KviOptions::loadHelpOptions()
{
	KviStr szTemp;
	g_pApp->getReadOnlyConfigPath(szTemp, "kvi.help.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("HelpBrowserOptions");

	QColor wh(255, 255, 255); // Background color
	m_clrHelpBack     = cfg.readColorEntry("BackgroundColor", wh);
	QColor bk(  0,   0,   0); // Foreground color
	m_clrHelpFore     = cfg.readColorEntry("ForegroundColor", bk);
	QColor bl(  0,  80, 120);
	m_clrHelpTitle    = cfg.readColorEntry("TitleColor", bl);
	m_clrHelpSubtitle = cfg.readColorEntry("SubtitleColor", bl);
	QColor rd(200,   0,   0);
	m_clrHelpExample  = cfg.readColorEntry("ExampleColor", rd);
	QFont f1(KVI_DEFAULT_FONT, KVI_DEFAULT_FONTSIZE);
	m_fntHelpNormal   = cfg.readFontEntry("NormalFont", f1);
	QFont f2 = KviConfig::getFixedFont();
	m_fntHelpFixed    = cfg.readFontEntry("FixedFont", f2);
}

void KviOptions::saveHelpOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.help.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("HelpBrowserOptions");

	cfg.writeEntry("BackgroundColor", m_clrHelpBack);
	cfg.writeEntry("ForegroundColor", m_clrHelpFore);
	cfg.writeEntry("TitleColor",      m_clrHelpTitle);
	cfg.writeEntry("SubtitleColor",   m_clrHelpSubtitle);
	cfg.writeEntry("ExampleColor",    m_clrHelpExample);
	cfg.writeEntry("NormalFont",      m_fntHelpNormal);
	cfg.writeEntry("FixedFont",       m_fntHelpFixed);
}

/**
 * ================= loadMdiOptions =================
 */
void KviOptions::loadMdiOptions()
{
	KviStr szTemp;
	g_pApp->getReadOnlyConfigPath(szTemp, "kvi.mdi.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("MdiManagerOptions");
	QColor bl(  0,   0,   0);
	QColor gr( 80, 255,   0);
	QColor dk( 80,  80,  80);
	QColor lt(180, 180, 180);
	m_clrMdiCaptionActiveBack       = cfg.readColorEntry("CaptionActiveBack", bl);
	m_clrMdiCaptionActiveFore       = cfg.readColorEntry("CaptionActiveFore", gr);
	m_clrMdiCaptionInactiveBack     = cfg.readColorEntry("CaptionInactiveBack", dk);
	m_clrMdiCaptionInactiveFore     = cfg.readColorEntry("CaptionInactiveFore", lt);
	QFont fnt(KVI_DEFAULT_FONT, KVI_DEFAULT_FONTSIZE);
	m_fntMdiCaption                 = cfg.readFontEntry("CaptionFont", fnt);
	m_clrMdiBack                    = cfg.readColorEntry("ManagerBackgroundColor", bl);
	m_szMdiBack                     = cfg.readEntry("ManagerBackgroundImage", "");

	m_bWindowsRememberProperties    = cfg.readBoolEntry("RememberProprietes", true);
	m_bWindowsRememberDockState     = cfg.readBoolEntry("RememberDockState", true);
	m_bWindowsRememberMaximizeState = cfg.readBoolEntry("RememberMaximizeState", true);

	loadMdiBackgroundImage();
}

void KviOptions::loadMdiBackgroundImage(const QPixmap *cached)
{
	loadBackgroundImage(m_szMdiBack, m_pixMdiBack, cached);
}

/**
 * =================== loadBackgroundImage ==================
 */
void KviOptions::loadBackgroundImage(KviStr &pix_file, QPixmap *pix_ptr, const QPixmap *cached)
{
	KviStr szTemp;
	if( !pix_file.isEmpty() ) {
		// Check the path
		if( g_pApp->findImage(szTemp, pix_file.ptr()) ) {
			// The file is on disk.
			// Check if we have a cached copy
			if( cached )
				*pix_ptr = *cached;
			else
				pix_ptr->load(szTemp.ptr());
			if( !pix_ptr->isNull() ) {
				pix_file = szTemp;
				return;
			}
		}
	}
	pix_ptr->resize(0, 0); // Make a null pixmap
	pix_file = "";
}

/**
 * ================== saveMdiOptions ================
 */
void KviOptions::saveMdiOptions()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.mdi.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("MdiManagerOptions");

	cfg.writeEntry("CaptionActiveBack",      m_clrMdiCaptionActiveBack);
	cfg.writeEntry("CaptionActiveFore",      m_clrMdiCaptionActiveFore);
	cfg.writeEntry("CaptionInactiveBack",    m_clrMdiCaptionInactiveBack);
	cfg.writeEntry("CaptionInactiveFore",    m_clrMdiCaptionInactiveFore);
	cfg.writeEntry("CaptionFont",            m_fntMdiCaption);
	cfg.writeEntry("ManagerBackgroundColor", m_clrMdiBack);
	cfg.writeEntry("ManagerBackgroundImage", m_szMdiBack.ptr());
	cfg.writeEntry("RememberProperties",     m_bWindowsRememberProperties);
	cfg.writeEntry("RememberDockState",      m_bWindowsRememberDockState);
	cfg.writeEntry("RememberMaximizeState",  m_bWindowsRememberMaximizeState);
}

/**
 * ================ loadAliases ==================
 */
void KviOptions::loadAliases()
{
	KviStr szTemp;
	if( !g_pApp->getReadOnlyConfigPath(szTemp, "kvi.alias.conf") )
		return;
	KviStr error;
	if( !g_pAliasManager->load(szTemp.ptr(), error) ) {
		g_pApp->warningBox(
			_i18n_("Error while parsing the aliases database:\n"\
			"%s\nSome aliases may not work correctly."),
			error.ptr()
		);
	}
}

/**
 * ================ saveAliases =================
 */
void KviOptions::saveAliases()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.alias.conf");
	if( !g_pAliasManager->save(szTemp.ptr()) ) {
		g_pApp->warningBox(
			_i18n_("Unable to save aliases to %s\n"\
			"Please check your write permissions."),
			szTemp.ptr()
		);
	}
}

/**
 * ================ loadEvents ==================
 */
void KviOptions::loadEvents()
{
	KviStr szTemp;
	if( !g_pApp->getReadOnlyConfigPath(szTemp, "kvi.event.conf") )
		return;
	KviStr error;
	if( !g_pEventManager->load(szTemp.ptr(), error) ) {
		g_pApp->warningBox(
			_i18n_("Error while parsing the events database:\n"\
			"%s\nSome event handlers may not work correctly."),
			error.ptr()
		);
	}
}

/**
 * ================ saveEvents =================
 */
void KviOptions::saveEvents()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.event.conf");
	if( !g_pEventManager->save(szTemp.ptr()) ) {
		g_pApp->warningBox(
			_i18n_("Unable to save events to %s\n"\
			"Please check your write permissions."),
			szTemp.ptr()
		);
	}
}

/**
 * ================ loadRawEvents ==================
 */
void KviOptions::loadRawEvents()
{
	KviStr szTemp;
	if( !g_pApp->getReadOnlyConfigPath(szTemp, "kvi.rawevent.conf") )
		return;
	KviStr error;
	if( !g_pRawEventManager->load(szTemp.ptr(), error) ) {
		g_pApp->warningBox(
			_i18n_("Error while parsing the raw events database:\n"\
			"%s\nSome event handlers may not work correctly."),
			error.ptr()
		);
	}
}

/**
 * ================ saveRawEvents =================
 */
void KviOptions::saveRawEvents()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.rawevent.conf");
	if( !g_pRawEventManager->save(szTemp.ptr()) ) {
		g_pApp->warningBox(
			_i18n_("Unable to save raw events to %s\n"\
			"Please check your write permissions."),
			szTemp.ptr()
		);
	}
}

void KviOptions::loadRegUsers()
{
	KviStr szTemp;
	if( !g_pApp->getReadOnlyConfigPath(szTemp, "kvi.regusersdb.conf") )
		return;
	KviStr error;
	if( !m_pRegUsersDb->load(szTemp.ptr(), error) )
		g_pApp->warningBox(_i18n_("Error while parsing the registered users database: %s."), error.ptr());
}

void KviOptions::saveRegUsers()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.regusersdb.conf");
	if( !m_pRegUsersDb->save(szTemp.ptr()) ) {
		g_pApp->warningBox(
			_i18n_("Unable to save registered users to %s\n"\
			"Please check your write permissions."),
			szTemp.ptr()
		);
	}
}

#ifdef COMPILE_PLUGIN_SUPPORT
void KviOptions::loadPlugins()
{
	KviStr szTemp;
	if( !g_pApp->getReadOnlyConfigPath(szTemp, "kvi.plugin.conf") )
		return;
	KviStr error;
	if( !g_pPluginManager->load(szTemp.ptr(), error) )
		g_pApp->warningBox(_i18n_("Encountered errors while\nparsing the plugins database:\n%s."), error.ptr());
}

void KviOptions::savePlugins()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.plugin.conf");
	if( !g_pPluginManager->save(szTemp.ptr()) ) {
		g_pApp->warningBox(
			_i18n_("Unable to save plugins to %s\n"\
			"Please check your write permissions."),
			szTemp.ptr()
		);
	}
}
#endif

void KviOptions::loadPopups()
{
	KviStr szTemp;
	if( !g_pApp->getReadOnlyConfigPath(szTemp, "kvi.popup.conf") )
		return;
	KviStr error;
	if( !KviUserPopupMenu::loadAllPopups(szTemp.ptr(), error) ) {
		g_pApp->warningBox(
			_i18n_("Error while parsing the popups database:\n"\
			"%s\nSome popups handlers may not work correctly."),
			error.ptr()
		);
	}
}

void KviOptions::savePopups()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.popup.conf");
	if( !KviUserPopupMenu::saveAllPopups(szTemp.ptr()) ) {
		g_pApp->warningBox(
			_i18n_("Unable to save popups to %s\n"\
			"Please check your write permissions."),
			szTemp.ptr()
		);
	}
}

void KviOptions::saveToolBar()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.toolbar.conf");
	if( !KviUserToolBarTemplate::saveToolBarTemplate(szTemp.ptr()) ) {
		g_pApp->warningBox(
			_i18n_("Unable to save toolbar to %s\n"\
			"Please check your write permissions."),
			szTemp.ptr()
		);
	}
}

void KviOptions::loadToolBar()
{
	KviStr szTemp;
	if( !g_pApp->getReadOnlyConfigPath(szTemp, "kvi.toolbar.conf") )
		return;
	KviStr error;
	if( !KviUserToolBarTemplate::loadToolBarTemplate(szTemp.ptr(), error) ) {
		g_pApp->warningBox(
			_i18n_("Error while parsing the toolbar database:\n"\
			"%s\nSome toolbar handlers may not work correctly."),
			error.ptr()
		);
	}
}

void KviOptions::loadWinProperties()
{
	KviStr szTemp;
	if( !g_pApp->getReadOnlyConfigPath(szTemp, "kvi.winproperties.conf") )
		return;
	KviStr error;
	if( !m_pWinPropertiesList->load(szTemp.ptr(), error) ) {
		g_pApp->warningBox(
			_i18n_("Error while parsing the window properties database:\n"\
			"%s\nSome windows may loose their settings."),
			error.ptr()
		);
	}
}

void KviOptions::saveWinProperties()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.winproperties.conf");
	if( !m_pWinPropertiesList->save(szTemp.ptr()) ) {
		g_pApp->warningBox(
			_i18n_("Unable to save the window properties to %s\n"\
			"Please check your write permissions."),
			szTemp.ptr()
		);
	}
}

void KviOptions::loadMimeTypes()
{
	KviStr szTemp;
	if( !g_pApp->getReadOnlyConfigPath(szTemp, "kvi.mime.conf") )
		return;
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("MimeTypes");
	m_pMimeManager->load(&cfg);
}

void KviOptions::saveMimeTypes()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.mime.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("MimeTypes");
	m_pMimeManager->save(&cfg);
}

void KviOptions::saveStrSub()
{
	KviStr szTemp;
	g_pApp->getLocalKVIrcDirectory(szTemp, KviApp::Config, "kvi.strsub.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("Substitution");
	m_pStrSub->save(&cfg);
}

void KviOptions::loadStrSub()
{
	KviStr szTemp;
	if( !g_pApp->getReadOnlyConfigPath(szTemp, "kvi.strsub.conf") )
		return;
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("Substitution");
	m_pStrSub->load(&cfg);
}

