// =============================================================================
//
//      --- kvi_command.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviCommand"

#include "kvi_command.h"
#include "kvi_debug.h"
#include "kvi_error.h"
#include "kvi_malloc.h"
#include "kvi_options.h"
#include "kvi_script_object.h"
#include "kvi_userparser.h"
#include "kvi_variablecache.h"
#include "kvi_window.h"

/**
 * ===========================================================================
 * Class KviCommand
 *
 * Wrapper for the command string buffer.
 * Contains a cache for local variables, parameters, and error codes
 *
 */
KviCommand::KviCommand(const char *pBuf, KviWindow *pWnd)
{
	m_err            = KVI_ERROR_NoError;
	m_ptr            = pBuf;
	m_wnd            = pWnd;
	m_pLocalVarCache = new KviVariableCache();
	m_pParamList     = 0;
	m_switchList     = new QPtrList<KviStr>;
	m_switchList->setAutoDelete(true);
	m_pScopeObject   = 0;
	m_flags          = KVI_COMMAND_FLAG_RECOGNIZE_OBJECT_SCOPE_OPERATOR;
}

KviCommand::~KviCommand()
{
	if( m_pParamList ) {
		delete m_pParamList;
		m_pParamList = 0;
	}
	delete m_pLocalVarCache;
	m_pLocalVarCache = 0;
	clearSwitchList();
	delete m_switchList;
	m_switchList = 0;
}

bool KviCommand::hasSwitch(char letter)
{
	for( KviStr *s = m_switchList->first(); s; s = m_switchList->next() ) {
		if( tolower(*(s->ptr())) == tolower(letter) )
			return true;
	}
	return false;
}

bool KviCommand::getSwitchValue(char letter, KviStr &buffer)
{
	for(KviStr *s = m_switchList->first(); s; s = m_switchList->next() ) {
		if( tolower(*(s->ptr())) == tolower(letter) ) {
			if( s->len() < 2 )
				return false;
			buffer = s->ptr();
			buffer.cutLeft(2);
			return true;
		}
	}
	return false;
}

bool KviCommand::hasError()
{
	return (m_err != KVI_ERROR_NoError);
}

void KviCommand::getParamCount(KviStr &buffer)
{
	if( !hasParams() )
		buffer.append("0");
	else {
		KviStr num(KviStr::Format, "%d", m_pParamList->count());
		buffer.append(num);
	}
}

void KviCommand::setReturnValue(const char *retBuffer)
{
	m_retBuffer = retBuffer;
}

/**
 * $1, $1- $0- $0-4 $21-24 $12-
 */
void KviCommand::getSingleParam(int param, KviStr &buffer)
{
	if( !hasParams() ) return;

	if( m_pParamList->count() > ((unsigned int) param) ) {
		KviStr *par = m_pParamList->at(param);
		if( par )
			buffer.append(*par);
	}
}

void KviCommand::getParam(int fromParam, int toParam, KviStr &buffer)
{
	if( !hasParams() ) return;

	if( ((unsigned int) fromParam) >= m_pParamList->count() )
		return;

	KviStr *par = m_pParamList->at(fromParam);
	if( !par )
		return;

	buffer.append(*par);
	if( toParam < fromParam ) {
		// Up to the end
		for( par = m_pParamList->next(); par; par = m_pParamList->next() ) {
			buffer.append(' ');
			buffer.append(*par);
		}
	} else {
		if( ((unsigned int) toParam) >= m_pParamList->count() )
			toParam = m_pParamList->count();
		fromParam++;
		par = m_pParamList->next();
		while( par && (fromParam <= toParam) ) {
			buffer.append(' ');
			buffer.append(*par);
			par = m_pParamList->next();
			fromParam++;
		}
	}
}

void KviCommand::setParams(const KviStr &param0, QPtrList<KviStr> *pList)
{
	__range_invalid(hasParams());
	if( !pList )
		pList = new QPtrList<KviStr>;
	pList->setAutoDelete(true);
	m_pParamList = pList;
	m_pParamList->insert(0, new KviStr(param0));
}

void KviCommand::setParams(const KviStr &param0, const char *paramBuffer)
{
	__range_valid(!hasParams());
	m_pParamList = new QPtrList<KviStr>;
	m_pParamList->setAutoDelete(true);

	m_pParamList->append(new KviStr(param0));
	const char *aux;
	for( ;; ) {
		// Skip initial spaces
		while( (*paramBuffer == ' ') || (*paramBuffer == '\t') )
			paramBuffer++;
		if( !*paramBuffer ) return;

		aux = paramBuffer;
		while( *paramBuffer && (*paramBuffer != ' ') && (*paramBuffer != '\t') )
			paramBuffer++;
		m_pParamList->append(new KviStr(aux, paramBuffer - aux));
	}
}

bool KviCommand::setParamsFromCommand(const KviStr &param0, KviCommand *cmd, KviUserParser *u)
{
	__range_invalid(hasParams());

	m_pParamList = new QPtrList<KviStr>;
	m_pParamList->setAutoDelete(true);
	m_pParamList->append(new KviStr(param0));
	for( ;; ) {
		cmd->skipSpace();
		if( ((*cmd->m_ptr) == ';') || ((*cmd->m_ptr) == '\0') || ((*cmd->m_ptr) == '\n') )
			break;
		cmd->clearBuffer();
		if( !u->processCmdSingleToken(cmd) )
			return false;
		if( cmd->m_buffer.hasData() )
			m_pParamList->append(new KviStr(cmd->m_buffer));
		else
			return u->processCmdFinalPart(cmd);
	}

	return true;
}

bool KviCommand::hasThisId()
{
	return (*(m_szThisId.ptr()) != '\0');
}

/**
 * This method ALWAYS returns false
 */
bool KviCommand::setError(int iError, const char *szLocation, const char *szErrToken)
{
	m_szErrorLocation = szLocation;
	m_err             = iError;
	if( szErrToken ) {
		m_szErrorToken.setStr(szErrToken, 40);
		if( m_szErrorToken.len() == 40 )
			m_szErrorToken += "...";
	}
	return false;
}

void KviCommand::skipWhitespace()
{
	while( *m_ptr && isspace(*m_ptr) )
		m_ptr++;
	if( *m_ptr == '\\' ) {
		// Escape char
		m_ptr++;
		if( *m_ptr && isspace(*m_ptr) ) {
			// Escaped whitespace
			m_ptr++;
			skipWhitespace();
		} else m_ptr--;
	}
}

void KviCommand::skipSpace()
{
	while( (*m_ptr == ' ') || (*m_ptr == '\t') )
		m_ptr++;
	if( *m_ptr == '\\' ) {
		// Escape char
		m_ptr++;
		if( (*m_ptr == '\n') || (*m_ptr == ' ') || (*m_ptr == '\t') ) {
			// Escaped newline, space or tab
			m_ptr++;
			skipSpace();
		} else m_ptr--;
	}
}

/**
 * This method ALWAYS returns true
 */
bool KviCommand::warning(const char *format, ...)
{
	__range_valid(!g_pOptions->m_bPedanticParser);
	if( g_pOptions->m_bAvoidParserWarnings )
		return true;

	char txt_ptr[512]; // It should be enough for all output...
	va_list list;
	va_start(list, format);
	if( kvi_vsnprintf(txt_ptr, 512, format, list) < 0 ) {
		// Just in case...
		va_end(list);
		int len = 512;
		char *long_txt_ptr = 0;
		int result;
		do {
			len += 512;
			// First time long_txt_ptr == 0 so it is equivalent to malloc
			// At least that is what the man page says...
			long_txt_ptr = (char *) kvi_realloc((void *) long_txt_ptr, len);
			va_start(list, format);
			result = kvi_vsnprintf(long_txt_ptr, len, format, list);
			va_end(list);
		} while( result < 0 );
		m_wnd->output(KVI_OUT_INTERNAL, "[Parser]: %s", long_txt_ptr);
		kvi_free((void *) long_txt_ptr);
	} else {
		// Successful vsnprintf
		va_end(list);
		m_wnd->output(KVI_OUT_INTERNAL, "[Parser]: %s", txt_ptr);
	}
	return true;
}
